# 🧪 Clean URLs Testing Guide

Complete testing suite and verification checklist for the clean URLs implementation.

---

## 📋 Table of Contents

1. [Automated Testing](#automated-testing)
2. [Manual Verification Checklist](#manual-verification-checklist)
3. [Debugging Instructions](#debugging-instructions)
4. [Troubleshooting](#troubleshooting)

---

## 🤖 Automated Testing

### Running the Test Suite

1. **Via Browser:**
   ```
   https://your-domain.com/Multi-Tanent/tools/test_clean_urls_comprehensive.php
   ```

2. **Via Command Line:**
   ```bash
   php tools/test_clean_urls_comprehensive.php
   ```

The test suite will automatically check:
- ✅ mod_rewrite module availability
- ✅ .htaccess file presence
- ✅ Directory protection files
- ✅ URL structure and format
- ✅ Security configurations

---

## ✅ Manual Verification Checklist

### 1. Server Configuration

- [ ] **mod_rewrite is enabled**
  ```bash
  # Check Apache modules
  apache2ctl -M | grep rewrite
  # or
  php -m | grep apache
  ```

- [ ] **.htaccess files are present**
  - [ ] Root `.htaccess`
  - [ ] `includes/.htaccess`
  - [ ] `vendor/.htaccess`
  - [ ] `uploads/.htaccess`

- [ ] **AllowOverride is enabled**
  ```apache
  # In Apache virtual host config
  <Directory "/path/to/Multi-Tanent">
      AllowOverride All
      Require all granted
  </Directory>
  ```

### 2. Clean URLs - Remove .php Extensions

Test these URLs (should work WITHOUT .php extension):

- [ ] **Admin Panel:**
  - `https://your-domain.com/Multi-Tanent/admin/dashboard`
  - `https://your-domain.com/Multi-Tanent/admin/students/list`
  - `https://your-domain.com/Multi-Tanent/admin/login`

- [ ] **Super Admin Panel:**
  - `https://your-domain.com/Multi-Tanent/super_admin/dashboard`
  - `https://your-domain.com/Multi-Tanent/super_admin/schools/list`

- [ ] **Parent Panel:**
  - `https://your-domain.com/Multi-Tanent/parent/dashboard`

- [ ] **Student Panel:**
  - `https://your-domain.com/Multi-Tanent/student/dashboard`

### 3. Tenant Isolation URLs

Test these URLs (school code in path):

- [ ] **Student Registration:**
  - `https://your-domain.com/Multi-Tanent/kine/register`
  - Should rewrite to: `register_student.php?school=kine`

- [ ] **Parent Registration:**
  - `https://your-domain.com/Multi-Tanent/kine/parent/register`
  - Should rewrite to: `parent_register.php?school=kine`

- [ ] **Parent Login:**
  - `https://your-domain.com/Multi-Tanent/kine/parent/login`
  - Should rewrite to: `parent_login.php?school=kine`

- [ ] **Student Login:**
  - `https://your-domain.com/Multi-Tanent/kine/login`
  - Should rewrite to: `student_login.php?school=kine`

- [ ] **Tenant Admin Access:**
  - `https://your-domain.com/Multi-Tanent/school/kine/admin/dashboard`
  - Should rewrite to: `admin/dashboard.php?school=kine`

### 4. Security Tests

Verify these files/directories are **BLOCKED** (should return 403 Forbidden):

- [ ] **Config Files:**
  - `https://your-domain.com/Multi-Tanent/config.php` ❌ Should be blocked
  - `https://your-domain.com/Multi-Tanent/.htaccess` ❌ Should be blocked

- [ ] **Includes Directory:**
  - `https://your-domain.com/Multi-Tanent/includes/functions.php` ❌ Should be blocked
  - `https://your-domain.com/Multi-Tanent/includes/config.php` ❌ Should be blocked

- [ ] **Vendor Directory:**
  - `https://your-domain.com/Multi-Tanent/vendor/autoload.php` ❌ Should be blocked
  - `https://your-domain.com/Multi-Tanent/vendor/` ❌ Should be blocked

- [ ] **Database Directory:**
  - `https://your-domain.com/Multi-Tanent/database/` ❌ Should be blocked

- [ ] **Logs Directory:**
  - `https://your-domain.com/Multi-Tanent/logs/` ❌ Should be blocked

- [ ] **Sensitive File Types:**
  - `https://your-domain.com/Multi-Tanent/backup.sql` ❌ Should be blocked
  - `https://your-domain.com/Multi-Tanent/error.log` ❌ Should be blocked

### 5. Upload Directory Protection

- [ ] **PHP files in uploads are blocked:**
  - Create a test PHP file in `uploads/test.php`
  - Try to access: `https://your-domain.com/Multi-Tanent/uploads/test.php` ❌ Should be blocked

- [ ] **Images/files still accessible:**
  - Upload an image: `uploads/test.jpg`
  - Access: `https://your-domain.com/Multi-Tanent/uploads/test.jpg` ✅ Should work

### 6. Trailing Slash Handling

- [ ] Test with trailing slash:
  - `https://your-domain.com/Multi-Tanent/admin/dashboard/` → Should redirect to `/admin/dashboard`

- [ ] Test without trailing slash:
  - `https://your-domain.com/Multi-Tanent/admin/dashboard` → Should work

### 7. Direct .php Access (Optional - Prevents direct access)

If enabled in .htaccess, these should redirect to clean URLs:

- [ ] `https://your-domain.com/Multi-Tanent/admin/dashboard.php` → Redirects to `/admin/dashboard`
- [ ] `https://your-domain.com/Multi-Tanent/admin/login.php` → Redirects to `/admin/login`

---

## 🐛 Debugging Instructions

### Enable mod_rewrite Logging

1. **Edit Apache config:**
   ```apache
   # Add to httpd.conf or virtual host config
   LogLevel rewrite:trace6
   RewriteLog /var/log/apache2/rewrite.log
   RewriteLogLevel 3
   ```

2. **Check logs:**
   ```bash
   tail -f /var/log/apache2/rewrite.log
   ```

### Test URL Rewriting with curl

```bash
# Test clean URL
curl -I https://your-domain.com/Multi-Tanent/admin/dashboard

# Test tenant URL
curl -I https://your-domain.com/Multi-Tanent/kine/register

# Test protected file (should return 403)
curl -I https://your-domain.com/Multi-Tanent/config.php
```

### Check Apache Error Logs

**Linux:**
```bash
tail -f /var/log/apache2/error.log
```

**Windows (XAMPP):**
```bash
tail -f C:\xampp\apache\logs\error.log
```

### Verify .htaccess is Being Read

Add a test rule that will definitely fail:
```apache
# Add to .htaccess temporarily
RewriteRule ^test-url$ - [F,L]
```

Then test: `https://your-domain.com/Multi-Tanent/test-url`

Should return 403. If it returns 404, .htaccess is not being read.

### Check AllowOverride Setting

In your Apache virtual host config or `httpd.conf`:

```apache
<Directory "/path/to/Multi-Tanent">
    AllowOverride All  # Must be "All", not "None"
    Require all granted
</Directory>
```

After changing, restart Apache:
```bash
sudo systemctl restart apache2
# or
sudo service apache2 restart
```

### Browser Developer Tools

1. Open browser Developer Tools (F12)
2. Go to Network tab
3. Navigate to a clean URL
4. Check:
   - **Request URL:** Should show the clean URL
   - **Response Status:** Should be 200 (not 404)
   - **Redirects:** Check if any redirects occurred

### PHP Debugging

Add this to test if .htaccess is working:

```php
<?php
// test_rewrite.php
echo "Request URI: " . $_SERVER['REQUEST_URI'] . "\n";
echo "Script Name: " . $_SERVER['SCRIPT_NAME'] . "\n";
echo "Query String: " . ($_SERVER['QUERY_STRING'] ?? 'none') . "\n";
```

Access via: `https://your-domain.com/Multi-Tanent/test-rewrite` (without .php)

---

## 🔧 Troubleshooting

### Issue: Clean URLs Return 404

**Possible Causes:**
1. mod_rewrite not enabled
2. AllowOverride is None
3. .htaccess file missing or corrupted
4. RewriteBase path incorrect

**Solutions:**
1. Enable mod_rewrite: `sudo a2enmod rewrite`
2. Set `AllowOverride All` in Apache config
3. Verify .htaccess exists in root
4. Check RewriteBase matches your installation path

### Issue: Protected Files Still Accessible

**Possible Causes:**
1. .htaccess files not in subdirectories
2. Apache not processing .htaccess
3. Incorrect file path in RewriteRule

**Solutions:**
1. Verify `includes/.htaccess`, `vendor/.htaccess` exist
2. Check AllowOverride setting
3. Test with direct file access

### Issue: PHP Files Execute in Uploads

**Possible Causes:**
1. `uploads/.htaccess` missing
2. PHP handler still active for uploads directory

**Solutions:**
1. Create `uploads/.htaccess` with PHP blocking rules
2. Verify Apache is processing the .htaccess

### Issue: Tenant URLs Not Working

**Possible Causes:**
1. RewriteRule pattern incorrect
2. School code format mismatch
3. Query parameter not being passed

**Solutions:**
1. Check RewriteRule syntax in .htaccess
2. Verify school codes match pattern `[a-zA-Z0-9_-]+`
3. Add `[QSA]` flag to preserve query strings

### Issue: Infinite Redirect Loop

**Possible Causes:**
1. Conflicting rewrite rules
2. Redirect in both .htaccess and PHP
3. HTTPS enforcement conflict

**Solutions:**
1. Review all RewriteRules for conflicts
2. Remove PHP redirects if using .htaccess
3. Check HTTPS enforcement logic

---

## 📝 Quick Reference

### Common .htaccess Commands

```apache
# Enable rewrite engine
RewriteEngine On

# Set base path
RewriteBase /Multi-Tanent/

# Block access
RewriteRule ^path$ - [F,L]

# Redirect
RewriteRule ^old$ /new [R=301,L]

# Rewrite (internal)
RewriteRule ^clean-url$ file.php [L,QSA]
```

### Apache Commands

```bash
# Enable mod_rewrite
sudo a2enmod rewrite

# Restart Apache
sudo systemctl restart apache2

# Check Apache status
sudo systemctl status apache2

# Check loaded modules
apache2ctl -M
```

---

## ✅ Success Criteria

All tests should pass:
- ✅ Clean URLs work without .php extension
- ✅ Tenant isolation URLs work correctly
- ✅ Protected files return 403 Forbidden
- ✅ Upload directories block PHP execution
- ✅ Directory browsing is disabled
- ✅ Sensitive files are inaccessible

---

## 📞 Support

If issues persist:
1. Check Apache error logs
2. Verify all .htaccess files are uploaded
3. Test with automated test suite
4. Review troubleshooting section above

---

**Last Updated:** 2025-01-16  
**Version:** 2.0

