-- ================================================================
-- FIX STUDENT NAME FIELDS - Database Schema Alignment
-- ================================================================
-- This script ensures the students table has the correct name fields
-- Run this in cPanel → phpMyAdmin → SQL tab
-- ================================================================

-- ================================================================
-- PART 1: Check current structure
-- ================================================================

-- Check what name fields exist
SELECT 
    'Current Structure' as info,
    COLUMN_NAME,
    DATA_TYPE,
    IS_NULLABLE
FROM information_schema.COLUMNS 
WHERE TABLE_NAME = 'students' 
AND TABLE_SCHEMA = DATABASE()
AND COLUMN_NAME IN ('first_name', 'last_name', 'full_name')
ORDER BY COLUMN_NAME;

-- ================================================================
-- PART 2: Add missing name fields (if needed)
-- ================================================================

-- Add first_name if missing
ALTER TABLE `students` 
ADD COLUMN IF NOT EXISTS `first_name` VARCHAR(100) NULL AFTER `id`;

-- Add last_name if missing  
ALTER TABLE `students` 
ADD COLUMN IF NOT EXISTS `last_name` VARCHAR(100) NULL AFTER `first_name`;

-- Add full_name if missing
ALTER TABLE `students` 
ADD COLUMN IF NOT EXISTS `full_name` VARCHAR(200) NULL AFTER `last_name`;

-- ================================================================
-- PART 3: Populate name fields from existing data
-- ================================================================

-- If full_name exists but first_name/last_name don't, split full_name
UPDATE `students` 
SET 
    `first_name` = TRIM(SUBSTRING_INDEX(`full_name`, ' ', 1)),
    `last_name` = TRIM(SUBSTRING_INDEX(`full_name`, ' ', -1))
WHERE (`first_name` IS NULL OR `first_name` = '') 
AND (`last_name` IS NULL OR `last_name` = '')
AND `full_name` IS NOT NULL 
AND `full_name` != '';

-- If first_name/last_name exist but full_name doesn't, combine them
UPDATE `students` 
SET `full_name` = CONCAT(TRIM(`first_name`), ' ', TRIM(`last_name`))
WHERE (`full_name` IS NULL OR `full_name` = '')
AND `first_name` IS NOT NULL 
AND `first_name` != ''
AND `last_name` IS NOT NULL 
AND `last_name` != '';

-- ================================================================
-- PART 4: Handle edge cases
-- ================================================================

-- For students with only first_name, set last_name to empty
UPDATE `students` 
SET `last_name` = ''
WHERE `last_name` IS NULL;

-- For students with only last_name, set first_name to empty  
UPDATE `students` 
SET `first_name` = ''
WHERE `first_name` IS NULL;

-- Ensure full_name is always populated
UPDATE `students` 
SET `full_name` = CONCAT(TRIM(COALESCE(`first_name`, '')), ' ', TRIM(COALESCE(`last_name`, '')))
WHERE `full_name` IS NULL OR `full_name` = '';

-- ================================================================
-- PART 5: Verify the fix
-- ================================================================

-- Show sample data
SELECT 
    'Sample Data' as info,
    id,
    username,
    first_name,
    last_name,
    full_name,
    email
FROM students 
ORDER BY id 
LIMIT 10;

-- Count records with proper names
SELECT 
    'Data Quality Check' as info,
    COUNT(*) as total_students,
    COUNT(CASE WHEN first_name IS NOT NULL AND first_name != '' THEN 1 END) as has_first_name,
    COUNT(CASE WHEN last_name IS NOT NULL AND last_name != '' THEN 1 END) as has_last_name,
    COUNT(CASE WHEN full_name IS NOT NULL AND full_name != '' THEN 1 END) as has_full_name
FROM students;

-- ================================================================
-- PART 6: Final verification
-- ================================================================

-- Check all name fields exist
SELECT 
    'Final Structure' as info,
    COLUMN_NAME,
    DATA_TYPE,
    IS_NULLABLE
FROM information_schema.COLUMNS 
WHERE TABLE_NAME = 'students' 
AND TABLE_SCHEMA = DATABASE()
AND COLUMN_NAME IN ('first_name', 'last_name', 'full_name')
ORDER BY COLUMN_NAME;

-- ================================================================
-- DONE! Dashboard should now work without name field errors!
-- ================================================================
