# 🖥️ Localhost Testing - Complete Setup Guide

## ✅ Your System is Ready to Test!

---

## 🚀 Quick Start (2 Minutes)

### Step 1: Run the Quick Test
```bash
cd C:\xampp\htdocs\Multi-Tanent
php quick_test.php
```

**Expected:**
```
✅ ALL TESTS PASSED!
Your multi-tenant system is ready to use!
```

### Step 2: Open in Browser
```
http://localhost/Multi-Tanent/admin/login.php
```

### Step 3: Login
```
Username: admin
Password: admin123
```

### Step 4: Select School
- You'll see a beautiful school selector
- Currently shows: **Kine Academy** and **Test School**
- Click on **Kine Academy**

### Step 5: You're In! 🎉
- Dashboard loads
- Shows "Kine Academy" in header
- Ready to create students!

---

## 🎓 Full Testing Tutorial

### Tutorial 1: Enable Super Admin Mode (2 minutes)

**Why?** To access the super admin panel and create new schools.

**Option A: Via Database (Recommended)**

1. Open phpMyAdmin: `http://localhost/phpmyadmin`
2. Select database: `multi_tenant_skoloi`
3. Click "SQL" tab
4. Run this query:
```sql
UPDATE admins 
SET academy_reference = NULL 
WHERE username = 'admin';
```
5. Click "Go"
6. Result: ✅ "1 row affected"

**Option B: Via Command Line**
```bash
mysql -u root -p
```
Then:
```sql
USE multi_tenant_skoloi;
UPDATE admins SET academy_reference = NULL WHERE username = 'admin';
SELECT username, academy_reference FROM admins WHERE username = 'admin';
-- Should show: admin | NULL
exit;
```

**Option C: Via PHP Script**

Create file: `set_super_admin.php`
```php
<?php
require_once 'config.php';
$pdo = getDB();

$stmt = $pdo->prepare("UPDATE admins SET academy_reference = NULL WHERE username = 'admin'");
$stmt->execute();

echo "✅ Super admin mode enabled!<br>";
echo "<a href='admin/login.php'>Login Again</a>";
```

Visit: `http://localhost/Multi-Tanent/set_super_admin.php`

**Verify:**
- Logout and login again
- You should now see "Super Admin" link in navigation (gold/yellow color)

---

### Tutorial 2: Create Your First Test School (5 minutes)

1. **Login as super admin**
   - Username: `admin`
   - Password: `admin123`

2. **Click "Super Admin"** in navigation
   - You'll see the super admin dashboard
   - Shows system-wide statistics

3. **Click "Add New School"**
   - Or go to: `http://localhost/Multi-Tanent/super_admin/schools/create.php`

4. **Fill in the form:**
   ```
   School Name: Rainbow Academy
   Reference Code: RAINBOW
   Description: Progressive learning center for creative minds
   Admin Email: admin@rainbow.com
   Admin Password: Rainbow123!
   ```

5. **Click "Create School"**

6. **Success Message Appears:**
   ```
   ✅ School created successfully! 
   Admin username: rainbow_admin
   ```

7. **Verify in School List:**
   - Go to: Super Admin → Manage Schools
   - You should see:
     - Kine Academy (KINE)
     - Test School (TEST)
     - Rainbow Academy (RAINBOW) ← New!

---

### Tutorial 3: Test School Isolation (10 minutes)

**Create Student in Kine Academy:**

1. **Switch to Kine Academy**
   - Click "Switch School" in navigation
   - Select "Kine Academy"

2. **Create Student:**
   - Dashboard → Click green "Create Student" button
   - Fill in:
     ```
     First Name: Alice
     Last Name: Kine
     Email: alice.kine@test.com
     Phone: +268 1111 1111
     Grade: Grade 7
     Password: Alice123!
     Confirm Password: Alice123!
     Subjects: Select 5-6 subjects
     ```
   - Click "Create Student"

3. **Note Alice's Details:**
   - Student appears in dashboard
   - Note her student ID (e.g., 2025102301)
   - Verify "School: Kine Academy" in her details

**Create Student in Rainbow Academy:**

1. **Switch to Rainbow**
   - Click "Switch School"
   - Select "Rainbow Academy"
   - Notice dashboard is EMPTY (no students)

2. **Create Student:**
   - Click "Create Student"
   - Fill in:
     ```
     First Name: Bob
     Last Name: Rainbow
     Email: bob.rainbow@test.com
     Phone: +268 2222 2222
     Grade: Grade 7
     Password: Bob123!
     Subjects: Select 5-6 subjects
     ```
   - Click "Create Student"

3. **Verify:**
   - Bob appears in Rainbow dashboard
   - Alice does NOT appear (she's in Kine)

**Test Cross-School Access:**

1. **While in Rainbow**, try to edit Alice:
   ```
   http://localhost/Multi-Tanent/admin/edit.php?id=[Alice's ID]
   ```
   
2. **Expected Result:**
   - ✅ Access Denied
   - OR redirected to dashboard
   - Cannot edit students from other schools

3. **Switch back to Kine**
   - Verify Alice IS visible and editable
   - Verify Bob is NOT visible

✅ **Perfect isolation!**

---

### Tutorial 4: Test School Settings (5 minutes)

**Customize Rainbow Academy:**

1. **Switch to Rainbow Academy**

2. **Go to School Settings**
   - Click "School Settings" in navigation
   - Or: `http://localhost/Multi-Tanent/admin/settings/school_settings.php`

3. **Update Settings:**
   ```
   School Name: Rainbow Progressive Academy
   School Email: contact@rainbow.academy
   School Phone: +268 7777 7777
   School Address: 123 Rainbow Street, Mbabane, Eswatini
   Academic Year: 2024/2025
   
   Branding:
   Primary Color: #FF6B6B (red/pink)
   Secondary Color: #4ECDC4 (teal)
   
   Email Settings:
   Email From Name: Rainbow Academy Team
   Email Signature: "Best regards,\nRainbow Academy\nWhere Dreams Come True"
   ```

4. **Click "Save Settings"**

5. **Verify:**
   - Success message appears
   - Reload page
   - School name in header shows "Rainbow Progressive Academy"

**Customize St. Mary's (if created):**

1. Switch to St. Mary's
2. Set different colors:
   ```
   Primary Color: #7C3AED (purple)
   Secondary Color: #F59E0B (orange)
   ```
3. Save
4. Switch between schools and see different branding!

---

### Tutorial 5: Test Payment Processing (10 minutes)

**Create Test Payment for Alice (Kine):**

Open phpMyAdmin or run:
```sql
-- Get Alice's ID first
SELECT id, first_name, last_name FROM students WHERE email = 'alice.kine@test.com';
-- Note the ID (e.g., 526)

-- Create payment
INSERT INTO payments 
(student_id, amount, currency, reference_id, external_id, msisdn, status, payment_method, academy_reference, platform_fee, admin_earnings, created_at)
VALUES 
(526, 3.00, 'SZL', UUID(), 'KINE_TEST001', '76123456', 'SUCCESSFUL', 'MOMO', 'KINE', 0.50, 2.50, NOW());
```

**Create Test Payment for Bob (Rainbow):**
```sql
-- Get Bob's ID
SELECT id FROM students WHERE email = 'bob.rainbow@test.com';
-- Note the ID (e.g., 527)

-- Create payment
INSERT INTO payments 
(student_id, amount, currency, reference_id, external_id, msisdn, status, payment_method, academy_reference, platform_fee, admin_earnings, created_at)
VALUES 
(527, 2.70, 'SZL', UUID(), 'RAIN_TEST001', '76789012', 'SUCCESSFUL', 'MOMO', 'RAINBOW', 0.50, 2.20, NOW());
```

**Verify in Dashboards:**

1. **Switch to Kine Academy**
   - Go to dashboard
   - Scroll to wallet section
   - Verify: Payment E3.00 visible
   - Verify: Earnings E2.50

2. **Switch to Rainbow Academy**
   - Go to dashboard
   - Verify: Payment E2.70 visible
   - Verify: Earnings E2.20
   - Verify: Alice's payment NOT visible

3. **Super Admin View**
   - Go to Super Admin dashboard
   - Verify total revenue: E5.70
   - Verify total platform fees: E1.00

✅ **Payment isolation perfect!**

---

### Tutorial 6: Test Super Admin Features (10 minutes)

**View All Schools:**
```
http://localhost/Multi-Tanent/super_admin/schools/list.php
```

You'll see a table with:
- School Name
- Reference Code
- Student Count
- Revenue
- Wallet Balance
- Status (Active/Suspended)

**Test School Details:**

1. Click "View" (eye icon) on Rainbow
2. See detailed statistics:
   - Students: 1 (Bob)
   - Payments: 1
   - Revenue: E2.70
   - Wallet: E2.20
3. See settings, branding, admin info

**Test School Editing:**

1. Click "Edit" (pencil icon)
2. Change school name to: "Rainbow Academy - Updated"
3. Save
4. Go back to list
5. Verify name updated

**Test Suspend/Activate:**

1. Click "Suspend" (pause icon) on Test school
2. Confirm suspension
3. School status changes to "Suspended"
4. Click "Activate" to re-enable
5. Status changes back to "Active"

---

### Tutorial 7: Test School Switching (3 minutes)

**Method 1: Via School Selector**
1. Click "Switch School" in navigation
2. You'll see grid of all schools
3. Click any school
4. Dashboard updates immediately
5. Header shows new school name

**Method 2: Via URL**
```
http://localhost/Multi-Tanent/select_school.php?school=KINE
http://localhost/Multi-Tanent/select_school.php?school=RAINBOW
http://localhost/Multi-Tanent/select_school.php?school=TEST
```

Each URL:
- Sets that school as active
- Redirects to dashboard
- Shows only that school's data

---

## 🧪 Advanced Testing

### Test 1: Session Persistence

1. Login and select Kine Academy
2. Close browser completely
3. Re-open browser
4. Go to: `http://localhost/Multi-Tanent/admin/dashboard.php`
5. Verify: Still in Kine Academy (session persists)

---

### Test 2: Multiple Browser Test

**Chrome:**
- Login as super admin
- Select Kine Academy
- Create student "Chrome User"

**Firefox:**
- Login as `rainbow_admin`
- Automatically in Rainbow Academy
- Create student "Firefox User"

**Edge:**
- Login as super admin
- Select St. Mary's
- Create student "Edge User"

**Verify:**
- Chrome shows only Chrome User (in Kine)
- Firefox shows only Firefox User (in Rainbow)
- Edge shows only Edge User (in St. Mary's)
- Each browser has independent session

---

### Test 3: SQL Injection Test

Try malicious input in school selector:
```
http://localhost/Multi-Tanent/select_school.php?school=KINE' OR '1'='1
```

**Expected:** 
- ✅ No error
- ✅ Invalid school message
- ✅ System protected

---

### Test 4: Direct Database Verification

```sql
-- Check all students have schools
SELECT 
    COALESCE(academy_reference, 'NULL') as school,
    COUNT(*) as count
FROM students
GROUP BY academy_reference;

-- Expected:
-- KINE    | 1 (Alice)
-- RAINBOW | 1 (Bob)
-- STMARY  | 1 (Charlie) [if created]

-- Check payments isolation
SELECT 
    p.academy_reference,
    COUNT(*) as payments,
    SUM(p.amount) as revenue,
    SUM(p.platform_fee) as platform_fees,
    SUM(p.admin_earnings) as school_earnings
FROM payments p
WHERE status = 'SUCCESSFUL'
GROUP BY p.academy_reference;

-- Check settings per school
SELECT 
    academy_reference,
    COUNT(*) as settings_count
FROM tenant_settings
GROUP BY academy_reference;
```

---

## 🎯 Testing Checklist

Print this and check off:

### Initial Setup
- [ ] XAMPP running
- [ ] MySQL running
- [ ] Database exists
- [ ] `php quick_test.php` passes

### Super Admin
- [ ] Can enable super admin mode
- [ ] Can access `/super_admin/dashboard.php`
- [ ] Can see all schools
- [ ] Can create new school
- [ ] Can edit school
- [ ] Can suspend school

### School Admin
- [ ] Can login as school admin
- [ ] Auto-assigned to their school
- [ ] Cannot access other schools
- [ ] Cannot access super admin panel

### Data Isolation
- [ ] Create student in School A
- [ ] Create student in School B
- [ ] School A shows only A students
- [ ] School B shows only B students
- [ ] Direct URL access blocked

### School Settings
- [ ] Can customize each school
- [ ] Settings save correctly
- [ ] Each school independent
- [ ] Branding applies

### Payments
- [ ] Create payment for School A
- [ ] Create payment for School B
- [ ] Payments filtered by school
- [ ] Platform fees tracked

### Navigation
- [ ] School switcher works
- [ ] Current school shown in header
- [ ] Super admin link (if super admin)
- [ ] School settings link

---

## 🐛 Common Issues & Fixes

### Issue: Undefined variable $pdo
**Fix:** ✅ Already fixed! File updated with `$pdo = getDB();`

### Issue: School selector shows no schools
**Fix:**
```sql
-- Check academy_references table
SELECT * FROM academy_references;

-- If empty, add default:
INSERT INTO academy_references (admin_id, academy_name, reference_code, is_active)
VALUES (1, 'Kine Academy', 'KINE', 1);
```

### Issue: Students not filtered by school
**Fix:**
```bash
# Re-run migrations
php database/run_all_missing_columns.php
```

### Issue: Cannot see super admin link
**Fix:**
```sql
UPDATE admins SET academy_reference = NULL WHERE username = 'admin';
```

### Issue: Headers already sent
**Fix:** 
- Check for any output before `header()` calls
- Remove any `echo` or whitespace before `<?php`

---

## 📊 Sample Test Data Generator

Create file: `generate_test_data.php`
```php
<?php
require_once 'config.php';
require_once 'includes/school_context.php';

$pdo = getDB();

echo "Generating test data...\n\n";

// Create 3 test schools
$schools = [
    ['name' => 'Rainbow Academy', 'code' => 'RAINBOW', 'email' => 'admin@rainbow.com'],
    ['name' => 'Grace High School', 'code' => 'GRACE', 'email' => 'admin@grace.com'],
    ['name' => 'Future Leaders', 'code' => 'FUTURE', 'email' => 'admin@future.com']
];

foreach ($schools as $school) {
    try {
        // Check if exists
        $stmt = $pdo->prepare("SELECT id FROM academy_references WHERE reference_code = ?");
        $stmt->execute([$school['code']]);
        
        if (!$stmt->fetch()) {
            $username = strtolower($school['code']) . '_admin';
            $password = password_hash($school['code'] . '123!', PASSWORD_DEFAULT);
            
            // Create admin
            $stmt = $pdo->prepare("INSERT INTO admins (username, password_hash, academy_reference) VALUES (?, ?, ?)");
            $stmt->execute([$username, $password, $school['code']]);
            $adminId = $pdo->lastInsertId();
            
            // Create academy
            $stmt = $pdo->prepare("
                INSERT INTO academy_references (admin_id, academy_name, reference_code, is_active)
                VALUES (?, ?, ?, 1)
            ");
            $stmt->execute([$adminId, $school['name'], $school['code']]);
            
            // Create wallet
            $stmt = $pdo->prepare("INSERT INTO admin_wallet (admin_id, balance, currency) VALUES (?, 0, 'SZL')");
            $stmt->execute([$adminId]);
            
            echo "✓ Created: {$school['name']} (Username: $username)\n";
        } else {
            echo "⚠ Exists: {$school['name']}\n";
        }
    } catch (PDOException $e) {
        echo "✗ Error creating {$school['name']}: " . $e->getMessage() . "\n";
    }
}

echo "\n✅ Test data generation complete!\n";
echo "\nSchools created:\n";
echo "1. RAINBOW → rainbow_admin / RAINBOW123!\n";
echo "2. GRACE → grace_admin / GRACE123!\n";
echo "3. FUTURE → future_admin / FUTURE123!\n";
```

Run: `php generate_test_data.php`

---

## 📱 Browser Testing Steps

### Chrome DevTools Testing

1. **Open DevTools** (F12)
2. **Go to Application tab**
3. **Check Session Storage:**
   - Should see `academy_reference`
   - Should see `academy_name`
   - Should see `school_data_cache`

4. **Go to Network tab**
5. **Perform action** (e.g., create student)
6. **Check requests:**
   - Verify POST data includes academy_reference
   - Verify responses filtered by school

---

## 🎬 Testing Videos You Can Record

### Video 1: System Overview (3 min)
1. Login screen
2. School selector
3. Dashboard overview
4. Create student
5. School switcher

### Video 2: Multi-School Demo (5 min)
1. Create 2 schools
2. Add students to each
3. Switch between them
4. Show data isolation
5. Show independent settings

### Video 3: Super Admin Power (5 min)
1. Super admin dashboard
2. Create new school
3. View all schools
4. Access different school dashboards
5. Manage schools

---

## ✅ Success Criteria

Your localhost testing is complete when:

- [ ] `quick_test.php` passes all tests
- [ ] Can login as super admin
- [ ] Can create new schools
- [ ] Can create students in different schools
- [ ] Students properly isolated
- [ ] Can switch between schools
- [ ] Settings save per school
- [ ] Payments filtered by school
- [ ] Cross-school access blocked
- [ ] No errors in console/logs

---

## 🎊 You're Ready!

Once all checklist items are ✅, your system is **tested and ready for production**!

---

## 📞 Quick Help

**Test Not Passing?**
```bash
php database/run_all_missing_columns.php
```

**Need to Reset?**
```bash
# Backup first!
mysqldump -u root -p multi_tenant_skoloi > backup.sql

# Then re-import
mysql -u root -p multi_tenant_skoloi < SKOLO_KINE_COMPLETE_DATABASE.sql
```

**Files Missing?**
- Check you're in: `C:\xampp\htdocs\Multi-Tanent`
- All files should be there

---

**Happy Testing!** 🧪

**Start here:** `php quick_test.php` then open browser!

🚀 **Your localhost multi-tenant system is ready!** 🚀

