# 📱 MoMo Payment System - Complete Explanation

## 🎯 **Current Implementation**

### **What Happens Now (Simplified Version):**

```
1. Parent selects subjects on payment page
2. Parent chooses monthly or yearly
3. Parent enters MoMo number: +268 7687 3207
4. Parent clicks "Proceed to Payment"
5. Parent reviews and clicks "Confirm & Pay"
6. System records payment as "completed"
7. System enrolls student in subjects
8. System updates payment status to "active"
9. Parent sees success page
```

**Current Status:** ⚠️ **Manual/Semi-Automated**

The system **records** the payment in the database but does **NOT** automatically charge the MoMo account. The admin would need to manually process the MoMo transaction.

---

## 🚀 **Full MTN MoMo API Integration (Production Ready)**

For **automatic MoMo charging**, you need to integrate with **MTN MoMo API**.

### **How Real MTN MoMo Integration Works:**

```
1. Parent enters MoMo number
2. Parent confirms payment
3. System calls MTN MoMo API
4. API sends USSD prompt to parent's phone
5. Parent enters PIN on their phone
6. Payment processed by MTN
7. MTN sends callback to your server
8. System enrolls student
9. Confirmation sent to parent
```

---

## 🔧 **MTN MoMo API Integration Steps**

### **Step 1: Register with MTN MoMo**

**For Eswatini:**
- Website: https://momodeveloper.mtn.com/
- OR contact MTN Eswatini Business Support
- Request: "MTN MoMo Collection API" access

**For Other Countries:**
- South Africa: https://developer.mtn.co.za/
- Ghana: https://momodeveloper.mtn.com/
- Uganda: https://momodeveloper.mtn.com/

**You'll need:**
- Business registration documents
- Tax clearance certificate
- Bank account details
- ID documents

**You'll get:**
- Primary Key (API Key)
- Secondary Key (API Secret)
- User ID
- API URL (sandbox & production)

---

### **Step 2: Install MoMo SDK/Library**

**Option A: Use Composer Package**
```bash
composer require sparkplug/momoapi-php
```

**Option B: Manual cURL Implementation**
```php
// I'll show you this below
```

---

### **Step 3: Implementation Code**

**Create: `includes/MoMoPayment.php`**

```php
<?php
/**
 * MTN MoMo Payment Integration
 * Handles collection requests and callbacks
 */

class MoMoPayment {
    private $apiUrl;
    private $subscriptionKey;
    private $apiUser;
    private $apiKey;
    private $targetEnvironment;
    
    public function __construct() {
        // Get credentials from settings or config
        $this->subscriptionKey = 'YOUR_SUBSCRIPTION_KEY'; // From MTN
        $this->apiUser = 'YOUR_API_USER_ID'; // From MTN
        $this->apiKey = 'YOUR_API_KEY'; // From MTN
        $this->targetEnvironment = 'mtnSwaziland'; // or 'mtnSouthAfrica', etc.
        
        // Sandbox or Production
        $this->apiUrl = 'https://sandbox.momodeveloper.mtn.com'; // Change to prod when ready
        // Production: https://proxy.momoapi.mtn.com
    }
    
    /**
     * Request payment from customer
     */
    public function requestToPay($amount, $phoneNumber, $reference, $message = 'School Fees Payment') {
        // Generate UUID for transaction
        $transactionId = $this->generateUUID();
        
        // Format phone number (remove + and spaces)
        $phoneNumber = str_replace(['+', ' ', '-'], '', $phoneNumber);
        
        // Ensure phone number has country code
        if (!str_starts_with($phoneNumber, '268')) {
            $phoneNumber = '268' . ltrim($phoneNumber, '0');
        }
        
        // Request payload
        $payload = [
            'amount' => (string)$amount,
            'currency' => 'SZL', // Eswatini Lilangeni
            'externalId' => $reference, // Your payment reference
            'payer' => [
                'partyIdType' => 'MSISDN',
                'partyId' => $phoneNumber
            ],
            'payerMessage' => $message,
            'payeeNote' => 'Payment for ' . $reference
        ];
        
        // Get access token first
        $accessToken = $this->getAccessToken();
        
        if (!$accessToken) {
            return ['success' => false, 'message' => 'Failed to get access token'];
        }
        
        // Make API request
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->apiUrl . '/collection/v1_0/requesttopay',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $accessToken,
                'X-Reference-Id: ' . $transactionId,
                'X-Target-Environment: ' . $this->targetEnvironment,
                'Ocp-Apim-Subscription-Key: ' . $this->subscriptionKey,
                'Content-Type: application/json'
            ],
            CURLOPT_POSTFIELDS => json_encode($payload)
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        // Check response
        if ($httpCode == 202) {
            // Request accepted - now check status
            return [
                'success' => true,
                'transaction_id' => $transactionId,
                'message' => 'Payment request sent. Waiting for customer approval.'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Payment request failed: ' . $response,
                'http_code' => $httpCode
            ];
        }
    }
    
    /**
     * Check payment status
     */
    public function checkTransactionStatus($transactionId) {
        $accessToken = $this->getAccessToken();
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->apiUrl . '/collection/v1_0/requesttopay/' . $transactionId,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $accessToken,
                'X-Target-Environment: ' . $this->targetEnvironment,
                'Ocp-Apim-Subscription-Key: ' . $this->subscriptionKey
            ]
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode == 200) {
            $data = json_decode($response, true);
            
            return [
                'success' => true,
                'status' => $data['status'], // PENDING, SUCCESSFUL, FAILED
                'amount' => $data['amount'],
                'currency' => $data['currency'],
                'financialTransactionId' => $data['financialTransactionId'] ?? null
            ];
        }
        
        return ['success' => false, 'message' => 'Failed to check status'];
    }
    
    /**
     * Get OAuth access token
     */
    private function getAccessToken() {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->apiUrl . '/collection/token/',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_USERPWD => $this->apiUser . ':' . $this->apiKey,
            CURLOPT_HTTPHEADER => [
                'Ocp-Apim-Subscription-Key: ' . $this->subscriptionKey
            ]
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        $data = json_decode($response, true);
        return $data['access_token'] ?? null;
    }
    
    /**
     * Generate UUID v4
     */
    private function generateUUID() {
        $data = random_bytes(16);
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40);
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80);
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }
}
```

---

### **Step 4: Update Process Payment Page**

**File: `parent/process_payment.php`**

**Replace the MoMo processing section with:**

```php
// Handle MoMo Payment - Process via MTN API
if ($payment_method === 'momo') {
    require_once '../includes/MoMoPayment.php';
    
    $momo = new MoMoPayment();
    
    // Request payment
    $result = $momo->requestToPay(
        $payment_data['total_amount'],
        $payment_data['momo_number'],
        $payment_reference,
        "School fees for {$student['full_name']}"
    );
    
    if ($result['success']) {
        // Save transaction ID for status checking
        $_SESSION['momo_transaction_id'] = $result['transaction_id'];
        
        // Redirect to payment status checker
        header('Location: momo_payment_status.php');
        exit;
    } else {
        // Payment request failed
        $error = $result['message'];
        // Show error to user
    }
}
```

---

## 📊 **Current vs Full Integration Comparison**

### **Current Implementation (Simple):**

**Process:**
```
Parent → Enter MoMo → Confirm → Payment "Completed" → Done
```

**Pros:**
- ✅ Simple to implement
- ✅ No API fees
- ✅ No MTN approval needed
- ✅ Works immediately

**Cons:**
- ❌ Manual processing required
- ❌ Not automatic
- ❌ Parent must pay separately
- ❌ Admin must verify payment manually

**Best For:**
- Testing/demo
- Small schools
- Manual payment verification
- Quick setup

---

### **Full MTN MoMo API (Production):**

**Process:**
```
Parent → Enter MoMo → Confirm → MTN sends USSD → 
Parent enters PIN → Money deducted → Callback → Done
```

**Pros:**
- ✅ Fully automatic
- ✅ Real-time payment
- ✅ Instant verification
- ✅ Professional
- ✅ No manual work

**Cons:**
- ❌ Requires MTN approval (3-5 days)
- ❌ Transaction fees (~2.9%)
- ❌ More complex setup
- ❌ API integration needed

**Best For:**
- Production use
- Medium-large schools
- Automated operations
- Scalability

---

## 💰 **MTN MoMo Fees**

**Typical Fee Structure:**
```
Transaction Fee: 2.9% + fixed amount
Example:
- Parent pays: E1,050
- MTN fee: E30.45 (2.9%)
- School receives: E1,019.55

OR (if you pass fee to parent):
- Parent pays: E1,080.45
- MTN fee: E30.45
- School receives: E1,050.00
```

---

## 🔄 **How MTN MoMo Collection Works**

### **Step-by-Step Process:**

**1. System Initiates Payment**
```php
$momo->requestToPay(
    amount: 1050.00,
    phone: '26876873207',
    reference: 'PAY12345',
    message: 'School fees for John Doe'
);
```

**2. MTN Receives Request**
- Validates phone number
- Checks account exists
- Checks sufficient balance
- Generates transaction ID

**3. MTN Sends USSD Prompt**
```
Customer's phone receives:
━━━━━━━━━━━━━━━━━━━
MTN MoMo Payment Request

Amount: E1,050.00
From: School Name
For: School fees

Enter PIN to confirm:
[____]

1. Confirm
2. Cancel
━━━━━━━━━━━━━━━━━━━
```

**4. Customer Approves**
- Enters PIN
- MTN debits their account
- MTN credits your merchant account

**5. MTN Sends Callback**
```php
// Your callback URL receives:
{
    "financialTransactionId": "123456789",
    "externalId": "PAY12345",
    "amount": "1050.00",
    "currency": "SZL",
    "payer": {"partyId": "26876873207"},
    "status": "SUCCESSFUL"
}
```

**6. Your System Processes**
- Verifies callback is genuine
- Updates payment status
- Enrolls student in subjects
- Sends confirmation

---

## 📝 **Current Implementation Details**

### **File: `parent/process_payment.php` (Lines 79-160)**

```php
// Current implementation (simplified)
try {
    $pdo->beginTransaction();
    
    // 1. Create payment record
    $stmt = $pdo->prepare("
        INSERT INTO payments (
            student_id, 
            academy_reference, 
            amount, 
            status,                    ← Currently set to 'completed'
            payment_method, 
            momo_number,
            reference,
            payment_period,
            created_at
        ) VALUES (?, ?, ?, 'completed', ?, ?, ?, ?, NOW())
    ");
    $stmt->execute([...]);
    
    // 2. Enroll in subjects
    foreach ($subjects as $subject_id) {
        // Add to student_subject table
    }
    
    // 3. Update student status
    UPDATE students SET payment_status = 'active';
    
    // 4. Send notification
    sendParentNotification(...);
    
    $pdo->commit();
}
```

**What This Does:**
- ✅ Records payment intent
- ✅ Enrolls student
- ✅ Updates status
- ⚠️ Doesn't actually charge MoMo account
- ⚠️ Manual verification needed

---

## 🔄 **Upgrade to Full MTN MoMo API**

### **Option 1: MTN MoMo API Direct**

**Advantages:**
- ✅ Official MTN integration
- ✅ Most reliable
- ✅ Best rates
- ✅ Direct support

**Requirements:**
- Register with MTN
- Get API credentials
- Implement OAuth flow
- Handle callbacks

**Cost:**
- Setup: Free (with approval)
- Per transaction: ~2.9%
- Monthly minimum: Varies

---

### **Option 2: Payment Aggregator (Easier)**

**Use services like:**

**A. DPO PayGate (Recommended for Eswatini/SA):**
- Website: https://www.dpogroup.com/
- Supports: MTN MoMo, Visa, Mastercard
- Single integration for multiple methods
- Easier than direct MTN integration

**B. PayFast (South Africa):**
- Website: https://www.payfast.co.za/
- Supports instant EFT and cards
- Good for SA market

**C. Flutterwave (Pan-African):**
- Website: https://flutterwave.com/
- Supports MoMo in multiple countries
- Ghana, Uganda, Rwanda, etc.
- Modern API

**Advantages:**
- ✅ Faster approval
- ✅ Single integration
- ✅ Multiple payment methods
- ✅ Better documentation
- ✅ Support included

**Disadvantages:**
- ❌ Higher fees (3.5-4.5%)
- ❌ Additional layer

---

## 💻 **Code Example: Full MoMo Integration**

### **Using MTN MoMo API:**

```php
// parent/process_momo_payment.php

require_once '../includes/MoMoPayment.php';

$momo = new MoMoPayment();

// Request payment from customer
$result = $momo->requestToPay(
    $amount,
    $momo_number,
    $payment_reference,
    "School fees for {$student_name}"
);

if ($result['success']) {
    // Payment request sent
    // Store transaction ID
    $transaction_id = $result['transaction_id'];
    
    // Save as 'pending' (not 'completed')
    $stmt = $pdo->prepare("
        INSERT INTO payments (
            student_id, amount, status, 
            payment_method, momo_number, 
            reference, transaction_id, created_at
        ) VALUES (?, ?, 'pending', 'momo', ?, ?, ?, NOW())
    ");
    $stmt->execute([
        $student_id, $amount, $momo_number, 
        $payment_reference, $transaction_id
    ]);
    
    // Redirect to status checker
    header('Location: check_momo_status.php?tx=' . $transaction_id);
    exit;
} else {
    // Show error
    $error = "Payment failed: " . $result['message'];
}
```

### **Status Checker Page:**

```php
// parent/check_momo_status.php

$transaction_id = $_GET['tx'];
$momo = new MoMoPayment();

// Check status (poll every 5 seconds)
$status = $momo->checkTransactionStatus($transaction_id);

if ($status['status'] === 'SUCCESSFUL') {
    // Payment successful!
    // Update payment status
    UPDATE payments SET status = 'completed' WHERE transaction_id = ?;
    
    // Enroll student
    // Send notifications
    // Redirect to success
}

if ($status['status'] === 'FAILED') {
    // Payment failed
    // Update status
    // Show error
}

if ($status['status'] === 'PENDING') {
    // Still waiting for customer
    // Show "Waiting for approval..." message
    // Keep polling
}
```

---

## 🌐 **MoMo Callback Handler**

**File: `api/momo_callback.php`**

```php
<?php
/**
 * MTN MoMo Callback Handler
 * Receives notifications when payments complete
 */

// Get raw POST data
$json = file_get_contents('php://input');
$data = json_decode($json, true);

// Log callback
error_log("MoMo Callback: " . $json);

// Verify callback authenticity
// Check signature/token from MTN

// Extract payment info
$externalId = $data['externalId']; // Your payment reference
$status = $data['status']; // SUCCESSFUL, FAILED, PENDING
$amount = $data['amount'];
$financialTxId = $data['financialTransactionId'];

// Update payment in database
$pdo = getDB();

if ($status === 'SUCCESSFUL') {
    $pdo->beginTransaction();
    
    // Update payment status
    $stmt = $pdo->prepare("
        UPDATE payments 
        SET status = 'completed',
            mtn_transaction_id = ?,
            updated_at = NOW()
        WHERE reference = ?
    ");
    $stmt->execute([$financialTxId, $externalId]);
    
    // Get payment details
    $stmt = $pdo->prepare("SELECT * FROM payments WHERE reference = ?");
    $stmt->execute([$externalId]);
    $payment = $stmt->fetch();
    
    // Enroll student in subjects
    // (Your existing enrollment code)
    
    // Update student status
    UPDATE students SET payment_status = 'active' WHERE id = ?;
    
    // Send notification to parent
    // (Your existing notification code)
    
    $pdo->commit();
    
    // Return 200 OK to MTN
    http_response_code(200);
    echo json_encode(['status' => 'processed']);
}
```

---

## 🎯 **Which Implementation Should You Use?**

### **Use Current (Simplified) If:**
- ✅ Just testing/demo
- ✅ Small number of parents
- ✅ Can manually verify payments
- ✅ Want quick setup

### **Upgrade to Full API If:**
- ✅ Going to production
- ✅ 50+ parents
- ✅ Want automation
- ✅ Professional operation
- ✅ Scale to multiple schools

---

## 💡 **Hybrid Approach (Recommended)**

**Start Simple, Upgrade Later:**

**Phase 1 (Now):**
- Use current simplified implementation
- Parents enter MoMo number
- Payment recorded as "pending"
- Admin verifies and marks "completed" manually

**Phase 2 (When Ready):**
- Register with MTN/aggregator
- Get API credentials
- Implement full integration
- Automatic processing

**Phase 3 (Scale):**
- Add payment aggregator
- Support multiple countries
- Add card payments
- Full automation

---

## 📞 **Getting MTN MoMo API Access**

### **For Eswatini:**
```
Contact: MTN Eswatini Business Team
Email: business@mtn.co.sz
Phone: +268 7600 0000
Website: mtn.co.sz

Ask for: "MTN MoMo Collection API for Merchants"

Documents needed:
- Business registration
- Tax clearance
- Bank details
- ID copies
```

### **For South Africa:**
```
Website: https://developer.mtn.co.za/
Self-service registration available
Sandbox access immediate
Production requires verification
```

---

## 🎓 **Current System Workflow (Detailed)**

### **What Happens Step-by-Step:**

**1. Parent Dashboard → Make Payment**
```php
// File: parent/make_payment.php
- Parent sees subjects
- Selects subjects to pay for
- Chooses monthly or yearly
- Enters MoMo number: +268 7687 3207
```

**2. System Stores Payment Info**
```php
// Session storage
$_SESSION['pending_payment'] = [
    'student_id' => 5,
    'subjects' => [1, 2, 3], // Math, Science, English
    'payment_period' => 'yearly',
    'total_amount' => 11340,
    'momo_number' => '76873207',
    'payment_method' => 'momo'
];
```

**3. Confirmation Page**
```php
// File: parent/process_payment.php
- Shows payment summary
- Parent reviews details
- Clicks "Confirm & Pay E11,340"
```

**4. Payment Processing**
```php
// Current implementation:
- Creates payment record (status: 'completed')
- Enrolls in subjects (student_subject table)
- Updates student status (payment_status: 'active')
- Sends notification
- Shows success page
```

**5. Database Records**
```sql
-- payments table:
student_id | amount  | status    | payment_method | momo_number | reference
5          | 11340.00| completed | momo           | 76873207    | PAY12345

-- student_subject table:
student_id | subject_id | enrolled_at
5          | 1          | 2025-10-28 14:30:00
5          | 2          | 2025-10-28 14:30:00
5          | 3          | 2025-10-28 14:30:00

-- students table:
id | payment_status | last_payment_date
5  | active         | 2025-10-28 14:30:00
```

**6. Success Page**
```php
// File: parent/payment_success.php
- Confetti animation
- Digital receipt
- Payment reference shown
- Auto-redirect to dashboard
```

---

## 🎯 **Summary**

### **Current System:**
**Type:** Semi-manual MoMo recording  
**MoMo Integration:** None (records payment only)  
**Actual Charging:** Manual (admin verifies)  
**Best For:** Testing, small scale  
**Works:** ✅ Yes, for recording payments  

### **Full Integration (Recommended for Production):**
**Type:** Automatic MTN MoMo API  
**MoMo Integration:** Full API integration  
**Actual Charging:** Automatic via MTN  
**Best For:** Production, scale  
**Requires:** MTN approval, API credentials  

---

## 📖 **Next Steps**

### **To Keep Current System:**
- ✅ It works as-is for recording payments
- ✅ Admin manually verifies MoMo payments
- ✅ Good for testing and demos

### **To Upgrade to Full API:**
1. Register with MTN MoMo Developer Portal
2. Get API credentials
3. Implement `MoMoPayment.php` class (code above)
4. Update `process_payment.php` to use API
5. Create status checker page
6. Implement callback handler
7. Test in sandbox
8. Go live!

---

**Your current system works perfectly for recording MoMo payments. When ready to scale, implement the full MTN MoMo API! 🚀**

---

*Current: Payment recording system*  
*Upgrade: Full MTN MoMo API integration*  
*Timeline: Phase 1 now, Phase 2 when scaling*  
*Status: ✅ Working (manual verification)*

