# 💰 School-Specific Pricing Configuration Guide

## ✅ **What Changed**

The payment system now **uses each school's specific pricing** from their settings!

---

## 🎯 **How It Works**

### **Each School Sets Their Own Fees:**

```
KINE Academy:    E350 per subject
SOS School:      E400 per subject  
TEST School:     E300 per subject
```

**Parents see the prices for THEIR school when they make a payment!**

---

## 🔧 **Configure Pricing for Each School**

### **Method 1: Via phpMyAdmin (Quick)**

**Run this SQL for each school:**

```sql
-- Set pricing for KINE Academy
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES 
    ('KINE', 'subject_fee', '350'),
    ('KINE', 'yearly_discount', '10'),
    ('KINE', 'currency_symbol', 'E')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);

-- Set pricing for SOS School
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES 
    ('SOSHIGH', 'subject_fee', '400'),
    ('SOSHIGH', 'yearly_discount', '15'),
    ('SOSHIGH', 'currency_symbol', 'E')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);

-- Set pricing for TEST School
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES 
    ('TEST', 'subject_fee', '300'),
    ('TEST', 'yearly_discount', '10'),
    ('TEST', 'currency_symbol', 'E')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);
```

---

### **Method 2: Via Admin Settings Page (Future)**

**In `admin/settings.php`, add a "Pricing" section:**

```
School Settings → Pricing:
- Subject Fee (per month): [E350]
- Yearly Discount %: [10%]
- Currency Symbol: [E]
```

---

## 📊 **Pricing Configuration Options**

### **Setting Keys:**

| Setting Key | Description | Default | Example |
|-------------|-------------|---------|---------|
| `subject_fee` | Price per subject per month | 350 | 400, 300, 500 |
| `yearly_discount` | Discount % for yearly payment | 10 | 10, 15, 20 |
| `currency_symbol` | Currency symbol to display | E | E, R, $, € |

---

## 💡 **Pricing Examples**

### **Scenario 1: Premium School (KINE)**
```sql
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES ('KINE', 'subject_fee', '450') -- E450 per subject
ON DUPLICATE KEY UPDATE setting_value = '450';
```

**Result:**
- Monthly (3 subjects): E1,350
- Yearly (3 subjects, 10% off): E14,580

---

### **Scenario 2: Budget School (TEST)**
```sql
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES ('TEST', 'subject_fee', '250') -- E250 per subject
ON DUPLICATE KEY UPDATE setting_value = '250';
```

**Result:**
- Monthly (3 subjects): E750
- Yearly (3 subjects, 10% off): E8,100

---

### **Scenario 3: Aggressive Yearly Promotion (SOS)**
```sql
-- Higher monthly fee + bigger yearly discount
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES 
    ('SOSHIGH', 'subject_fee', '400'),
    ('SOSHIGH', 'yearly_discount', '20')  -- 20% off yearly!
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);
```

**Result:**
- Monthly (3 subjects): E1,200
- Yearly (3 subjects, 20% off): E11,520 (Save E3,480!)

**Psychology:** Bigger discount encourages upfront yearly payments

---

## 🌍 **Multi-Currency Support**

### **For International Schools:**

```sql
-- South African School (Rand)
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES 
    ('SACADEMY', 'subject_fee', '500'),
    ('SACADEMY', 'currency_symbol', 'R')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);
-- Result: R500 per subject

-- Nigerian School (Naira)
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES 
    ('NGSCHOOL', 'subject_fee', '15000'),
    ('NGSCHOOL', 'currency_symbol', '₦')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);
-- Result: ₦15,000 per subject

-- US School (Dollar)
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
VALUES 
    ('USSCHOOL', 'subject_fee', '50'),
    ('USSCHOOL', 'currency_symbol', '$')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);
-- Result: $50 per subject
```

---

## 📋 **Quick Setup for All Schools**

### **Set Default Pricing for All Existing Schools:**

```sql
-- Get all schools
SELECT reference_code, academy_name FROM academy_references;

-- For each school, set default pricing:
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
SELECT reference_code, 'subject_fee', '350'
FROM academy_references
WHERE is_active = 1
ON DUPLICATE KEY UPDATE setting_value = '350';

INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
SELECT reference_code, 'yearly_discount', '10'
FROM academy_references
WHERE is_active = 1
ON DUPLICATE KEY UPDATE setting_value = '10';

INSERT INTO tenant_settings (academy_reference, setting_key, setting_value)
SELECT reference_code, 'currency_symbol', 'E'
FROM academy_references
WHERE is_active = 1
ON DUPLICATE KEY UPDATE setting_value = 'E';
```

---

## 🧪 **Test Different Pricing**

### **Step 1: Set Different Prices**
```sql
UPDATE tenant_settings 
SET setting_value = '400' 
WHERE academy_reference = 'KINE' AND setting_key = 'subject_fee';

UPDATE tenant_settings 
SET setting_value = '300' 
WHERE academy_reference = 'SOSHIGH' AND setting_key = 'subject_fee';
```

### **Step 2: Test as Parent**
1. Register parent for KINE school student
2. Go to make payment
3. Should see: **E400** per subject

4. Register parent for SOSHIGH school student
5. Go to make payment
6. Should see: **E300** per subject

---

## 📊 **Verify Current Pricing**

**Check what pricing each school has:**

```sql
SELECT 
    ar.reference_code,
    ar.academy_name,
    MAX(CASE WHEN ts.setting_key = 'subject_fee' THEN ts.setting_value END) as subject_fee,
    MAX(CASE WHEN ts.setting_key = 'yearly_discount' THEN ts.setting_value END) as yearly_discount,
    MAX(CASE WHEN ts.setting_key = 'currency_symbol' THEN ts.setting_value END) as currency
FROM academy_references ar
LEFT JOIN tenant_settings ts ON ar.reference_code = ts.academy_reference
    AND ts.setting_key IN ('subject_fee', 'yearly_discount', 'currency_symbol')
WHERE ar.is_active = 1
GROUP BY ar.reference_code, ar.academy_name
ORDER BY ar.academy_name;
```

**Expected output:**
```
reference_code | academy_name | subject_fee | yearly_discount | currency
KINE           | Kine Academy | 350         | 10              | E
SOSHIGH        | SOS          | 350         | 10              | E
TEST           | TEST School  | 350         | 10              | E
```

---

## 💰 **Pricing Strategies**

### **Strategy 1: Uniform Pricing**
```
All schools: E350/subject
Advantage: Simple, consistent
Best for: Standard market positioning
```

### **Strategy 2: Tiered Pricing**
```
Premium schools: E450/subject
Standard schools: E350/subject
Budget schools: E250/subject
Advantage: Market segmentation
Best for: Different demographics
```

### **Strategy 3: Discount Incentive**
```
All schools: E400/month
Yearly discount: 20% (aggressive)
Advantage: Encourages upfront payment
Best for: Cash flow optimization
```

---

## 🎯 **Files Updated**

```
✅ parent/make_payment.php       - Uses school's subject_fee setting
✅ parent/process_payment.php    - Uses school's currency_symbol
✅ parent/payment_success.php    - Uses school's currency_symbol
✅ parent/dashboard.php          - Uses school's currency_symbol
```

---

## 📝 **What Parents See Now**

### **KINE Academy Parent (E350/subject):**
```
Mathematics - E350/month
Science - E350/month
English - E350/month

Monthly (3 subjects): E1,050
Yearly (3 subjects): E11,340 (Save E1,260!)
```

### **SOS School Parent (E400/subject, if configured):**
```
Mathematics - E400/month
Science - E400/month
English - E400/month

Monthly (3 subjects): E1,200
Yearly (3 subjects): E12,960 (Save E1,440!)
```

**Each school's pricing is completely independent!** ✅

---

## 🚀 **Quick Setup Commands**

### **Set up KINE Academy pricing:**
```sql
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value) VALUES
('KINE', 'subject_fee', '350'),
('KINE', 'yearly_discount', '10'),
('KINE', 'currency_symbol', 'E')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);
```

### **Set up SOS School pricing:**
```sql
INSERT INTO tenant_settings (academy_reference, setting_key, setting_value) VALUES
('SOSHIGH', 'subject_fee', '400'),
('SOSHIGH', 'yearly_discount', '15'),
('SOSHIGH', 'currency_symbol', 'E')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);
```

---

## ✅ **Verification**

After setting pricing, verify:

```sql
-- Check KINE pricing
SELECT setting_key, setting_value 
FROM tenant_settings 
WHERE academy_reference = 'KINE' 
AND setting_key IN ('subject_fee', 'yearly_discount', 'currency_symbol');

-- Expected:
-- subject_fee | 350
-- yearly_discount | 10
-- currency_symbol | E
```

---

## 🎉 **Summary**

**Before:**
- ❌ Global pricing (all schools the same)
- ❌ Hardcoded "E" currency symbol
- ❌ No flexibility

**After:**
- ✅ Each school sets own pricing
- ✅ Configurable currency symbols
- ✅ School-specific discounts
- ✅ Complete flexibility
- ✅ Multi-currency ready

---

## 📤 **Files to Upload**

```
✅ parent/make_payment.php
✅ parent/process_payment.php
✅ parent/payment_success.php
✅ parent/dashboard.php
```

---

## 🧪 **Test Now**

1. **Set pricing for your school** (run SQL above)
2. **Upload all 4 files**
3. **Login as parent**
4. **Click "Make Payment"**
5. **Verify pricing shows YOUR school's fee!** ✅

---

**Your payment system now respects each school's individual pricing! 🚀**

---

*Feature: School-Specific Pricing*  
*Status: ✅ Complete*  
*Impact: HIGH - Professional multi-tenant pricing*  
*Configuration: Via tenant_settings table*

