# 🔗 Tenant Directory Manager - Integration Guide

## 🎯 **How to Integrate**

This guide shows you how to integrate the Tenant Directory Manager into your existing multi-tenant system.

---

## 📋 **Files to Upload**

```
✅ includes/tenant_directory_manager.php           - Core class
✅ TENANT_DIRECTORY_MANAGER_GUIDE.md              - Complete documentation
✅ TENANT_DIRECTORY_MANAGER_EXAMPLES.php          - Usage examples
✅ TENANT_DIRECTORY_MANAGER_INTEGRATION.md        - This file
✅ test_tenant_directory_manager.php              - Test suite (delete after testing)
```

---

## 🚀 **Step 1: Upload Files**

**Via cPanel File Manager:**
1. Upload `tenant_directory_manager.php` to `/includes/`
2. Upload documentation files to root or `/docs/`
3. Upload test file temporarily

---

## 🧪 **Step 2: Test the System**

Visit: `https://test.melanegroup.com/Multi-Tanent/test_tenant_directory_manager.php`

**Expected Results:**
```
✅ Test 1: Initialize Tenant - PASS
✅ Test 2: Get Tenant Paths - PASS
✅ Test 3: Verify Directory Structure - PASS
✅ Test 4: Configuration Management - PASS
✅ Test 5: Update Configuration - PASS
✅ Test 6: Calculate Storage Usage - PASS
✅ Test 7: Update Storage Quota - PASS
✅ Test 8: Tenant ID Validation - PASS
✅ Test 9: Security Files Created - PASS
✅ Test 10: Cleanup - PASS
```

After testing, **DELETE the test file** from production!

---

## 🔌 **Step 3: Integration Points**

### **A. School Creation (`super_admin/schools/create.php`)**

**Add after successful database insertion:**

```php
<?php
// Existing code: Insert school into database
$stmt = $conn->prepare("
    INSERT INTO tenants (academy_reference, name, contact_email, contact_phone)
    VALUES (?, ?, ?, ?)
");
$stmt->execute([$academy_reference, $school_name, $email, $phone]);

// ✅ NEW: Initialize tenant directories
require_once '../../includes/tenant_directory_manager.php';
$dirManager = new TenantDirectoryManager($conn);

$initResult = $dirManager->initializeTenant(
    $academy_reference,
    $school_name,
    [
        'contact_email' => $email,
        'contact_phone' => $phone,
        'storage_quota' => 1073741824, // 1GB default
        'subscription_tier' => 'basic'
    ]
);

if (!$initResult['success']) {
    // Log error but don't block school creation
    error_log("Failed to initialize directories for {$academy_reference}: " . $initResult['error']);
    $_SESSION['warning'] = "School created but directory initialization failed.";
} else {
    $_SESSION['success'] = "School created successfully with {$initResult['directories_created']} directories.";
}
?>
```

---

### **B. Admin Dashboard - Storage Widget**

**Add to `admin/dashboard.php`:**

```php
<?php
// Show storage usage widget
require_once '../includes/tenant_directory_manager.php';
$dirManager = new TenantDirectoryManager($conn);

$storage = $dirManager->calculateTenantStorageUsage($_SESSION['academy_reference']);

if ($storage['success']) {
    ?>
    <div class="storage-widget">
        <h3>Storage Usage</h3>
        <div class="progress-bar">
            <div class="progress" style="width: <?= $storage['percentage'] ?>%"></div>
        </div>
        <p>
            Used: <?= $storage['total_formatted'] ?> of <?= $storage['quota_formatted'] ?>
            (<?= $storage['percentage'] ?>%)
        </p>
        <?php if ($storage['percentage'] > 80): ?>
            <p class="warning">⚠️ Storage quota is over 80%!</p>
        <?php endif; ?>
    </div>
    <?php
}
?>
```

---

### **C. Super Admin - School Management**

**Add to `super_admin/schools/list.php`:**

```php
<?php
// Add storage column to schools list
require_once '../../includes/tenant_directory_manager.php';
$dirManager = new TenantDirectoryManager($conn);

// In your foreach loop for schools:
foreach ($schools as $school) {
    $storage = $dirManager->calculateTenantStorageUsage($school['academy_reference']);
    
    echo "<tr>";
    echo "<td>{$school['name']}</td>";
    echo "<td>{$school['academy_reference']}</td>";
    
    // Storage column
    if ($storage['success']) {
        $color = $storage['percentage'] > 80 ? 'red' : ($storage['percentage'] > 50 ? 'orange' : 'green');
        echo "<td style='color: {$color}'>";
        echo "{$storage['total_formatted']} / {$storage['quota_formatted']} ({$storage['percentage']}%)";
        echo "</td>";
    } else {
        echo "<td>N/A</td>";
    }
    
    echo "<td><!-- actions --></td>";
    echo "</tr>";
}
?>
```

---

### **D. File Upload Integration**

**When uploading files, use the path methods:**

```php
<?php
require_once 'includes/tenant_directory_manager.php';
$dirManager = new TenantDirectoryManager($conn);

// Get upload path
$uploadPath = $dirManager->getTenantUploadPath(
    $_SESSION['academy_reference'],
    'documents'  // or 'profile_photos', 'payment_receipts', 'reports'
);

// Upload file
if (isset($_FILES['document'])) {
    $filename = basename($_FILES['document']['name']);
    $targetFile = $uploadPath . '/' . $filename;
    
    if (move_uploaded_file($_FILES['document']['tmp_name'], $targetFile)) {
        echo "✅ File uploaded successfully";
    } else {
        echo "❌ Upload failed";
    }
}
?>
```

---

### **E. Database Backup Integration**

**In your backup script (`cron/backup_database.php`):**

```php
<?php
require_once '../includes/tenant_directory_manager.php';
$dirManager = new TenantDirectoryManager($conn);

// Get all tenants
$stmt = $conn->query("SELECT academy_reference FROM tenants");
$tenants = $stmt->fetchAll(PDO::FETCH_COLUMN);

foreach ($tenants as $tenant_id) {
    // Get backup path
    $backupPath = $dirManager->getTenantBackupPath($tenant_id, 'database');
    
    // Generate backup filename
    $backupFile = $backupPath . '/backup_' . date('Y-m-d_His') . '.sql';
    
    // Create backup
    $command = "mysqldump -u " . DB_USER . " -p" . DB_PASS . " " . DB_NAME . " > " . $backupFile;
    exec($command);
    
    echo "✅ Backup created for {$tenant_id}\n";
}
?>
```

---

### **F. Logging Integration**

**For custom logging:**

```php
<?php
require_once 'includes/tenant_directory_manager.php';
$dirManager = new TenantDirectoryManager($conn);

// Get log path
$logPath = $dirManager->getTenantLogPath($_SESSION['academy_reference'], 'payments');

// Write log
$logFile = $logPath . '/payment_' . date('Y-m-d') . '.log';
$logEntry = date('Y-m-d H:i:s') . " - Payment processed: {$payment_id}\n";

file_put_contents($logFile, $logEntry, FILE_APPEND);
?>
```

---

## 📊 **Step 4: Create Monitoring Scripts**

### **Storage Monitor (Daily Cron)**

**Create:** `cron/monitor_storage.php`

```php
<?php
require_once '../config.php';
require_once '../includes/tenant_directory_manager.php';

$conn = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
$dirManager = new TenantDirectoryManager($conn);

// Get all tenants
$stmt = $conn->query("SELECT academy_reference, name, contact_email FROM tenants");
$tenants = $stmt->fetchAll(PDO::FETCH_ASSOC);

foreach ($tenants as $tenant) {
    $storage = $dirManager->calculateTenantStorageUsage($tenant['academy_reference']);
    
    if ($storage['success'] && $storage['percentage'] > 80) {
        // Send warning email
        $to = $tenant['contact_email'];
        $subject = "Storage Quota Warning - " . $tenant['name'];
        $message = "Your storage usage is at {$storage['percentage']}%.\n\n";
        $message .= "Used: {$storage['total_formatted']}\n";
        $message .= "Quota: {$storage['quota_formatted']}\n";
        $message .= "Available: {$storage['available_formatted']}\n\n";
        $message .= "Please clean up old files or contact support to increase your quota.";
        
        mail($to, $subject, $message);
        
        echo "⚠️ Warning sent to {$tenant['name']}\n";
    }
}
?>
```

**Add to crontab:**
```bash
0 8 * * * /usr/bin/php /path/to/cron/monitor_storage.php
```

---

### **Health Check (Weekly)**

**Create:** `cron/health_check.php`

```php
<?php
require_once '../config.php';
require_once '../includes/tenant_directory_manager.php';

$conn = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
$dirManager = new TenantDirectoryManager($conn);

$stmt = $conn->query("SELECT academy_reference, name FROM tenants");
$tenants = $stmt->fetchAll(PDO::FETCH_ASSOC);

$report = "=== TENANT HEALTH CHECK REPORT ===\n";
$report .= "Date: " . date('Y-m-d H:i:s') . "\n\n";

foreach ($tenants as $tenant) {
    $verification = $dirManager->verifyTenantDirectories($tenant['academy_reference']);
    $storage = $dirManager->calculateTenantStorageUsage($tenant['academy_reference']);
    
    $status = $verification['success'] ? '✅' : '❌';
    $report .= "{$status} {$tenant['name']}\n";
    $report .= "  Directories: {$verification['directories_ok']} OK";
    
    if ($verification['directories_missing'] > 0) {
        $report .= ", {$verification['directories_missing']} missing";
    }
    
    $report .= "\n";
    $report .= "  Storage: {$storage['total_formatted']} ({$storage['percentage']}%)\n\n";
}

// Email report to admin
mail('admin@melanegroup.com', 'Weekly Tenant Health Report', $report);

echo $report;
?>
```

---

## ⚙️ **Step 5: Update Existing Code**

### **Replace Hardcoded Paths**

**Before:**
```php
$uploadDir = 'uploads/documents/';
```

**After:**
```php
$dirManager = new TenantDirectoryManager($conn);
$uploadDir = $dirManager->getTenantUploadPath($_SESSION['academy_reference'], 'documents');
```

---

### **Add Storage Checks**

**Before:**
```php
move_uploaded_file($_FILES['file']['tmp_name'], $targetPath);
```

**After:**
```php
// Check storage quota first
$storage = $dirManager->calculateTenantStorageUsage($_SESSION['academy_reference']);

if ($storage['percentage'] >= 95) {
    die("Storage quota exceeded. Please contact support.");
}

move_uploaded_file($_FILES['file']['tmp_name'], $targetPath);
```

---

## 🎯 **Step 6: Configuration**

### **Set Default Quotas**

In your school creation form, add quota selection:

```html
<select name="storage_quota">
    <option value="1073741824">1 GB (Basic)</option>
    <option value="5368709120">5 GB (Standard)</option>
    <option value="10737418240">10 GB (Premium)</option>
    <option value="21474836480">20 GB (Enterprise)</option>
</select>
```

---

### **Update Database Schema (Optional)**

Add storage quota to tenants table:

```sql
ALTER TABLE tenants 
ADD COLUMN storage_quota BIGINT DEFAULT 1073741824 AFTER contact_phone;
```

Then update initialization to use database value:

```php
$stmt = $conn->prepare("SELECT storage_quota FROM tenants WHERE academy_reference = ?");
$stmt->execute([$academy_reference]);
$quota = $stmt->fetchColumn();

$initResult = $dirManager->initializeTenant($academy_reference, $school_name, [
    'storage_quota' => $quota ?: 1073741824
]);
```

---

## ✅ **Integration Checklist**

- [ ] Files uploaded to correct locations
- [ ] Test suite run successfully
- [ ] Integrated into school creation
- [ ] Added storage widget to admin dashboard
- [ ] Added storage column to super admin list
- [ ] Updated file upload to use path methods
- [ ] Integrated with backup system
- [ ] Created monitoring scripts
- [ ] Set up cron jobs
- [ ] Updated hardcoded paths
- [ ] Added storage quota checks
- [ ] Tested with real school creation
- [ ] Verified directories created correctly
- [ ] Deleted test file from production

---

## 🚀 **You're Done!**

Your Tenant Directory Manager is now fully integrated!

**Next:** Test by creating a new school and verifying:
1. ✅ Directories are created
2. ✅ Security files are in place
3. ✅ Configuration file exists
4. ✅ Storage tracking works
5. ✅ File uploads work
6. ✅ Backups save correctly

---

*File: TENANT_DIRECTORY_MANAGER_INTEGRATION.md*  
*Purpose: Integration guide for existing systems*  
*Last Updated: 2025*

