# 🔐 Tenant File Upload System - Complete Guide

## 🎯 **Overview**

The `TenantFileUpload` class provides **secure, validated file uploads** for your multi-tenant education platform with comprehensive security checks, metadata tracking, and database integration.

---

## ✨ **Features**

- ✅ **Category-Specific Upload Methods** - Documents, photos, receipts, reports
- ✅ **MIME Type Validation** - Not just extensions
- ✅ **Magic Byte Checking** - Verify actual file format
- ✅ **File Size Limits** - Per category limits
- ✅ **Unique Filenames** - Timestamp + random + sanitized
- ✅ **Database Tracking** - All uploads logged
- ✅ **Metadata Support** - Tags, descriptions, relationships
- ✅ **Soft Delete** - Never lose file records
- ✅ **Security First** - Multiple validation layers
- ✅ **Virus Scan Ready** - Integration hooks provided

---

## 🚀 **Quick Start**

### **Initialize**

```php
<?php
require_once 'config.php';
require_once 'includes/tenant_directory_manager.php';
require_once 'includes/tenant_file_upload.php';

// Database connection
$conn = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);

// Initialize upload handler
$dirManager = new TenantDirectoryManager($conn);
$uploader = new TenantFileUpload($conn, $dirManager);
?>
```

---

## 📋 **Core Methods**

### **1. uploadDocument()**

Upload general documents (PDF, Word, Excel).

```php
$result = $uploader->uploadDocument(
    $tenant_id,     // Tenant identifier
    $_FILES['file'], // File from form
    [               // Optional metadata
        'description' => 'Course syllabus',
        'tags' => 'syllabus,mathematics,2025',
        'related_id' => 123,
        'related_type' => 'course'
    ]
);

// Returns:
[
    'success' => true,
    'file_id' => 456,
    'filename' => '1706600000_a1b2c3d4_syllabus.pdf',
    'original_name' => 'math_syllabus.pdf',
    'path' => '/full/path/to/file',
    'relative_path' => 'school_tenant/uploads/documents/...',
    'size' => 245678,
    'url' => '/serve_file.php?tenant=...&type=documents&file=...'
]
```

**Allowed Types:**
- PDF (10MB max)
- DOC, DOCX (10MB max)
- XLS, XLSX (10MB max)

---

### **2. uploadProfilePhoto()**

Upload student/staff profile photos with validation.

```php
$result = $uploader->uploadProfilePhoto(
    $tenant_id,
    $student_id,
    $_FILES['profile_photo']
);

// Returns:
[
    'success' => true,
    'file_id' => 789,
    'filename' => 'student_123_1706600000_a1b2c3d4.jpg',
    'path' => '/full/path/to/photo',
    'relative_path' => 'school_tenant/uploads/profile_photos/...',
    'url' => '/serve_file.php?tenant=...&type=profile_photos&file=...'
]
```

**Features:**
- ✅ Auto-deletes old profile photo
- ✅ Validates image dimensions (100x100 min, 4000x4000 max)
- ✅ Checks image integrity
- ✅ Updates student record automatically
- ✅ Optimizes image (optional)

**Allowed Types:**
- JPG, JPEG (2MB max)
- PNG (2MB max)

---

### **3. uploadPaymentReceipt()**

Upload payment receipts with payment reference.

```php
$result = $uploader->uploadPaymentReceipt(
    $tenant_id,
    $payment_id,
    $_FILES['receipt']
);

// Returns:
[
    'success' => true,
    'file_id' => 321,
    'filename' => 'payment_10001_1706600000_a1b2c3d4.pdf',
    'url' => '/serve_file.php?...'
]
```

**Features:**
- ✅ Links receipt to payment record
- ✅ Updates payment table with receipt path
- ✅ Supports PDF and images

**Allowed Types:**
- PDF (5MB max)
- JPG, PNG (5MB max)

---

### **4. uploadReport()**

Upload generated reports.

```php
$result = $uploader->uploadReport(
    $tenant_id,
    'attendance',  // Report type
    $_FILES['report'],
    [
        'description' => 'January 2025 attendance report',
        'tags' => 'attendance,january,2025'
    ]
);

// Returns:
[
    'success' => true,
    'file_id' => 654,
    'filename' => 'attendance_20250130_1706600000_a1b2c3d4.pdf',
    'url' => '/serve_file.php?...'
]
```

**Allowed Types:**
- PDF (20MB max)
- XLSX (20MB max)

---

### **5. validateFile()**

Validate file before upload.

```php
$validation = $uploader->validateFile($_FILES['file'], 'document');

if ($validation['valid']) {
    // File is valid, proceed with upload
    echo "MIME type: " . $validation['mime_type'];
    echo "Extension: " . $validation['extension'];
} else {
    // Show error
    echo "Error: " . $validation['error'];
}

// Returns:
[
    'valid' => true,
    'mime_type' => 'application/pdf',
    'extension' => 'pdf'
]
```

**Validation Checks:**
1. ✅ File was uploaded
2. ✅ No upload errors
3. ✅ File size within limit
4. ✅ Extension allowed
5. ✅ MIME type allowed
6. ✅ Magic bytes match

---

### **6. sanitizeFilename()**

Clean unsafe filenames.

```php
$safe = $uploader->sanitizeFilename('my document (final) [copy].pdf');
// Returns: my_document_final_copy.pdf

$safe = $uploader->sanitizeFilename('student @#$% photo.jpg');
// Returns: student_photo.jpg

$safe = $uploader->sanitizeFilename('../../../etc/passwd');
// Returns: etc_passwd
```

**What It Does:**
- Removes special characters
- Replaces spaces with underscores
- Removes path components
- Limits to 100 characters

---

### **7. generateUniqueFilename()**

Generate unique, collision-resistant filename.

```php
$unique = $uploader->generateUniqueFilename('report.pdf');
// Returns: 1706600000_a1b2c3d4e5f6g7h8_report.pdf

// Format: {timestamp}_{16-char-random}_{sanitized-original}.{ext}
```

**Components:**
- Timestamp (10 digits)
- Random hex (16 characters)
- Sanitized original name
- Original extension

**Benefits:**
- ✅ Guaranteed unique
- ✅ Chronologically sortable
- ✅ Human-readable
- ✅ Preserves original name context

---

### **8. deleteFile()**

Soft delete file with logging.

```php
$result = $uploader->deleteFile($tenant_id, 'path/to/file.pdf');

// Returns:
[
    'success' => true,
    'message' => 'File deleted successfully'
]
```

**What It Does:**
- ✅ Soft delete (marks as deleted in DB)
- ✅ Preserves file metadata
- ✅ Logs deletion
- ✅ Records who deleted it
- ✅ Physical file preserved (can be cleaned up later)

---

## 🔒 **Security Features**

### **1. MIME Type Validation**

Not fooled by renamed extensions:

```php
// User renames virus.exe to virus.pdf
// Extension check: ✅ PASS (it's .pdf)
// MIME check: ❌ FAIL (MIME is application/x-msdownload)
// Result: Upload blocked ✅
```

---

### **2. Magic Byte Validation**

Checks actual file content:

```php
// File magic bytes for PDF: %PDF (hex: 25504446)
// File magic bytes for JPEG: FFD8FFE0

// Fake PDF (renamed .txt file): ❌ Blocked
// Real PDF: ✅ Allowed
```

---

### **3. File Size Limits**

Per-category limits enforced:

```php
// Document (10MB): 15MB file → ❌ Blocked
// Photo (2MB): 3MB image → ❌ Blocked
// Receipt (5MB): 4MB PDF → ✅ Allowed
// Report (20MB): 25MB file → ❌ Blocked
```

---

### **4. Filename Sanitization**

Prevents attacks:

```php
// Path traversal: ../../../etc/passwd → etc_passwd ✅
// SQL injection: file'; DROP TABLE-- → file_DROP_TABLE ✅
// XSS: <script>alert()</script> → script_alert ✅
```

---

### **5. Random Filenames**

Unpredictable file paths:

```php
// Original: report.pdf
// Stored: 1706600000_a1b2c3d4e5f6g7h8_report.pdf

// Attacker can't guess: ✅
// No overwrites: ✅
// Chronologically ordered: ✅
```

---

## 📊 **File Limits**

| Category | Extensions | MIME Types | Max Size |
|----------|-----------|------------|----------|
| **Documents** | pdf, doc, docx, xls, xlsx | application/pdf, application/msword, etc. | 10 MB |
| **Photos** | jpg, jpeg, png | image/jpeg, image/png | 2 MB |
| **Receipts** | pdf, jpg, jpeg, png | application/pdf, image/jpeg, image/png | 5 MB |
| **Reports** | pdf, xlsx | application/pdf, application/vnd.openxmlformats... | 20 MB |

---

## 🎯 **Practical Examples**

### **Example 1: Student Document Upload**

```php
<?php
session_start();
require_once 'includes/tenant_file_upload.php';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['document'])) {
    $uploader = new TenantFileUpload($conn, $dirManager);
    
    $result = $uploader->uploadDocument(
        $_SESSION['academy_reference'],
        $_FILES['document'],
        [
            'description' => $_POST['description'],
            'related_id' => $_POST['student_id'],
            'related_type' => 'student'
        ]
    );
    
    if ($result['success']) {
        $_SESSION['success'] = "Document uploaded successfully!";
        header('Location: dashboard.php');
    } else {
        $_SESSION['error'] = $result['error'];
    }
}
?>

<form method="post" enctype="multipart/form-data">
    <input type="file" name="document" accept=".pdf,.doc,.docx" required>
    <input type="text" name="description" placeholder="Document description">
    <input type="hidden" name="student_id" value="<?= $student_id ?>">
    <button type="submit">Upload</button>
</form>
```

---

### **Example 2: Profile Photo Upload**

```php
<?php
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['photo'])) {
    $uploader = new TenantFileUpload($conn, $dirManager);
    
    $result = $uploader->uploadProfilePhoto(
        $_SESSION['academy_reference'],
        $_SESSION['student_id'],
        $_FILES['photo']
    );
    
    if ($result['success']) {
        echo json_encode([
            'success' => true,
            'photo_url' => $result['url'],
            'message' => 'Photo uploaded successfully!'
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'error' => $result['error']
        ]);
    }
    exit;
}
?>

<form method="post" enctype="multipart/form-data" id="photo-form">
    <div class="photo-upload">
        <img id="preview" src="default-avatar.png" alt="Profile Photo">
        <input type="file" name="photo" accept="image/jpeg,image/png" required>
    </div>
    <button type="submit">Upload Photo</button>
</form>

<script>
// Preview before upload
document.querySelector('input[type="file"]').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('preview').src = e.target.result;
        };
        reader.readAsDataURL(file);
    }
});

// AJAX upload
document.getElementById('photo-form').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    const response = await fetch('upload_photo.php', {
        method: 'POST',
        body: formData
    });
    
    const result = await response.json();
    
    if (result.success) {
        alert('Photo uploaded successfully!');
        document.getElementById('preview').src = result.photo_url;
    } else {
        alert('Upload failed: ' + result.error);
    }
});
</script>
```

---

### **Example 3: Payment Receipt Upload**

```php
<?php
// After payment confirmation
if ($payment_successful) {
    if (isset($_FILES['receipt'])) {
        $uploader = new TenantFileUpload($conn, $dirManager);
        
        $result = $uploader->uploadPaymentReceipt(
            $_SESSION['academy_reference'],
            $payment_id,
            $_FILES['receipt']
        );
        
        if ($result['success']) {
            // Receipt uploaded and linked to payment
            echo "Receipt uploaded: " . $result['url'];
        }
    }
}
?>
```

---

### **Example 4: Report Generation & Upload**

```php
<?php
// Generate report, then upload
function generateAndUploadAttendanceReport($tenant_id, $month, $year) {
    global $uploader;
    
    // 1. Generate report (your existing code)
    $reportData = generateAttendanceData($month, $year);
    $pdfContent = createPDFReport($reportData);
    
    // 2. Save to temporary file
    $tempFile = sys_get_temp_dir() . '/temp_report_' . time() . '.pdf';
    file_put_contents($tempFile, $pdfContent);
    
    // 3. Create file array for upload
    $file = [
        'name' => "attendance_report_{$year}_{$month}.pdf",
        'type' => 'application/pdf',
        'tmp_name' => $tempFile,
        'error' => UPLOAD_ERR_OK,
        'size' => filesize($tempFile)
    ];
    
    // 4. Upload
    $result = $uploader->uploadReport(
        $tenant_id,
        'attendance',
        $file,
        [
            'description' => "Attendance report for {$month}/{$year}",
            'tags' => "attendance,report,{$month},{$year}"
        ]
    );
    
    // 5. Cleanup temp file
    unlink($tempFile);
    
    return $result;
}
?>
```

---

## 🔍 **Validation System**

### **Multi-Layer Validation**

```
Layer 1: Upload Error Check
         └─> Check $_FILES['error'] === UPLOAD_ERR_OK

Layer 2: File Size Check
         └─> Compare with category limit

Layer 3: Extension Check
         └─> Whitelist of allowed extensions

Layer 4: MIME Type Check
         └─> Use finfo to get actual MIME type

Layer 5: Magic Byte Check
         └─> Read file header, verify format

Result: ✅ File is genuinely what it claims to be
```

---

### **Validation Example**

```php
$validation = $uploader->validateFile($_FILES['file'], 'document');

if ($validation['valid']) {
    echo "✅ Valid!";
    echo "MIME: " . $validation['mime_type'];
    echo "Ext: " . $validation['extension'];
} else {
    echo "❌ Invalid: " . $validation['error'];
}
```

---

## 📁 **File Storage**

### **Directory Structure**

```
tenants/school_{tenant_id}/
├── uploads/
│   ├── documents/
│   │   └── 1706600000_a1b2c3d4_report.pdf
│   ├── profile_photos/
│   │   └── student_123_1706600000_a1b2c3d4.jpg
│   ├── payment_receipts/
│   │   └── payment_10001_1706600000_a1b2c3d4.pdf
│   └── reports/
│       └── attendance_20250130_1706600000_a1b2c3d4.pdf
```

---

### **Filename Format**

```
{timestamp}_{random}_{sanitized_original}.{extension}

Example:
1706600000_a1b2c3d4e5f6g7h8_student_report.pdf

Where:
- 1706600000 = Unix timestamp (sortable)
- a1b2c3d4e5f6g7h8 = 16-char random hex (unique)
- student_report = Sanitized original name
- .pdf = Original extension
```

---

## 🗄️ **Database Tracking**

Every upload is tracked in `tenant_files` table:

```sql
INSERT INTO tenant_files (
    tenant_id,           -- 'soshigh_demo'
    file_category,       -- 'document'
    original_filename,   -- 'student_report.pdf'
    stored_filename,     -- '1706600000_a1b2c3d4_student_report.pdf'
    file_path,           -- 'school_soshigh/uploads/documents/...'
    file_size,           -- 245678 (bytes)
    mime_type,           -- 'application/pdf'
    file_hash,           -- SHA-256 hash
    uploaded_by,         -- User ID from session
    description,         -- Optional description
    tags,                -- Optional tags
    related_id,          -- Student ID, Payment ID, etc.
    related_type,        -- 'student', 'payment', etc.
    upload_date          -- CURRENT_TIMESTAMP
) VALUES (...);
```

---

## 🛡️ **Security Best Practices**

### **1. Never Trust User Input**

```php
// ❌ BAD - Using original filename
$path = 'uploads/' . $_FILES['file']['name'];
move_uploaded_file($_FILES['file']['tmp_name'], $path);

// ✅ GOOD - Using sanitized unique filename
$unique = $uploader->generateUniqueFilename($_FILES['file']['name']);
$path = $uploadPath . '/' . $unique;
move_uploaded_file($_FILES['file']['tmp_name'], $path);
```

---

### **2. Validate MIME Types**

```php
// ❌ BAD - Only checking extension
$ext = pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION);
if ($ext === 'pdf') { /* upload */ }

// ✅ GOOD - Checking actual MIME type
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mime = finfo_file($finfo, $_FILES['file']['tmp_name']);
if ($mime === 'application/pdf') { /* upload */ }
```

---

### **3. Check Magic Bytes**

```php
// ✅ Reads first 8 bytes of file
// ✅ Compares with known file signatures
// ✅ Blocks renamed executables
// ✅ Ensures file integrity
```

---

### **4. Set Proper Permissions**

```php
// After upload
chmod($targetPath, 0644);  // rw-r--r--
// Owner: read+write
// Group: read only
// Others: read only
```

---

### **5. Store Outside Web Root (Optional)**

```php
// Instead of: public_html/uploads/
// Use: /home/user/tenant_files/

// Files served through PHP script
// Direct access: ✅ Blocked
// Authenticated access only: ✅ Enforced
```

---

## 🧪 **Testing**

### **Test Each Upload Type**

```php
// Test document
$doc_result = $uploader->uploadDocument($tenant_id, $_FILES['doc'], []);
assert($doc_result['success'] === true);

// Test photo
$photo_result = $uploader->uploadProfilePhoto($tenant_id, 123, $_FILES['photo']);
assert($photo_result['success'] === true);

// Test receipt
$receipt_result = $uploader->uploadPaymentReceipt($tenant_id, 456, $_FILES['receipt']);
assert($receipt_result['success'] === true);

// Test report
$report_result = $uploader->uploadReport($tenant_id, 'test', $_FILES['report']);
assert($report_result['success'] === true);
```

---

### **Test Validation**

```php
// Test invalid file type
$invalid_file = ['tmp_name' => 'virus.exe', 'size' => 1000, ...];
$result = $uploader->validateFile($invalid_file, 'document');
assert($result['valid'] === false);

// Test oversized file
$large_file = ['size' => 50000000, ...];  // 50MB
$result = $uploader->validateFile($large_file, 'document');
assert($result['valid'] === false);
```

---

## ⚙️ **Configuration**

### **Modify File Limits**

Edit `includes/tenant_file_upload.php`:

```php
private $fileLimits = [
    'document' => 20971520,   // 20MB (increased from 10MB)
    'photo' => 5242880,       // 5MB (increased from 2MB)
    'receipt' => 10485760,    // 10MB (increased from 5MB)
    'report' => 52428800      // 50MB (increased from 20MB)
];
```

---

### **Add Allowed File Types**

```php
private $allowedExtensions = [
    'document' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx'],  // Added PPT
    'photo' => ['jpg', 'jpeg', 'png', 'gif', 'webp'],  // Added GIF, WEBP
];

private $allowedMimeTypes = [
    'document' => [
        'application/pdf',
        'application/msword',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'application/vnd.ms-excel',
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        'application/vnd.ms-powerpoint',  // Added
        'application/vnd.openxmlformats-officedocument.presentationml.presentation'  // Added
    ],
];
```

---

## 🔌 **Integration with Virus Scanning**

### **ClamAV Integration (Ready)**

```php
// In validateFile() method, add after magic bytes check:

// Virus scan (if ClamAV is available)
if (function_exists('clamscan')) {
    $scanResult = clamscan($file['tmp_name']);
    if ($scanResult === 'FOUND') {
        return ['valid' => false, 'error' => 'Virus detected in file'];
    }
}

// Or use ClamAV command line
$output = shell_exec("clamscan " . escapeshellarg($file['tmp_name']));
if (strpos($output, 'FOUND') !== false) {
    return ['valid' => false, 'error' => 'Virus detected'];
}
```

---

## 📚 **Complete Usage Guide**

See `TENANT_FILE_UPLOAD_EXAMPLES.php` for:
- 11 practical examples
- AJAX upload
- Drag & drop
- Batch uploads
- Complete handlers

---

## ✅ **Summary**

The `TenantFileUpload` class provides:

- ✅ **Secure uploads** with multiple validation layers
- ✅ **Category-specific** methods and limits
- ✅ **Database tracking** for all files
- ✅ **Metadata support** for searchability
- ✅ **Soft delete** for data preservation
- ✅ **Logging** for audit trails
- ✅ **Unique filenames** to prevent collisions
- ✅ **Magic byte validation** for security
- ✅ **Production-ready** code

**Perfect for multi-tenant education platforms!** 🎓

---

*File: TENANT_FILE_UPLOAD_GUIDE.md*  
*Last Updated: 2025*  
*Version: 1.0*

