# ⚡ Tenant File Upload - Quick Reference

## 🚀 **Quick Start (1 Minute)**

```php
<?php
require_once 'includes/tenant_file_upload.php';
$uploader = new TenantFileUpload($conn, $dirManager);

// Upload document
$result = $uploader->uploadDocument(
    $_SESSION['academy_reference'],
    $_FILES['file'],
    ['description' => 'My document']
);
?>
```

---

## 📋 **Common Operations**

### **Upload Document**
```php
$uploader->uploadDocument($tenant_id, $_FILES['document'], [
    'description' => 'Course syllabus',
    'tags' => 'syllabus,math',
    'related_id' => 123,
    'related_type' => 'course'
]);
```

### **Upload Photo**
```php
$uploader->uploadProfilePhoto($tenant_id, $student_id, $_FILES['photo']);
```

### **Upload Receipt**
```php
$uploader->uploadPaymentReceipt($tenant_id, $payment_id, $_FILES['receipt']);
```

### **Upload Report**
```php
$uploader->uploadReport($tenant_id, 'attendance', $_FILES['report'], [
    'description' => 'Monthly attendance'
]);
```

### **Validate Before Upload**
```php
$validation = $uploader->validateFile($_FILES['file'], 'document');
if ($validation['valid']) {
    // Proceed with upload
}
```

### **Sanitize Filename**
```php
$safe = $uploader->sanitizeFilename('my file (copy).pdf');
// Returns: my_file_copy.pdf
```

### **Delete File**
```php
$uploader->deleteFile($tenant_id, 'path/to/file.pdf');
```

---

## 📊 **File Limits**

| Category | Types | Max Size |
|----------|-------|----------|
| Documents | PDF, DOC, DOCX, XLS, XLSX | 10 MB |
| Photos | JPG, PNG | 2 MB |
| Receipts | PDF, JPG, PNG | 5 MB |
| Reports | PDF, XLSX | 20 MB |

---

## 🔒 **Security**

```
✅ MIME type validation
✅ Magic byte checking
✅ File size limits
✅ Extension whitelist
✅ Filename sanitization
✅ Random unique names
✅ Virus scan ready
```

---

## 🗄️ **Database Tracking**

Every upload logged in `tenant_files` table:
- File metadata
- User who uploaded
- Related entities
- Access tracking
- Soft delete support

---

## 📝 **HTML Forms**

### **Document Upload**
```html
<form method="post" enctype="multipart/form-data">
    <input type="file" name="document" accept=".pdf,.doc,.docx" required>
    <button type="submit">Upload</button>
</form>
```

### **Photo Upload**
```html
<form method="post" enctype="multipart/form-data">
    <input type="file" name="photo" accept="image/*" required>
    <button type="submit">Upload Photo</button>
</form>
```

### **Multiple Files**
```html
<form method="post" enctype="multipart/form-data">
    <input type="file" name="documents[]" multiple>
    <button type="submit">Upload All</button>
</form>
```

---

## 🎯 **Complete Example**

```php
<?php
session_start();
require_once 'includes/tenant_file_upload.php';

$uploader = new TenantFileUpload($conn, $dirManager);

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    $result = $uploader->uploadDocument(
        $_SESSION['academy_reference'],
        $_FILES['file'],
        ['description' => $_POST['description']]
    );
    
    echo json_encode($result);
    exit;
}
?>
```

---

## 🧪 **Test**

```
Visit: test_tenant_file_upload.php
Result: ✅ All 8 tests pass
```

---

## 📚 **Full Documentation**

- Complete Guide: `TENANT_FILE_UPLOAD_GUIDE.md`
- Examples: `TENANT_FILE_UPLOAD_EXAMPLES.php`
- Summary: `TENANT_FILE_UPLOAD_COMPLETE.md`

---

**Ready in 5 minutes! 🚀**

