<?php
/**
 * Configure Subjects for Each Grade
 * Allows admin to set which subjects are available for each grade
 */

require_once '../includes/functions.php';
require_once '../includes/school_context.php';
require_once '../includes/curriculum_catalog.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();

if (!$academy_reference) {
    die("Error: No school selected. Please select a school first.");
}

// Handle POST request to update subject mappings
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'update_mappings') {
        $grade_id = intval($_POST['grade_id']);
        $subject_ids = $_POST['subject_ids'] ?? [];
        
        try {
            // Start transaction
            $pdo->beginTransaction();
            
            // Clear existing mappings for this grade
            $stmt = $pdo->prepare("DELETE FROM grade_subject WHERE grade_id = ?");
            $stmt->execute([$grade_id]);
            
            // Insert new mappings
            if (!empty($subject_ids)) {
                $stmt = $pdo->prepare("INSERT INTO grade_subject (grade_id, subject_id) VALUES (?, ?)");
                foreach ($subject_ids as $subject_id) {
                    $stmt->execute([$grade_id, intval($subject_id)]);
                }
            }
            
            $pdo->commit();
            
            $success_message = "Subject mappings updated successfully for grade ID $grade_id!";
        } catch (Exception $e) {
            $pdo->rollBack();
            $error_message = "Error updating mappings: " . $e->getMessage();
        }
    }
}

// Get only grades that this school has enabled in school_grades table
$stmt = $pdo->prepare("
    SELECT g.*
    FROM grades g
    INNER JOIN school_grades sg ON g.id = sg.grade_id
    WHERE sg.academy_reference = ? AND sg.is_active = 1
    ORDER BY sg.display_order, g.name
");
$stmt->execute([$academy_reference]);
$grades = $stmt->fetchAll();

// If no grades configured yet, show all grades (backward compatibility)
if (empty($grades)) {
    $grades = getGrades();
    $no_grades_configured = true;
} else {
    $no_grades_configured = false;
}

$curriculumMap = getCurriculumSubjectMap();
$maxCodeLength = getSubjectCodeMaxLength($pdo);
$subjectLookupStmt = $pdo->prepare('SELECT * FROM subjects WHERE code = ?');
$fallbackStmt = $pdo->prepare('
    SELECT s.*
    FROM subjects s
    INNER JOIN grade_subject gs ON gs.subject_id = s.id
    WHERE gs.grade_id = ?
    ORDER BY s.name
');

$grade_subject_options = [];

foreach ($grades as $grade) {
    $gradeSubjects = [];

    if (isset($curriculumMap[$grade['name']])) {
        foreach ($curriculumMap[$grade['name']] as $definition) {
            $code = buildCurriculumSubjectCode($grade['name'], $definition['name'], $definition['code'] ?? null, $maxCodeLength);
            $subjectLookupStmt->execute([$code]);
            $subject = $subjectLookupStmt->fetch();
            $subjectLookupStmt->closeCursor();

            if ($subject) {
                $gradeSubjects[] = $subject;
            }
        }
    }

    if (!$gradeSubjects) {
        // Fallback: use any subjects currently mapped to this grade
        $fallbackStmt->execute([$grade['id']]);
        $gradeSubjects = $fallbackStmt->fetchAll();
    }

    $grade_subject_options[$grade['id']] = $gradeSubjects;
}

// Get current mappings
$current_mappings = [];
foreach ($grades as $grade) {
    $stmt = $pdo->prepare("
        SELECT subject_id 
        FROM grade_subject 
        WHERE grade_id = ?
    ");
    $stmt->execute([$grade['id']]);
    $current_mappings[$grade['id']] = $stmt->fetchAll(PDO::FETCH_COLUMN);
}

include 'partials/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h2 class="mb-1">
                        <i class="bi bi-gear-fill me-2"></i>
                        Configure Subjects for Grades
                    </h2>
                    <p class="text-muted mb-0">Set which subjects are available for each grade level</p>
                </div>
                <a href="dashboard.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-2"></i>Back to Dashboard
                </a>
            </div>

            <?php if (isset($success_message)): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <i class="bi bi-check-circle-fill me-2"></i>
                <?php echo htmlspecialchars($success_message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <?php if (isset($error_message)): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                <?php echo htmlspecialchars($error_message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <!-- Grade Selection Notice -->
            <?php if ($no_grades_configured): ?>
            <div class="alert alert-warning shadow-sm mb-4">
                <div class="row align-items-center">
                    <div class="col-md-9">
                        <h5 class="alert-heading mb-2">
                            <i class="bi bi-exclamation-triangle me-2"></i>No Grades Configured
                        </h5>
                        <p class="mb-0">
                            You haven't selected which grades your school offers yet. 
                            Click "Manage Grades" to choose which grade levels you want to offer 
                            (Grade 4-7 for primary, Form 1-5 for secondary, or both).
                        </p>
                    </div>
                    <div class="col-md-3 text-end">
                        <a href="manage_grades.php" class="btn btn-warning">
                            <i class="bi bi-sliders me-2"></i>Manage Grades
                        </a>
                    </div>
                </div>
            </div>
            <?php else: ?>
            <div class="alert alert-info shadow-sm mb-4">
                <div class="row align-items-center">
                    <div class="col-md-9">
                        <strong><i class="bi bi-info-circle me-2"></i>Active Grades:</strong>
                        <?= count($grades) ?> grade(s) enabled for your school.
                        Configure subjects for each grade below.
                    </div>
                    <div class="col-md-3 text-end">
                        <a href="manage_grades.php" class="btn btn-sm btn-outline-info">
                            <i class="bi bi-sliders me-2"></i>Change Grades
                        </a>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Subject Configuration Cards -->
            <div class="row g-4">
                <?php foreach ($grades as $grade): 
                    $is_primary = preg_match('/^Grade [4-7]$/i', $grade['name']);
                    $card_color = $is_primary ? 'success' : 'primary';
                    $icon = $is_primary ? 'book' : 'mortarboard-fill';
                ?>
                <div class="col-md-6 col-lg-4">
                    <div class="card h-100 shadow-sm">
                        <div class="card-header bg-<?= $card_color ?> text-white">
                            <h5 class="mb-0">
                                <i class="bi bi-<?= $icon ?> me-2"></i>
                                <?php echo htmlspecialchars($grade['name']); ?>
                                <?php if ($is_primary): ?>
                                <span class="badge bg-light text-success float-end">Primary</span>
                                <?php else: ?>
                                <span class="badge bg-light text-primary float-end">Secondary</span>
                                <?php endif; ?>
                            </h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" id="form-<?php echo $grade['id']; ?>">
                                <input type="hidden" name="action" value="update_mappings">
                                <input type="hidden" name="grade_id" value="<?php echo $grade['id']; ?>">
                                
                                <p class="small text-muted mb-3">
                                    <i class="bi bi-info-circle me-1"></i>
                                    Select subjects available for this grade:
                                </p>
                                
                                <div class="subject-list" style="max-height: 400px; overflow-y: auto;">
                                    <?php foreach ($grade_subject_options[$grade['id']] as $subject): ?>
                                    <div class="form-check mb-2">
                                        <input 
                                            class="form-check-input" 
                                            type="checkbox" 
                                            name="subject_ids[]" 
                                            value="<?php echo $subject['id']; ?>"
                                            id="subject-<?php echo $grade['id']; ?>-<?php echo $subject['id']; ?>"
                                            <?php echo in_array($subject['id'], $current_mappings[$grade['id']]) ? 'checked' : ''; ?>
                                        >
                                        <label class="form-check-label" for="subject-<?php echo $grade['id']; ?>-<?php echo $subject['id']; ?>">
                                            <?php echo htmlspecialchars($subject['name']); ?>
                                            <?php if (!empty($subject['code'])): ?>
                                                <small class="text-muted">(<?php echo htmlspecialchars($subject['code']); ?>)</small>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                
                                <div class="mt-3 pt-3 border-top">
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <small class="text-muted">
                                            Selected: <span class="badge bg-primary" id="count-<?php echo $grade['id']; ?>">
                                                <?php echo count($current_mappings[$grade['id']]); ?>
                                            </span>
                                        </small>
                                    </div>
                                    <button type="submit" class="btn btn-primary w-100">
                                        <i class="bi bi-save me-2"></i>Save Changes
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>

            <!-- Quick Actions -->
            <div class="card mt-4">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="bi bi-lightning-fill me-2"></i>
                        Quick Actions
                    </h5>
                    <div class="row g-3">
                        <div class="col-md-4">
                            <button class="btn btn-outline-success w-100" onclick="selectAllSubjects()">
                                <i class="bi bi-check-all me-2"></i>
                                Select All Subjects for All Grades
                            </button>
                        </div>
                        <div class="col-md-4">
                            <button class="btn btn-outline-warning w-100" onclick="clearAllSubjects()">
                                <i class="bi bi-x-circle me-2"></i>
                                Clear All Selections
                            </button>
                        </div>
                        <div class="col-md-4">
                            <a href="dashboard.php" class="btn btn-outline-secondary w-100">
                                <i class="bi bi-arrow-left me-2"></i>
                                Back to Dashboard
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Update selected count when checkboxes change
document.querySelectorAll('form[id^="form-"]').forEach(form => {
    const gradeId = form.querySelector('[name="grade_id"]').value;
    const countBadge = document.getElementById('count-' + gradeId);
    
    form.addEventListener('change', function(e) {
        if (e.target.name === 'subject_ids[]') {
            const checked = form.querySelectorAll('input[name="subject_ids[]"]:checked').length;
            countBadge.textContent = checked;
        }
    });
});

function selectAllSubjects() {
    if (confirm('Select all subjects for all grades?')) {
        document.querySelectorAll('input[type="checkbox"][name="subject_ids[]"]').forEach(cb => {
            cb.checked = true;
        });
        // Update all counts
        document.querySelectorAll('[id^="count-"]').forEach(badge => {
            const form = badge.closest('form');
            const checked = form.querySelectorAll('input[name="subject_ids[]"]:checked').length;
            badge.textContent = checked;
        });
    }
}

function clearAllSubjects() {
    if (confirm('Clear all subject selections for all grades?')) {
        document.querySelectorAll('input[type="checkbox"][name="subject_ids[]"]').forEach(cb => {
            cb.checked = false;
        });
        // Update all counts
        document.querySelectorAll('[id^="count-"]').forEach(badge => {
            badge.textContent = '0';
        });
    }
}
</script>

<style>
.subject-list {
    padding-right: 0.5rem;
}

.subject-list::-webkit-scrollbar {
    width: 6px;
}

.subject-list::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 10px;
}

.subject-list::-webkit-scrollbar-thumb {
    background: #6f42c1;
    border-radius: 10px;
}

.subject-list::-webkit-scrollbar-thumb:hover {
    background: #5a32a3;
}

.form-check-input:checked {
    background-color: #6f42c1;
    border-color: #6f42c1;
}

.form-check-label {
    font-size: 0.9rem;
}
</style>

<?php include 'partials/footer.php'; ?>

