<?php
require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();
requireSchool();

$currentSchool = getCurrentSchool();
$errors = [];
$success = false;

// Enable detailed error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>Admin Create Student - Debug Mode</h1>";

// Test database connection first
echo "<h2>Database Connection Test</h2>";
try {
    $pdo = getDB();
    echo "✅ Database connection successful<br>";
} catch (Exception $e) {
    echo "❌ Database connection failed: " . $e->getMessage() . "<br>";
    echo "<p><strong>This is why you're getting 'unexpected error' - the database is not accessible.</strong></p>";
    echo "<h3>MySQL Fix Required:</h3>";
    echo "<ol>";
    echo "<li>Open XAMPP Control Panel</li>";
    echo "<li>Stop MySQL if running</li>";
    echo "<li>Delete these files:";
    echo "<ul>";
    echo "<li>C:\\xampp\\mysql\\data\\ib_logfile0</li>";
    echo "<li>C:\\xampp\\mysql\\data\\ib_logfile1</li>";
    echo "</ul></li>";
    echo "<li>Restart MySQL</li>";
    echo "</ol>";
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    echo "<h2>Form Submission Test</h2>";
    
    $firstName = trim($_POST['first_name'] ?? '');
    $lastName = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $gradeId = (int)($_POST['grade_id'] ?? 0);
    $password = (string)($_POST['password'] ?? '');
    $confirm = (string)($_POST['password_confirm'] ?? '');
    $subjectIds = array_filter(array_map('intval', (array)($_POST['subject_ids'] ?? [])));

    echo "Form data received:<br>";
    echo "- First Name: '$firstName'<br>";
    echo "- Last Name: '$lastName'<br>";
    echo "- Email: '$email'<br>";
    echo "- Grade ID: $gradeId<br>";
    echo "- Password: " . (strlen($password) > 0 ? 'Set' : 'Empty') . "<br>";
    echo "- Subject IDs: " . json_encode($subjectIds) . "<br>";

    if ($firstName === '') $errors[] = 'First name is required.';
    if ($lastName === '') $errors[] = 'Last name is required.';
    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Valid email is required.';
    if ($gradeId <= 0) $errors[] = 'Please select a grade.';
    if ($password === '' || strlen($password) < 8) $errors[] = 'Password must be at least 8 characters.';
    if ($password !== $confirm) $errors[] = 'Passwords do not match.';
    
    // Unique email (per tenant)
    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare('SELECT id FROM students WHERE email = ? AND academy_reference = ?');
            $stmt->execute([$email, $currentSchool]);
            if ($stmt->fetch()) {
                $errors[] = 'Email address is already registered.';
            }
        } catch (Exception $e) {
            $errors[] = 'Database error: ' . $e->getMessage();
        }
    }
    
    if (empty($errors)) {
        try {
            echo "<h3>Attempting to create student...</h3>";
            $studentId = createStudent([
                'first_name' => $firstName,
                'last_name' => $lastName,
                'email' => $email,
                'grade_id' => $gradeId,
                'password_plain' => $password,
                'subject_ids' => $subjectIds,
                'academy_reference' => $currentSchool
            ]);
            echo "✅ Student created successfully with ID: $studentId<br>";
            $success = true;
        } catch (Throwable $e) {
            echo "❌ Error creating student: " . $e->getMessage() . "<br>";
            echo "Error details: " . $e->getTraceAsString() . "<br>";
            $errors[] = 'Error creating student: ' . $e->getMessage();
        }
    }
}

$grades = getGrades();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Student - Debug Mode</title>    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container my-4">
        <div class="card">
            <div class="card-body">
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <i class="bi bi-check-circle me-1"></i>Student created successfully!
                    </div>
                <?php endif; ?>

                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <h6>Errors:</h6>
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <form method="POST">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">First Name *</label>
                            <input class="form-control" name="first_name" placeholder="e.g. Jane" required />
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Last Name *</label>
                            <input class="form-control" name="last_name" placeholder="e.g. Doe" required />
                        </div>
                        <div class="col-md-8">
                            <label class="form-label">Email *</label>
                            <input type="email" class="form-control" name="email" placeholder="you@example.com" required />
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Grade *</label>
                            <select class="form-select" name="grade_id" required>
                                <option value="">Select grade</option>
                                <?php foreach ($grades as $grade): ?>
                                    <option value="<?php echo $grade['id']; ?>"><?php echo htmlspecialchars($grade['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Create Password *</label>
                            <input type="password" class="form-control" name="password" minlength="8" placeholder="At least 8 characters" required />
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Confirm Password *</label>
                            <input type="password" class="form-control" name="password_confirm" minlength="8" placeholder="Re-type password" required />
                        </div>
                        <div class="col-12">
                            <label class="form-label">Subjects (choose one or more)</label>
                            <div id="subjectsContainer" class="border rounded p-3 bg-light" style="min-height: 120px;">
                                <div class="text-muted text-center py-4">
                                    <i class="bi bi-mortarboard fs-1 d-block mb-2"></i>
                                    Please select a grade first to see available subjects
                                </div>
                            </div>
                        </div>
                        <div class="col-12">
                            <button type="submit" class="btn btn-primary">Create Student</button>
                            <a href="dashboard.php" class="btn btn-secondary">Back to Dashboard</a>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
    const gradeSelect = document.getElementById('gradeSelect');
    const subjectsContainer = document.getElementById('subjectsContainer');
    
    function renderSubjects(items){
        // Update the checkbox container
        subjectsContainer.innerHTML = '';
        if(!items || !items.length){
            subjectsContainer.innerHTML = `
                <div class="text-muted text-center py-4">
                    <i class="bi bi-exclamation-circle fs-1 d-block mb-2"></i>
                    No subjects available for this grade yet.
                </div>
            `;
            return;
        }
        
        // Create header with Select All / Clear All buttons
        const buttonRow = document.createElement('div');
        buttonRow.className = 'row mb-3';
        buttonRow.innerHTML = `
            <div class="col-12">
                <div class="d-flex gap-2">
                    <button type="button" class="btn btn-sm btn-outline-primary" id="selectAllSubjects">
                        <i class="bi bi-check-square me-1"></i>Select All
                    </button>
                    <button type="button" class="btn btn-sm btn-outline-secondary" id="clearAllSubjects">
                        <i class="bi bi-square me-1"></i>Clear All
                    </button>
                </div>
            </div>
        `;
        subjectsContainer.appendChild(buttonRow);
        
        // Create checkbox grid
        const row = document.createElement('div');
        row.className = 'row g-2';
        
        items.forEach(s => {
            const col = document.createElement('div');
            col.className = 'col-md-6 col-lg-4';
            
            const checkboxDiv = document.createElement('div');
            checkboxDiv.className = 'form-check subject-checkbox p-2';
            
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.className = 'form-check-input';
            checkbox.name = 'subject_ids[]';
            checkbox.value = s.id;
            checkbox.id = 'subject_' + s.id;
            
            const label = document.createElement('label');
            label.className = 'form-check-label w-100';
            label.htmlFor = 'subject_' + s.id;
            label.textContent = s.name + (s.code ? ' ('+s.code+')' : '');
            
            checkboxDiv.appendChild(checkbox);
            checkboxDiv.appendChild(label);
            col.appendChild(checkboxDiv);
            row.appendChild(col);
        });
        
        subjectsContainer.appendChild(row);
        
        // Add event listeners for Select All / Clear All buttons
        document.getElementById('selectAllSubjects').addEventListener('click', function() {
            const checkboxes = subjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
            checkboxes.forEach(cb => cb.checked = true);
        });
        
        document.getElementById('clearAllSubjects').addEventListener('click', function() {
            const checkboxes = subjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
            checkboxes.forEach(cb => cb.checked = false);
        });
    }
    
    async function loadSubjects(){
        const gid = gradeSelect.value;
        renderSubjects([]);
        if(!gid) return;
        
        // Show loading message
        subjectsContainer.innerHTML = `
            <div class="text-muted text-center py-4">
                <i class="bi bi-hourglass-split fs-1 d-block mb-2"></i>
                Loading subjects...
            </div>
        `;
        
        try{
            const res = await fetch('../register.php?action=subjects&grade_id='+encodeURIComponent(gid), {headers: {'Accept':'application/json'}});
            if(!res.ok) {
                throw new Error('Server error: ' + res.status);
            }
            const data = await res.json();
            renderSubjects(data || []);
        }catch(e){
            console.error('Error loading subjects:', e);
            // Fallback: show sample subjects when database is not available
            const sampleSubjects = [
                {id: 1, name: 'Mathematics', code: 'MATH'},
                {id: 2, name: 'English', code: 'ENG'},
                {id: 3, name: 'Science', code: 'SCI'},
                {id: 4, name: 'History', code: 'HIST'},
                {id: 5, name: 'Geography', code: 'GEO'},
                {id: 6, name: 'Art', code: 'ART'},
                {id: 7, name: 'Physical Education', code: 'PE'},
                {id: 8, name: 'Computer Studies', code: 'CS'}
            ];
            
            subjectsContainer.innerHTML = `
                <div class="alert alert-warning mb-3">
                    <i class="bi bi-exclamation-triangle me-1"></i>
                    Database connection issue. Showing sample subjects for testing.
                </div>
            `;
            renderSubjects(sampleSubjects);
        }
    }
    
    gradeSelect?.addEventListener('change', loadSubjects);
    </script>
</body>
</html>
