<?php
/**
 * Test Moodle Connection
 * Quick diagnostic tool to verify Moodle API connectivity
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/moodle_api.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$schoolName = getSchoolName();

$tests = [];
$overall_status = 'success';
$test_time = date('Y-m-d H:i:s');

// Test 1: Check Moodle Settings
$moodleAPI = new MoodleAPI($academy_reference, $pdo);

// Get Moodle settings for display
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ? AND setting_key LIKE 'moodle_%'");
$stmt->execute([$academy_reference]);
$moodle_settings = [];
while ($row = $stmt->fetch()) {
    $moodle_settings[$row['setting_key']] = $row['setting_value'];
}

$tests['settings'] = [
    'name' => 'Moodle Settings',
    'status' => $moodleAPI->isEnabled() ? 'success' : 'error',
    'message' => $moodleAPI->isEnabled() ? 'Moodle is configured and enabled' : 'Moodle settings not configured',
    'details' => [
        'Enabled' => ($moodle_settings['moodle_enabled'] ?? 0) ? 'Yes' : 'No',
        'URL Configured' => !empty($moodle_settings['moodle_url'] ?? '') ? 'Yes' : 'No',
        'Token Configured' => !empty($moodle_settings['moodle_token'] ?? '') ? 'Yes' : 'No'
    ]
];

if ($moodleAPI->isEnabled()) {
    // Test 2: Test Connection
    try {
        $connectionTest = $moodleAPI->testConnection();
        $tests['siteinfo'] = [
            'name' => 'Site Connection',
            'status' => $connectionTest['success'] ? 'success' : 'error',
            'message' => $connectionTest['success'] ? 'Connected to Moodle site' : 'Connection failed',
            'details' => [
                'URL' => $connectionTest['moodle_url'] ?? 'N/A',
                'Status' => $connectionTest['message'] ?? 'N/A',
            ]
        ];
        if (!$connectionTest['success']) {
            $overall_status = 'error';
        }
    } catch (Exception $e) {
        $tests['siteinfo'] = [
            'name' => 'Site Connection',
            'status' => 'error',
            'message' => 'Failed to connect to Moodle',
            'details' => ['Error' => $e->getMessage()]
        ];
        $overall_status = 'error';
    }
    
    // Test 3: List Courses
    try {
        $courses_result = $moodleAPI->getCourses();
        
        // getCourses() returns ['success' => bool, 'data' => array] or ['success' => false, 'error' => string]
        $courses = [];
        if (isset($courses_result['success']) && $courses_result['success'] && isset($courses_result['data'])) {
            $courses = is_array($courses_result['data']) ? $courses_result['data'] : [];
        }
        
        $course_count = count($courses);
        $sample_course = null;
        if ($course_count > 0 && isset($courses[0]) && is_array($courses[0])) {
            $sample_course = $courses[0]['fullname'] ?? null;
        }
        
        $tests['courses'] = [
            'name' => 'Course Access',
            'status' => (isset($courses_result['success']) && $courses_result['success']) ? 'success' : 'error',
            'message' => (isset($courses_result['success']) && $courses_result['success']) 
                ? ($course_count . ' courses available') 
                : ($courses_result['error'] ?? 'Could not fetch courses'),
            'details' => [
                'Total Courses' => $course_count,
                'Sample' => $sample_course ?? 'No courses'
            ]
        ];
        
        if (!isset($courses_result['success']) || !$courses_result['success']) {
            if ($overall_status !== 'error') $overall_status = 'warning';
        }
    } catch (Exception $e) {
        $tests['courses'] = [
            'name' => 'Course Access',
            'status' => 'warning',
            'message' => 'Could not fetch courses',
            'details' => ['Error' => $e->getMessage()]
        ];
        if ($overall_status !== 'error') $overall_status = 'warning';
    }
    
    // Test 4: Check API Token
    $tests['permissions'] = [
        'name' => 'API Permissions',
        'status' => 'success',
        'message' => 'API token configured',
        'details' => ['Token' => 'Present', 'Status' => 'Active']
    ];
    
    // Test 5: Database Mappings
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count 
            FROM subject_moodle_mapping 
            WHERE academy_reference = ?
        ");
        $stmt->execute([$academy_reference]);
        $mapping_count = $stmt->fetchColumn();
        
        $tests['mappings'] = [
            'name' => 'Subject Mappings',
            'status' => $mapping_count > 0 ? 'success' : 'warning',
            'message' => $mapping_count . ' mappings configured',
            'details' => [
                'Total' => $mapping_count,
                'Action' => $mapping_count === 0 ? 'Configure mappings first' : 'Ready'
            ]
        ];
    } catch (Exception $e) {
        $tests['mappings'] = [
            'name' => 'Subject Mappings',
            'status' => 'error',
            'message' => 'Database error',
            'details' => ['Error' => $e->getMessage()]
        ];
    }
    
    // Test 6: Student Sync Status
    try {
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN moodle_user_id IS NOT NULL THEN 1 ELSE 0 END) as synced
            FROM students 
            WHERE academy_reference = ?
        ");
        $stmt->execute([$academy_reference]);
        $sync_stats = $stmt->fetch();
        
        $tests['students'] = [
            'name' => 'Student Sync',
            'status' => $sync_stats['synced'] > 0 ? 'success' : 'info',
            'message' => $sync_stats['synced'] . ' of ' . $sync_stats['total'] . ' students synced',
            'details' => [
                'Total Students' => $sync_stats['total'],
                'Synced' => $sync_stats['synced'],
                'Not Synced' => $sync_stats['total'] - $sync_stats['synced']
            ]
        ];
    } catch (Exception $e) {
        $tests['students'] = [
            'name' => 'Student Sync',
            'status' => 'error',
            'message' => 'Database error',
            'details' => ['Error' => $e->getMessage()]
        ];
    }
    
} else {
    $overall_status = 'error';
}

$moodle_enabled = $moodleAPI->isEnabled();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Moodle Connection - <?= htmlspecialchars($schoolName) ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .test-card {
            transition: all 0.3s;
            border-left: 4px solid #dee2e6;
        }
        .test-card.success { border-left-color: #28a745; }
        .test-card.warning { border-left-color: #ffc107; }
        .test-card.error { border-left-color: #dc3545; }
        .test-card.info { border-left-color: #17a2b8; }
        
        .status-icon {
            font-size: 48px;
            margin-bottom: 10px;
        }
        .status-icon.success { color: #28a745; }
        .status-icon.warning { color: #ffc107; }
        .status-icon.error { color: #dc3545; }
        .status-icon.info { color: #17a2b8; }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-plug text-primary"></i> Moodle Connection Test</h2>
                <p class="text-muted mb-0">Diagnostic results for <?= htmlspecialchars($schoolName) ?></p>
            </div>
            <div>
                <a href="index.php" class="btn btn-outline-secondary me-2">
                    <i class="bi bi-arrow-left"></i> Moodle Hub
                </a>
                <a href="javascript:location.reload()" class="btn btn-primary">
                    <i class="bi bi-arrow-repeat"></i> Retest
                </a>
            </div>
        </div>

        <?php if (!$moodle_enabled): ?>
        <!-- Not Configured Alert -->
        <div class="alert alert-danger">
            <h4><i class="bi bi-exclamation-triangle"></i> Moodle Not Configured</h4>
            <p>Please configure your Moodle API settings first.</p>
            <a href="settings.php" class="btn btn-primary">
                <i class="bi bi-gear"></i> Configure Moodle API
            </a>
        </div>
        <?php else: ?>

        <!-- Overall Status -->
        <div class="card mb-4 shadow-sm">
            <div class="card-body text-center">
                <?php if ($overall_status === 'success'): ?>
                <i class="bi bi-check-circle-fill status-icon success"></i>
                <h3 class="text-success">All Tests Passed!</h3>
                <p class="text-muted mb-0">Your Moodle integration is working correctly.</p>
                <?php elseif ($overall_status === 'warning'): ?>
                <i class="bi bi-exclamation-triangle-fill status-icon warning"></i>
                <h3 class="text-warning">Connection OK with Warnings</h3>
                <p class="text-muted mb-0">Moodle is connected but some features may be limited.</p>
                <?php else: ?>
                <i class="bi bi-x-circle-fill status-icon error"></i>
                <h3 class="text-danger">Connection Failed</h3>
                <p class="text-muted mb-0">Please check your Moodle configuration.</p>
                <?php endif; ?>
                <small class="d-block mt-2 text-muted">
                    <i class="bi bi-clock"></i> Last tested: <?= $test_time ?>
                </small>
            </div>
        </div>

        <!-- Individual Tests -->
        <div class="row">
            <?php foreach ($tests as $test): ?>
            <div class="col-md-6 mb-3">
                <div class="card test-card <?= $test['status'] ?> h-100">
                    <div class="card-body">
                        <div class="d-flex align-items-center mb-3">
                            <?php if ($test['status'] === 'success'): ?>
                            <i class="bi bi-check-circle-fill text-success me-2" style="font-size: 24px;"></i>
                            <?php elseif ($test['status'] === 'warning'): ?>
                            <i class="bi bi-exclamation-triangle-fill text-warning me-2" style="font-size: 24px;"></i>
                            <?php elseif ($test['status'] === 'error'): ?>
                            <i class="bi bi-x-circle-fill text-danger me-2" style="font-size: 24px;"></i>
                            <?php else: ?>
                            <i class="bi bi-info-circle-fill text-info me-2" style="font-size: 24px;"></i>
                            <?php endif; ?>
                            <h5 class="mb-0"><?= htmlspecialchars($test['name']) ?></h5>
                        </div>
                        
                        <p class="mb-3"><?= htmlspecialchars($test['message']) ?></p>
                        
                        <?php if (!empty($test['details'])): ?>
                        <div class="table-responsive">
                            <table class="table table-sm table-borderless mb-0">
                                <?php foreach ($test['details'] as $key => $value): ?>
                                <tr>
                                    <td class="text-muted" style="width: 40%;"><?= htmlspecialchars($key) ?>:</td>
                                    <td><strong><?= htmlspecialchars($value ?? '') ?></strong></td>
                                </tr>
                                <?php endforeach; ?>
                            </table>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <!-- Action Items -->
        <div class="card mt-4">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="bi bi-list-check"></i> Next Steps</h5>
            </div>
            <div class="card-body">
                <?php if (!$moodleAPI->isEnabled()): ?>
                <div class="alert alert-danger">
                    <strong>Configure Moodle First:</strong>
                    <ol class="mb-0 mt-2">
                        <li>Go to <a href="settings.php" class="alert-link">Moodle API Settings</a></li>
                        <li>Enter your Moodle URL and API token</li>
                        <li>Save settings and return here to test</li>
                    </ol>
                </div>
                <?php else: ?>
                <div class="row">
                    <div class="col-md-6">
                        <h6>✅ Recommended Actions:</h6>
                        <ul>
                            <?php if (isset($tests['mappings']) && $tests['mappings']['details']['Total'] == 0): ?>
                            <li><a href="configure_mapping.php">Configure subject-course mappings</a></li>
                            <?php endif; ?>
                            <?php if (isset($tests['students']) && $tests['students']['details']['Not Synced'] > 0): ?>
                            <li><a href="sync_students.php">Sync students to Moodle</a></li>
                            <?php endif; ?>
                            <li><a href="../../test_moodle_functions.php?key=test123" target="_blank">Run complete API tests</a></li>
                            <li><a href="../../test_moodle_complete_workflow.php?key=test123" target="_blank">Test full workflow</a></li>
                        </ul>
                    </div>
                    <div class="col-md-6">
                        <h6>📚 Documentation:</h6>
                        <ul>
                            <li><a href="../../MOODLE_MAPPING_GUIDE.md" target="_blank">Mapping Guide</a></li>
                            <li><a href="../../README_PAYMENT_SYSTEM.md" target="_blank">Payment System</a></li>
                            <li><a href="index.php">Moodle Management Hub</a></li>
                        </ul>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Technical Details -->
        <div class="card mt-3">
            <div class="card-header">
                <h6 class="mb-0"><i class="bi bi-gear"></i> Technical Details</h6>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <small class="text-muted">
                            <strong>Academy:</strong> <?= htmlspecialchars($academy_reference) ?><br>
                            <strong>Test Time:</strong> <?= date('Y-m-d H:i:s') ?><br>
                            <strong>PHP Version:</strong> <?= phpversion() ?>
                        </small>
                    </div>
                    <div class="col-md-6">
                        <small class="text-muted">
                            <strong>Tests Run:</strong> <?= count($tests) ?><br>
                            <strong>Status:</strong> <span class="text-<?= $overall_status === 'success' ? 'success' : ($overall_status === 'warning' ? 'warning' : 'danger') ?>"><?= strtoupper($overall_status) ?></span><br>
                            <strong>Server:</strong> <?= $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown' ?>
                        </small>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

