<?php
/**
 * API Endpoint: Get Tenant Storage Info
 * 
 * GET /api/storage/tenant/{id}
 * 
 * Returns storage usage for a specific tenant
 */

header('Content-Type: application/json');

session_start();
require_once '../../config.php';
require_once '../../includes/tenant_directory_manager.php';

// Check authentication
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

// Get tenant ID from URL
$tenant_id = $_GET['id'] ?? null;

if (!$tenant_id) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing tenant_id parameter']);
    exit;
}

// Check permission (super admin can view all, regular admin only their own)
if ($_SESSION['role'] !== 'super_admin' && $_SESSION['academy_reference'] !== $tenant_id) {
    http_response_code(403);
    echo json_encode(['error' => 'Access denied']);
    exit;
}

try {
    // Database connection
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    // Initialize directory manager
    $dirManager = new TenantDirectoryManager($conn);
    
    // Get storage usage
    $usage = $dirManager->calculateTenantStorageUsage($tenant_id);
    
    if (!$usage['success']) {
        http_response_code(404);
        echo json_encode(['error' => 'Tenant not found or error calculating storage']);
        exit;
    }
    
    // Get file statistics
    $stmt = $conn->prepare("
        SELECT 
            file_category,
            COUNT(*) as file_count,
            SUM(file_size) as total_size
        FROM tenant_files
        WHERE tenant_id = ? AND is_deleted = 0
        GROUP BY file_category
    ");
    $stmt->execute([$tenant_id]);
    $categoryStats = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Return data
    echo json_encode([
        'success' => true,
        'tenant_id' => $tenant_id,
        'storage' => $usage,
        'categories' => $categoryStats,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

