"""
Configuration file for Python Moodle Sync Bot
Contains all database, API, and application settings
"""

import os
from pathlib import Path

# Database configuration
DB_CONFIG = {
    'host': '127.0.0.1',
    'port': 3306,
    'database': 'skolo_kine',  # Local XAMPP database name
    'user': 'root',            # Local XAMPP username
    'password': '',            # Local XAMPP password (empty)
    'charset': 'utf8mb4'
}

# SMTP configuration
SMTP_CONFIG = {
    'host': 'mail.melanegroup.com',
    'port': 587,
    'username': 'info@melanegroup.com',
    'password': 'your-email-password',  # Replace with actual password
    'encryption': 'tls',
    'from_email': 'info@melanegroup.com',
    'from_name': 'Skolo-Kine Learning Platform'
}

# Application settings
APP_CONFIG = {
    'name': 'Skolo-Kine',
    'max_students': 500,
    'admin_username': 'admin',
    'admin_password': 'admin123',
    'whatsapp_group_link': 'https://chat.whatsapp.com/G6drhwZoJtZJsJXfINOhV2?mode=ems_wa_t',
    'debug': True
}

# Moodle configuration
MOODLE_CONFIG = {
    'url': 'https://skoloi.melanegroup.com/kine/',
    'api_url': 'https://skoloi.melanegroup.com/kine/webservice/rest/server.php',
    'api_token': 'edb3c3837e8d5b89ae35b92d1dfc40e2'  # Update with your token
}

# Bot configuration with environment support
import os

BOT_CONFIG = {
    # Environment
    'environment': os.getenv('BOT_ENV', 'development'),
    'debug': os.getenv('BOT_DEBUG', 'false').lower() == 'true',
    
    # API Configuration
    'skolo_api_url': os.getenv('SKOLO_API_URL', 'https://kine.melanegroup.com/Skolo-Kine/api/export_csv.php'),
    'skolo_api_key': os.getenv('SKOLO_API_KEY', 'skolo_prod_2024_8f9a2b3c4d5e6f7a8b9c0d1e2f3a4b5c'),
    'moodle_ws_function': 'core_user_create_users',
    
    # Performance Settings
    'batch_size': int(os.getenv('BOT_BATCH_SIZE', '5')),
    'max_workers': int(os.getenv('BOT_MAX_WORKERS', '3')),
    'timeout': int(os.getenv('BOT_TIMEOUT', '30')),
    
    # Retry Logic
    'max_retries': int(os.getenv('BOT_MAX_RETRIES', '3')),
    'retry_delay': int(os.getenv('BOT_RETRY_DELAY', '2')),
    
    # SSL and Security
    'verify_ssl': os.getenv('BOT_VERIFY_SSL', 'false').lower() == 'true',
    
    # Features
    'enable_metrics': os.getenv('BOT_ENABLE_METRICS', 'true').lower() == 'true',
    'enable_caching': os.getenv('BOT_ENABLE_CACHING', 'true').lower() == 'true',
    'cache_ttl': int(os.getenv('BOT_CACHE_TTL', '3600')),
    
    # Monitoring
    'enable_health_checks': os.getenv('BOT_ENABLE_HEALTH_CHECKS', 'true').lower() == 'true',
    'health_check_interval': int(os.getenv('BOT_HEALTH_CHECK_INTERVAL', '300')),  # 5 minutes
}

# File paths
BASE_DIR = Path(__file__).parent.parent
BOT_DIR = Path(__file__).parent
TEMP_DIR = BOT_DIR / 'temp'
LOGS_DIR = BOT_DIR / 'logs'
EXPORTS_DIR = BASE_DIR / 'exports'

# Ensure directories exist
TEMP_DIR.mkdir(exist_ok=True)
LOGS_DIR.mkdir(exist_ok=True)
EXPORTS_DIR.mkdir(exist_ok=True)

# Log file paths
LOG_FILES = {
    'sync': LOGS_DIR / 'python_sync.log',
    'test_user_creation': LOGS_DIR / 'test_user_creation.log',
    'test_sync': LOGS_DIR / 'test_sync.log'
}

# Lock file for preventing concurrent runs
LOCK_FILE = TEMP_DIR / 'python_sync.lock'
LOCK_TIMEOUT = 300  # 5 minutes

# Subject to course mapping (subject_id -> course_shortname)
# Mapped to your Moodle course shortnames
SUBJECT_TO_COURSE_MAPPING = {
    1: 'COS100',   # Consumer Science
    2: 'MTH100',   # Mathematics
    3: 'MTH100',   # Mathematics (mapped to same as subject 2)
    4: 'ENG100',   # English (mapped to same as subject 5)
    5: 'ENG100',   # English
    6: 'SCI100',   # Science (mapped to same as subject 18)
    7: 'SIS100',   # Siswati
    8: 'COS100',   # Consumer Science (mapped to same as subject 1)
    9: 'REL100',   # Religious Education (mapped to same as subject 21)
    10: 'SIS100',  # Siswati (mapped to same as subject 7)
    11: 'ICT100',  # ICT (mapped to same as subject 24)
    18: 'SCI100',  # Science
    21: 'REL100',  # Religious Education
    24: 'ICT100',  # ICT
    27: 'EXA100',  # Expressive Arts
    31: 'SOS100',  # Social Studies
    32: 'AGR100',  # Agriculture
    33: 'HPE100',  # HPE
    99: 'COS100',  # Map to existing Consumer Science course
    100: 'MTH100', # Map to existing Mathematics course
    101: 'SCI100', # Map to existing Science course
}

# Course name patterns for automatic mapping
COURSE_NAME_PATTERNS = {
    'mathematics': 2,
    'math': 2,
    'english': 5,
    'science': 18,
    'consumer': 1,
    'siswati': 7,
    'religious': 21,
    'ict': 24,
    'arts': 27,
    'social': 31,
    'agriculture': 32,
    'hpe': 33
}

# Database connection string for SQLAlchemy
def get_database_url():
    """Get database URL for SQLAlchemy"""
    return f"mysql+pymysql://{DB_CONFIG['user']}:{DB_CONFIG['password']}@{DB_CONFIG['host']}:{DB_CONFIG['port']}/{DB_CONFIG['database']}?charset={DB_CONFIG['charset']}"

# Environment variable overrides
def load_from_env():
    """Load configuration from environment variables if available"""
    # Database
    if os.getenv('DB_HOST'):
        DB_CONFIG['host'] = os.getenv('DB_HOST')
    if os.getenv('DB_PORT'):
        DB_CONFIG['port'] = int(os.getenv('DB_PORT'))
    if os.getenv('DB_NAME'):
        DB_CONFIG['database'] = os.getenv('DB_NAME')
    if os.getenv('DB_USER'):
        DB_CONFIG['user'] = os.getenv('DB_USER')
    if os.getenv('DB_PASS'):
        DB_CONFIG['password'] = os.getenv('DB_PASS')
    
    # Moodle
    if os.getenv('MOODLE_API_URL'):
        MOODLE_CONFIG['api_url'] = os.getenv('MOODLE_API_URL')
    if os.getenv('MOODLE_API_TOKEN'):
        MOODLE_CONFIG['api_token'] = os.getenv('MOODLE_API_TOKEN')
    
    # SMTP
    if os.getenv('SMTP_HOST'):
        SMTP_CONFIG['host'] = os.getenv('SMTP_HOST')
    if os.getenv('SMTP_USERNAME'):
        SMTP_CONFIG['username'] = os.getenv('SMTP_USERNAME')
    if os.getenv('SMTP_PASSWORD'):
        SMTP_CONFIG['password'] = os.getenv('SMTP_PASSWORD')

# Helper functions for subject mapping management
def validate_subject_mapping():
    """Validate the subject to course mapping configuration"""
    issues = []
    
    for subject_id, course_shortname in SUBJECT_TO_COURSE_MAPPING.items():
        if not course_shortname:
            issues.append(f"Subject ID {subject_id} has empty course shortname")
        elif not isinstance(course_shortname, str):
            issues.append(f"Subject ID {subject_id} has non-string course shortname: {type(course_shortname)}")
        elif not course_shortname.strip():
            issues.append(f"Subject ID {subject_id} has whitespace-only course shortname")
    
    return issues

def get_missing_subject_ids(csv_subject_ids):
    """Get subject IDs that are in CSV but not in mapping"""
    configured_ids = set(SUBJECT_TO_COURSE_MAPPING.keys())
    csv_ids = set(csv_subject_ids)
    return sorted(csv_ids - configured_ids)

def add_subject_mapping(subject_id, course_shortname):
    """Add a new subject mapping (for dynamic configuration)"""
    if not isinstance(subject_id, int):
        raise ValueError("Subject ID must be an integer")
    if not isinstance(course_shortname, str) or not course_shortname.strip():
        raise ValueError("Course shortname must be a non-empty string")
    
    SUBJECT_TO_COURSE_MAPPING[subject_id] = course_shortname.strip()

# Load environment variables
load_from_env()
