#!/usr/bin/env python3
"""
Moodle Sync Bot - Python Version
Downloads CSV from Skolo-Kine and uploads to Moodle every 60 seconds

Configuration:
1. Update the configuration in config.py
2. Set up cron job: */1 * * * * /usr/bin/python3 /path/to/moodle_sync_bot.py
3. Make sure the script has write permissions to the temp directory
"""

import os
import sys
import time
import json
import csv
import re
import logging
import requests
from datetime import datetime
from pathlib import Path
from typing import List, Dict, Optional, Any
from urllib.parse import urlencode
from dataclasses import dataclass, field
import concurrent.futures
import threading
import pickle
import shutil

# Windows compatibility for file locking
try:
    import fcntl
    HAS_FCNTL = True
except ImportError:
    HAS_FCNTL = False

# Add the bot directory to the Python path
sys.path.insert(0, str(Path(__file__).parent))

from config import (
    BOT_CONFIG, MOODLE_CONFIG, TEMP_DIR, LOGS_DIR, LOCK_FILE, 
    LOCK_TIMEOUT, SUBJECT_TO_COURSE_MAPPING, COURSE_NAME_PATTERNS
)
from database import get_verified_students, get_subjects_for_student

# Suppress SSL warnings for cleaner logs
import urllib3
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

# Configure logging
def setup_logging():
    """Setup logging configuration"""
    log_file = LOGS_DIR / 'sync.log'
    
    # Create console handler with UTF-8 encoding for Windows compatibility
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.INFO)
    console_formatter = logging.Formatter('[%(asctime)s] [%(levelname)s] %(message)s')
    console_handler.setFormatter(console_formatter)
    
    # Create file handler
    file_handler = logging.FileHandler(log_file, encoding='utf-8')
    file_handler.setLevel(logging.INFO)
    file_formatter = logging.Formatter('[%(asctime)s] [%(levelname)s] %(message)s')
    file_handler.setFormatter(file_formatter)
    
    # Setup logger
    logger = logging.getLogger(__name__)
    logger.setLevel(logging.INFO)
    logger.addHandler(console_handler)
    logger.addHandler(file_handler)
    
    return logger

logger = setup_logging()

@dataclass
class MetricsTracker:
    """Comprehensive metrics tracking for the sync process"""
    start_time: float = field(default_factory=time.time)
    users_processed: int = 0
    users_created: int = 0
    users_skipped: int = 0
    users_with_errors: int = 0
    enrollments_successful: int = 0
    enrollments_failed: int = 0
    enrollments_skipped: int = 0
    api_calls_made: int = 0
    errors_by_type: Dict[str, int] = field(default_factory=dict)
    performance_breakdown: Dict[str, float] = field(default_factory=dict)
    
    def log_metrics(self, bot_instance):
        """Log comprehensive metrics at the end"""
        total_time = time.time() - self.start_time
        bot_instance.log_message("🎉 Moodle sync completed successfully!")
        bot_instance.log_message("📊 === COMPREHENSIVE SYNC METRICS ===")
        bot_instance.log_message(f"⏱️  Total Sync Time: {total_time:.2f} seconds")
        bot_instance.log_message(f"👥 Total Users Processed: {self.users_processed}")
        bot_instance.log_message(f"🆕 New Users Created: {self.users_created}")
        bot_instance.log_message(f"⏭️  Existing Users Skipped: {self.users_skipped}")
        bot_instance.log_message(f"❌ Users with Errors: {self.users_with_errors}")
        bot_instance.log_message(f"📚 Enrollments Successful: {self.enrollments_successful}")
        bot_instance.log_message(f"📚 Enrollments Failed: {self.enrollments_failed}")
        bot_instance.log_message(f"📚 Enrollments Skipped: {self.enrollments_skipped}")
        bot_instance.log_message(f"🌐 API Calls Made: {self.api_calls_made}")
        
        if self.performance_breakdown:
            bot_instance.log_message("📊 === PERFORMANCE BREAKDOWN ===")
            for stage, duration in self.performance_breakdown.items():
                bot_instance.log_message(f"🔧 {stage}: {duration:.2f}s")
        
        if self.errors_by_type:
            bot_instance.log_message("📊 === ERROR BREAKDOWN ===")
            for error_type, count in self.errors_by_type.items():
                bot_instance.log_message(f"❌ {error_type}: {count}")
        
        bot_instance.log_message("📊 === END METRICS ===")
        
        # Save metrics to file for admin portal
        try:
            metrics_data = {
                'users_processed': self.users_processed,
                'users_created': self.users_created,
                'users_skipped': self.users_skipped,
                'users_with_errors': self.users_with_errors,
                'enrollments_successful': self.enrollments_successful,
                'enrollments_failed': self.enrollments_failed,
                'enrollments_skipped': self.enrollments_skipped,
                'api_calls_made': self.api_calls_made,
                'total_sync_time': total_time,
                'performance_breakdown': self.performance_breakdown,
                'errors_by_type': self.errors_by_type,
                'timestamp': datetime.now().isoformat()
            }
            
            metrics_file = bot_instance.temp_dir / 'metrics.json'
            with open(metrics_file, 'w') as f:
                json.dump(metrics_data, f, indent=2)
        except Exception as e:
            bot_instance.log_message(f"⚠️ Failed to save metrics: {e}", 'WARNING')

class MoodleSyncBot:
    """Main bot class for syncing students to Moodle"""
    
    def __init__(self):
        self.config = BOT_CONFIG
        self.moodle_config = MOODLE_CONFIG
        self.session = requests.Session()
        self.session.headers.update({
            'User-Agent': 'Moodle-Sync-Bot-Python/1.0'
        })
        self.metrics = MetricsTracker() if self.config.get('enable_metrics', True) else None
        self.cache_dir = Path(__file__).parent / 'cache'
        self.cache_dir.mkdir(exist_ok=True)
        
        # Create temp directory for monitoring files
        self.temp_dir = Path(__file__).parent / 'temp'
        self.temp_dir.mkdir(exist_ok=True)
        
        # Set up lock file path for Python bot
        self.lock_file = self.temp_dir / 'python_sync.lock'
        
    def log_message(self, message: str, level: str = 'INFO'):
        """Log a message with timestamp"""
        timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        log_entry = f"[{timestamp}] [{level}] {message}"
        
        if level == 'ERROR':
            logger.error(message)
        elif level == 'WARNING':
            logger.warning(message)
        else:
            logger.info(message)
    
    def get_cached_courses(self) -> Optional[List[Dict]]:
        """Get cached Moodle courses"""
        if not self.config.get('enable_caching', True):
            return None
            
        cache_file = self.cache_dir / 'moodle_courses.pkl'
        if cache_file.exists():
            cache_age = time.time() - cache_file.stat().st_mtime
            cache_ttl = self.config.get('cache_ttl', 3600)  # 1 hour default
            
            if cache_age < cache_ttl:
                try:
                    with open(cache_file, 'rb') as f:
                        courses = pickle.load(f)
                        self.log_message(f"📦 Loaded {len(courses)} courses from cache (age: {cache_age:.1f}s)")
                        return courses
                except Exception as e:
                    self.log_message(f"⚠️ Failed to load cache: {e}", 'WARNING')
        
        return None
    
    def cache_courses(self, courses: List[Dict]):
        """Cache Moodle courses"""
        if not self.config.get('enable_caching', True):
            return
            
        cache_file = self.cache_dir / 'moodle_courses.pkl'
        try:
            with open(cache_file, 'wb') as f:
                pickle.dump(courses, f)
                self.log_message(f"💾 Cached {len(courses)} courses")
        except Exception as e:
            self.log_message(f"⚠️ Failed to cache courses: {e}", 'WARNING')
    
    def create_batches(self, items: List[Any], batch_size: int) -> List[List[Any]]:
        """Create batches from a list"""
        return [items[i:i + batch_size] for i in range(0, len(items), batch_size)]
    
    def upload_users_parallel(self, users: List[Dict], max_workers: int = None) -> Dict[str, Any]:
        """Upload users in parallel for better performance with duplicate email handling"""
        if not users:
            return {'success': True, 'message': 'No users to upload'}
        
        # First, check for duplicate emails and filter them out
        filtered_users = self.filter_duplicate_emails(users)
        
        if not filtered_users:
            self.log_message("⚠️ All users have duplicate emails, skipping upload")
            return {'success': True, 'message': 'All users have duplicate emails', 'successful_uploads': 0, 'failed_uploads': len(users)}
        
        max_workers = max_workers or self.config.get('max_workers', 3)
        batch_size = self.config['batch_size']
        batches = self.create_batches(filtered_users, batch_size)
        
        self.log_message(f"🚀 Starting parallel upload of {len(filtered_users)} users in {len(batches)} batches using {max_workers} workers")
        self.log_message(f"📊 Filtered out {len(users) - len(filtered_users)} users with duplicate emails")
        
        results = []
        successful_uploads = 0
        failed_uploads = 0
        
        with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as executor:
            # Submit all batches
            future_to_batch = {
                executor.submit(self.upload_to_moodle, batch): (i, batch) 
                for i, batch in enumerate(batches)
            }
            
            # Process completed batches
            for future in concurrent.futures.as_completed(future_to_batch):
                batch_num, batch = future_to_batch[future]
                try:
                    result = future.result()
                    if result:
                        successful_uploads += len(batch)
                        self.log_message(f"✅ Batch {batch_num + 1}/{len(batches)} completed successfully ({len(batch)} users)")
                    else:
                        failed_uploads += len(batch)
                        self.log_message(f"❌ Batch {batch_num + 1}/{len(batches)} failed ({len(batch)} users)", 'WARNING')
                except Exception as e:
                    failed_uploads += len(batch)
                    result = False  # Set result to False for exception case
                    self.log_message(f"❌ Batch {batch_num + 1}/{len(batches)} failed with exception: {e}", 'ERROR')
                
                results.append(result)
        
        self.log_message(f"🎯 Parallel upload completed: {successful_uploads} successful, {failed_uploads} failed")
        return {
            'success': successful_uploads > 0,
            'successful_uploads': successful_uploads,
            'failed_uploads': failed_uploads,
            'results': results
        }
    
    def download_csv(self) -> str:
        """Download CSV from Skolo-Kine API"""
        url = f"{self.config['skolo_api_url']}?api_key={self.config['skolo_api_key']}"
        
        try:
            response = self.session.get(
                url,
                timeout=self.config['timeout'],
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            if not response.text.strip():
                raise Exception("Empty CSV data received")
            
            return response.text
            
        except requests.exceptions.RequestException as e:
            raise Exception(f"HTTP Error: {e}")
    
    def parse_csv(self, csv_data: str) -> List[Dict[str, Any]]:
        """Parse CSV data - Extended format: username, firstname, lastname, password, email, grade_name, subject_ids"""
        # Remove BOM and control characters
        csv_data = csv_data.replace('\ufeff', '')
        csv_data = re.sub(r'[\x00-\x1F\x7F]', '', csv_data)
        
        # The CSV is malformed - all data is on one line
        # Split by student ID pattern (10 digits starting with 2025)
        header = "username,firstname,lastname,password,email,grade_name,subject_ids"
        data = csv_data.replace(header, "")
        
        pattern = r'(\d{10},[^,]+,[^,]+,[^,]+,[^@]+@[^,]+,"[^"]+","[^"]+")'
        matches = re.findall(pattern, data)
        
        users = []
        
        for match in matches:
            # Parse CSV line
            reader = csv.reader([match])
            row = next(reader)
            
            if len(row) >= 7:
                # Parse subject IDs (comma-separated)
                subject_ids = []
                if row[6].strip():
                    subject_ids = [int(x.strip()) for x in row[6].strip('"').split(',') if x.strip()]
                
                users.append({
                    'username': row[0].strip(),
                    'firstname': row[1].strip('"'),
                    'lastname': row[2].strip('"'),
                    'password': row[3].strip('"'),
                    'email': row[4].strip('"'),
                    'grade_name': row[5].strip('"'),
                    'subject_ids': subject_ids
                })
        
        return users
    
    def format_username_for_moodle(self, username: str) -> str:
        """Format username to match what users receive in their email credentials"""
        formatted_username = username.strip()
        
        # Fix usernames that start with 0 or are purely numeric (Moodle doesn't like these)
        if formatted_username.startswith('0') or formatted_username.isdigit():
            # Convert to a valid format: add 'user_' prefix
            formatted_username = f"user_{formatted_username}"
        
        # Convert to lowercase for consistency
        formatted_username = formatted_username.lower()
        
        return formatted_username
    
    def get_original_username(self, formatted_username: str) -> str:
        """Get original username from formatted username for Moodle lookups"""
        original_username = formatted_username.strip()
        
        # Remove 'user_' prefix if present
        if original_username.startswith('user_'):
            original_username = original_username[5:]  # Remove 'user_' prefix
        
        return original_username

    def clean_user_data(self, user: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Clean and validate user data for Moodle compatibility"""
        # Clean and validate data to prevent "Invalid parameter value detected" errors
        clean_user = {
            'username': user['username'][:100].strip(),
            'firstname': user['firstname'][:100].strip(),
            'lastname': user['lastname'][:100].strip(),
            'email': user['email'][:100].strip(),
            'password': user['password'][:100].strip(),
            'auth': 'manual'
        }
        
        # Remove problematic characters that can cause Moodle API errors
        clean_user['firstname'] = re.sub(r'[<>"\']', '', clean_user['firstname'])
        clean_user['lastname'] = re.sub(r'[<>"\']', '', clean_user['lastname'])
        clean_user['username'] = re.sub(r'[<>"\']', '', clean_user['username'])
        
        # Additional cleaning for Moodle compatibility
        clean_user['firstname'] = re.sub(r'[^\w\s\-\.]', '', clean_user['firstname'])
        clean_user['lastname'] = re.sub(r'[^\w\s\-\.]', '', clean_user['lastname'])
        clean_user['username'] = re.sub(r'[^a-zA-Z0-9_\-\.]', '', clean_user['username'])
        
        # Format username to match what users receive in their email credentials
        original_username = clean_user['username']
        clean_user['username'] = self.format_username_for_moodle(clean_user['username'])
        
        if original_username != clean_user['username']:
            self.log_message(f"Fixed username for Moodle compatibility: {original_username} -> {clean_user['username']}", 'WARNING')
        
        # Properly capitalize names (first letter uppercase, rest lowercase)
        clean_user['firstname'] = clean_user['firstname'].title()
        clean_user['lastname'] = clean_user['lastname'].title()
        
        # Ensure names don't contain only numbers or special characters
        if re.match(r'^[\d\s\-\.]+$', clean_user['firstname']):
            clean_user['firstname'] = 'Student'
        if re.match(r'^[\d\s\-\.]+$', clean_user['lastname']):
            clean_user['lastname'] = 'User'
        
        # Ensure required fields are not empty
        if not clean_user['firstname']:
            clean_user['firstname'] = 'Student'
        if not clean_user['lastname']:
            clean_user['lastname'] = 'User'
        if not clean_user['username']:
            return None  # Skip users with empty usernames
        
        # Validate email format
        email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        if not re.match(email_pattern, clean_user['email']):
            self.log_message(f"Warning: Invalid email format for user {clean_user['username']}: {clean_user['email']}", 'WARNING')
            return None
        
        # Check for problematic email domains or characters
        if '..' in clean_user['email'] or ' ' in clean_user['email']:
            self.log_message(f"Warning: Email contains invalid characters for user {clean_user['username']}: {clean_user['email']}", 'WARNING')
            return None
        
        # Additional Moodle-specific validations
        if len(clean_user['username']) < 2:
            self.log_message(f"Warning: Username too short for user {clean_user['username']}", 'WARNING')
            return None
        
        # Ensure username doesn't start with a number (Moodle requirement)
        if clean_user['username'][0].isdigit():
            clean_user['username'] = f"user_{clean_user['username']}"
            self.log_message(f"Fixed username starting with number: {user['username']} -> {clean_user['username']}", 'WARNING')
        
        # Ensure username is not empty after cleaning
        if not clean_user['username']:
            self.log_message(f"Warning: Username is empty after cleaning for user {user.get('username', 'unknown')}", 'WARNING')
            return None
        
        # Use password from CSV, but ensure it meets Moodle requirements
        # If password is too short or doesn't meet requirements, generate a secure password
        password_pattern = r'^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).{8,}$'
        if len(clean_user['password']) < 8 or not re.match(password_pattern, clean_user['password']):
            # Generate a secure password based on user data
            secure_password = self.generate_secure_password(clean_user)
            self.log_message(f"Warning: Password doesn't meet requirements for user {clean_user['username']}, using secure password", 'WARNING')
            clean_user['password'] = secure_password
        
        # Ensure email is lowercase
        clean_user['email'] = clean_user['email'].lower()
        
        return clean_user
    
    def generate_secure_password(self, user: Dict) -> str:
        """Generate a secure password that meets Moodle's strict requirements
        Uses the user's lastname as base and adds required characters
        This function MUST match the PHP generateSecurePasswordForMoodle() function
        """
        # Get user's lastname as base
        lastname = user.get('lastname', 'User').strip()
        
        # Ensure lastname is at least 3 characters
        if len(lastname) < 3:
            lastname = lastname + "123"
        
        # Create a secure password that meets ALL Moodle requirements:
        # - Minimum 8 characters (preferably 12+)
        # - At least one uppercase letter
        # - At least one lowercase letter  
        # - At least one number
        # - At least one special character
        
        # Start with the full lastname (this makes it more personal and memorable)
        password = lastname.lower()
        
        # Add required uppercase letter (use first letter of lastname capitalized)
        password += lastname[0].upper()
        
        # Add required numbers (use a consistent pattern based on lastname length)
        number_suffix = str(len(lastname)) + str(len(lastname) + 1)
        password += number_suffix
        
        # Add required special character (use a consistent one)
        password += "!"
        
        # If password is still too short, add more characters
        if len(password) < 12:
            additional_chars = 12 - len(password)
            # Add more numbers to reach 12 characters
            for i in range(additional_chars):
                password += str(i + 2)  # Add sequential numbers
        
        # Ensure it meets all requirements
        if not self.validate_password_requirements(password):
            # If it doesn't meet requirements, create a guaranteed valid password
            password = self.create_guaranteed_valid_password(lastname)
        
        return password
    
    def validate_password_requirements(self, password: str) -> bool:
        """Validate that password meets Moodle requirements"""
        if len(password) < 8:
            return False
        
        has_upper = any(c.isupper() for c in password)
        has_lower = any(c.islower() for c in password)
        has_digit = any(c.isdigit() for c in password)
        has_special = any(c in "!@#$%^&*()_+-=[]{}|;:,.<>?" for c in password)
        
        return has_upper and has_lower and has_digit and has_special
    
    def create_guaranteed_valid_password(self, lastname: str) -> str:
        """Create a password that is guaranteed to meet all requirements"""
        import random
        import string
        
        # Template: [lastname][UPPER][lower][digit][special][random]
        password = lastname[:3].lower()  # First 3 chars of lastname
        password += random.choice(string.ascii_uppercase)  # Uppercase
        password += random.choice(string.ascii_lowercase)  # Lowercase
        password += random.choice(string.digits)  # Digit
        password += random.choice("!@#$%^&*")  # Special char
        
        # Add random characters to reach 12+ length
        for _ in range(7):  # Total length will be 12
            password += random.choice(string.ascii_letters + string.digits)
        
        return password
    
    def diagnose_invalid_parameter_error(self, users: List[Dict], api_response: Dict) -> None:
        """Diagnose specific issues causing invalid parameter errors"""
        self.log_message("🔍 Diagnosing invalid parameter error...", 'WARNING')
        
        # Check for common issues
        issues_found = []
        
        for i, user in enumerate(users):
            user_issues = []
            
            # Check username issues
            username = user.get('username', '')
            if not username or len(username) < 2:
                user_issues.append(f"Invalid username: '{username}'")
            elif username.startswith('0') and not username.startswith('user_'):
                user_issues.append(f"Username starts with 0: '{username}'")
            
            # Check email issues
            email = user.get('email', '')
            if not email or '@' not in email:
                user_issues.append(f"Invalid email: '{email}'")
            
            # Check name issues
            firstname = user.get('firstname', '')
            lastname = user.get('lastname', '')
            if not firstname or len(firstname.strip()) < 1:
                user_issues.append(f"Invalid firstname: '{firstname}'")
            if not lastname or len(lastname.strip()) < 1:
                user_issues.append(f"Invalid lastname: '{lastname}'")
            
            # Check password issues
            password = user.get('password', '')
            if not password or len(password) < 8:
                user_issues.append(f"Password too short: '{password}'")
            
            if user_issues:
                issues_found.append(f"User {i} ({username}): {', '.join(user_issues)}")
        
        if issues_found:
            self.log_message("🚨 Issues found in user data:", 'ERROR')
            for issue in issues_found:
                self.log_message(f"   - {issue}", 'ERROR')
        else:
            self.log_message("✅ No obvious data issues found", 'WARNING')
    
    def validate_user_data(self, user: Dict) -> Dict[str, Any]:
        """Enhanced user data validation with detailed feedback"""
        result = {'valid': True, 'errors': [], 'warnings': []}
        
        # Check required fields
        required_fields = ['username', 'email', 'firstname', 'lastname']
        for field in required_fields:
            if not user.get(field) or not str(user[field]).strip():
                result['valid'] = False
                result['errors'].append(f"Missing required field: {field}")
        
        if not result['valid']:
            return result
        
        # Validate email format
        email = user['email'].strip()
        email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        if not re.match(email_pattern, email):
            result['valid'] = False
            result['errors'].append(f"Invalid email format: {email}")
        
        # Check email length
        if len(email) > 100:
            result['valid'] = False
            result['errors'].append(f"Email too long (max 100 characters): {len(email)}")
        
        # Check for suspicious email patterns
        suspicious_patterns = [
            r'test@test\.com',
            r'example@example\.com',
            r'admin@admin\.com',
            r'user@user\.com'
        ]
        for pattern in suspicious_patterns:
            if re.search(pattern, email, re.IGNORECASE):
                result['warnings'].append(f"Suspicious email pattern detected: {email}")
        
        # Validate username format
        username = user['username'].strip()
        if len(username) < 2:
            result['valid'] = False
            result['errors'].append(f"Username too short (min 2 characters): {username}")
        
        if len(username) > 100:
            result['valid'] = False
            result['errors'].append(f"Username too long (max 100 characters): {username}")
        
        # Check for valid characters
        if not re.match(r'^[a-zA-Z0-9_-]+$', username):
            result['valid'] = False
            result['errors'].append(f"Username contains invalid characters (only letters, numbers, underscores, and hyphens allowed): {username}")
        
        # Check for reserved usernames
        reserved_usernames = ['admin', 'administrator', 'root', 'user', 'guest', 'test', 'moodle', 'system']
        if username.lower() in reserved_usernames:
            result['valid'] = False
            result['errors'].append(f"Username is reserved: {username}")
        
        # Validate name fields
        for field in ['firstname', 'lastname']:
            name = user[field].strip()
            if len(name) < 1:
                result['warnings'].append(f"{field} is empty, using default")
            elif len(name) > 100:
                result['warnings'].append(f"{field} too long (max 100 characters): {len(name)}")
        
        return result
    
    def health_check(self) -> Dict[str, Any]:
        """Comprehensive health check system"""
        health_status = {
            'timestamp': datetime.now().isoformat(),
            'moodle_connection': False,
            'database_connection': False,
            'api_responsive': False,
            'disk_space': False,
            'memory_usage': False,
            'cache_status': False,
            'overall_health': 'UNKNOWN'
        }
        
        try:
            # Test Moodle connection
            self.log_message("🔍 Testing Moodle connection...")
            health_status['moodle_connection'] = self.test_moodle_connection()
        except Exception as e:
            self.log_message(f"❌ Moodle connection failed: {e}", 'ERROR')
        
        try:
            # Test database connection (if available)
            # This would test the database connection if we had one
            health_status['database_connection'] = True  # Placeholder
        except Exception as e:
            self.log_message(f"❌ Database connection failed: {e}", 'ERROR')
        
        try:
            # Test API responsiveness
            response = self.session.get(
                self.moodle_config['api_url'],
                timeout=10,
                verify=self.config['verify_ssl']
            )
            health_status['api_responsive'] = response.status_code == 200
        except Exception as e:
            self.log_message(f"❌ API responsiveness test failed: {e}", 'ERROR')
        
        try:
            # Check disk space
            disk_usage = shutil.disk_usage('/')
            free_space_gb = disk_usage.free / (1024**3)
            health_status['disk_space'] = free_space_gb > 1.0  # At least 1GB free
            health_status['disk_free_gb'] = round(free_space_gb, 2)
        except Exception as e:
            self.log_message(f"❌ Disk space check failed: {e}", 'ERROR')
        
        try:
            # Check memory usage (basic)
            import psutil
            memory = psutil.virtual_memory()
            health_status['memory_usage'] = memory.percent < 90  # Less than 90% used
            health_status['memory_percent'] = memory.percent
        except ImportError:
            # psutil not available, skip memory check
            health_status['memory_usage'] = True
            health_status['memory_percent'] = 'N/A'
        except Exception as e:
            self.log_message(f"❌ Memory check failed: {e}", 'ERROR')
        
        try:
            # Check cache status
            cache_file = self.cache_dir / 'moodle_courses.pkl'
            health_status['cache_status'] = cache_file.exists()
            if cache_file.exists():
                cache_age = time.time() - cache_file.stat().st_mtime
                health_status['cache_age_hours'] = round(cache_age / 3600, 2)
        except Exception as e:
            self.log_message(f"❌ Cache status check failed: {e}", 'ERROR')
        
        # Determine overall health
        critical_checks = ['moodle_connection', 'api_responsive']
        warning_checks = ['disk_space', 'memory_usage', 'cache_status']
        
        if all(health_status[check] for check in critical_checks):
            if all(health_status[check] for check in warning_checks):
                health_status['overall_health'] = 'HEALTHY'
            else:
                health_status['overall_health'] = 'WARNING'
        else:
            health_status['overall_health'] = 'CRITICAL'
        
        # Log health status
        self.log_message(f"🏥 Health Check: {health_status['overall_health']}")
        self.log_message(f"📊 Moodle: {'✅' if health_status['moodle_connection'] else '❌'}")
        self.log_message(f"📊 API: {'✅' if health_status['api_responsive'] else '❌'}")
        self.log_message(f"📊 Disk: {'✅' if health_status['disk_space'] else '❌'} ({health_status.get('disk_free_gb', 'N/A')}GB free)")
        self.log_message(f"📊 Memory: {'✅' if health_status['memory_usage'] else '❌'} ({health_status.get('memory_percent', 'N/A')}% used)")
        self.log_message(f"📊 Cache: {'✅' if health_status['cache_status'] else '❌'}")
        
        # Save health status to file for admin portal
        try:
            health_file = self.temp_dir / 'health_status.json'
            with open(health_file, 'w') as f:
                json.dump(health_status, f, indent=2)
        except Exception as e:
            self.log_message(f"⚠️ Failed to save health status: {e}", 'WARNING')
        
        return health_status
    
    def upload_with_progress(self, users: List[Dict]) -> Dict[str, Any]:
        """Upload with real-time progress updates"""
        if not users:
            return {'success': True, 'message': 'No users to upload'}
        
        total_users = len(users)
        batch_size = self.config['batch_size']
        batches = self.create_batches(users, batch_size)
        total_batches = len(batches)
        
        self.log_message(f"📤 Starting upload of {total_users} users in {total_batches} batches...")
        
        successful_uploads = 0
        failed_uploads = 0
        
        for i, batch in enumerate(batches):
            batch_num = i + 1
            progress = (batch_num / total_batches) * 100
            
            self.log_message(f"📤 Uploading batch {batch_num}/{total_batches} ({progress:.1f}%) - {len(batch)} users...")
            
            try:
                result = self.upload_to_moodle(batch)
                if result:
                    successful_uploads += len(batch)
                    self.log_message(f"✅ Batch {batch_num} completed successfully")
                else:
                    failed_uploads += len(batch)
                    self.log_message(f"❌ Batch {batch_num} failed", 'WARNING')
            except Exception as e:
                failed_uploads += len(batch)
                self.log_message(f"❌ Batch {batch_num} failed with exception: {e}", 'ERROR')
        
        self.log_message(f"🎯 Upload completed: {successful_uploads} successful, {failed_uploads} failed")
        return {
            'success': successful_uploads > 0,
            'successful_uploads': successful_uploads,
            'failed_uploads': failed_uploads
        }
    
    def filter_duplicate_emails(self, users: List[Dict]) -> List[Dict]:
        """Filter out users with duplicate emails (both in Moodle and within batch)"""
        filtered_users = []
        duplicate_count = 0
        seen_emails = set()  # Track emails within current batch
        
        for user in users:
            email = user.get('email', '').strip().lower()
            if not email:
                self.log_message(f"⚠️ User {user.get('username', 'unknown')} has empty email, skipping", 'WARNING')
                duplicate_count += 1
                continue
            
            # Validate email format first
            if not self.validate_email_format(email):
                self.log_message(f"⚠️ Invalid email format {email} for user {user.get('username', 'unknown')}, skipping", 'WARNING')
                duplicate_count += 1
                continue
            
            # Check for duplicates within current batch
            if email in seen_emails:
                self.log_message(f"⚠️ Duplicate email {email} within batch for user {user.get('username', 'unknown')}, skipping", 'WARNING')
                duplicate_count += 1
                continue
            
            # Check if email already exists in Moodle
            if self.email_exists_in_moodle(email):
                self.log_message(f"⚠️ Email {email} already exists in Moodle, skipping user {user.get('username', 'unknown')}", 'WARNING')
                duplicate_count += 1
            else:
                seen_emails.add(email)
                filtered_users.append(user)
        
        if duplicate_count > 0:
            self.log_message(f"📊 Filtered out {duplicate_count} users with duplicate/invalid emails")
        
        return filtered_users
    
    def validate_email_format(self, email: str) -> bool:
        """Comprehensive email validation"""
        import re
        
        # Basic format validation
        email_pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
        if not re.match(email_pattern, email):
            return False
        
        # Length validation
        if len(email) > 100:
            return False
        
        # Check for suspicious patterns
        suspicious_patterns = [
            r'\.{2,}',  # Multiple consecutive dots
            r'@.*@',    # Multiple @ symbols
            r'^\.',     # Starts with dot
            r'\.$',     # Ends with dot
            r'\.@',     # Dot before @
            r'@\.',     # Dot after @
        ]
        
        for pattern in suspicious_patterns:
            if re.search(pattern, email):
                return False
        
        # Check for valid domain
        domain = email.split('@')[1]
        if len(domain) < 4:  # Minimum domain length
            return False
        
        return True
    
    def email_exists_in_moodle(self, email: str) -> bool:
        """Check if email already exists in Moodle"""
        try:
            post_data = {
                'wstoken': self.moodle_config['api_token'],
                'wsfunction': 'core_user_get_users_by_field',
                'moodlewsrestformat': 'json',
                'field': 'email',
                'values[0]': email
            }
            
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=10,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            # If result is a list with users, email exists
            if isinstance(result, list) and len(result) > 0:
                return True
            
            return False
            
        except Exception as e:
            self.log_message(f"⚠️ Error checking email {email}: {e}", 'WARNING')
            # If we can't check, assume it doesn't exist to avoid blocking valid users
            return False
    
    def upload_to_moodle(self, users: List[Dict[str, Any]]) -> Dict[str, Any]:
        """Upload users to Moodle with enhanced error handling and metrics"""
        if not users:
            self.log_message("No users to upload")
            return {'success': True, 'message': 'No users to upload'}
        
        # Track API call
        if self.metrics:
            self.metrics.api_calls_made += 1
        
        # Clean and validate all users with comprehensive validation
        clean_users = []
        validation_errors = []
        
        for i, user in enumerate(users):
            # First, validate the raw user data
            validation_result = self.validate_user_data(user)
            if not validation_result['valid']:
                error_msg = f"User {i} validation failed: {', '.join(validation_result['errors'])}"
                self.log_message(f"❌ {error_msg}", 'ERROR')
                validation_errors.append(error_msg)
                if self.metrics:
                    self.metrics.users_with_errors += 1
                    self.metrics.errors_by_type['validation_failed'] = self.metrics.errors_by_type.get('validation_failed', 0) + 1
                continue
            
            # Then clean the user data
            clean_user = self.clean_user_data(user)
            if clean_user:
                # Final validation of cleaned data
                final_validation = self.validate_user_data(clean_user)
                if final_validation['valid']:
                    clean_users.append(clean_user)
                else:
                    error_msg = f"User {i} failed final validation: {', '.join(final_validation['errors'])}"
                    self.log_message(f"❌ {error_msg}", 'ERROR')
                    validation_errors.append(error_msg)
                    if self.metrics:
                        self.metrics.users_with_errors += 1
                        self.metrics.errors_by_type['final_validation_failed'] = self.metrics.errors_by_type.get('final_validation_failed', 0) + 1
            else:
                error_msg = f"User {i} failed data cleaning"
                self.log_message(f"❌ {error_msg}", 'ERROR')
                validation_errors.append(error_msg)
                if self.metrics:
                    self.metrics.users_with_errors += 1
                    self.metrics.errors_by_type['cleaning_failed'] = self.metrics.errors_by_type.get('cleaning_failed', 0) + 1
        
        if not clean_users:
            self.log_message("No valid users to upload after validation and cleaning")
            if validation_errors:
                self.log_message(f"Validation errors: {len(validation_errors)}", 'ERROR')
                for error in validation_errors[:5]:  # Show first 5 errors
                    self.log_message(f"  - {error}", 'ERROR')
            return {'success': False, 'message': 'No valid users to upload', 'errors': validation_errors}
        
        # Log validation summary
        self.log_message(f"✅ Data validation passed: {len(clean_users)}/{len(users)} users valid")
        if validation_errors:
            self.log_message(f"⚠️ {len(validation_errors)} users failed validation", 'WARNING')
        
        # Prepare Moodle API request
        post_data = {
            'wstoken': self.moodle_config['api_token'],
            'wsfunction': self.config['moodle_ws_function'],
            'moodlewsrestformat': 'json'
        }
        
        # Add users data in the correct format for Moodle API
        for index, user in enumerate(clean_users):
            post_data[f"users[{index}][username]"] = user['username']
            post_data[f"users[{index}][firstname]"] = user['firstname']
            post_data[f"users[{index}][lastname]"] = user['lastname']
            post_data[f"users[{index}][email]"] = user['email']
            post_data[f"users[{index}][password]"] = user['password']
            post_data[f"users[{index}][auth]"] = user['auth']
        
        try:
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=60,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            if 'exception' in result:
                # Track error in metrics
                if self.metrics:
                    error_type = result.get('errorcode', 'unknown_error')
                    self.metrics.errors_by_type[error_type] = self.metrics.errors_by_type.get(error_type, 0) + 1
                
                # Enhanced error logging with context
                timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
                user_count = len(clean_users)
                self.log_message(f"Moodle API Error Response: {json.dumps(result)}", 'ERROR')
                self.log_message(f"Post data sent: {json.dumps(post_data)}", 'ERROR')
                self.log_message(f"Number of users being uploaded: {user_count}", 'ERROR')
                
                # Log each user's data for debugging
                for index, user in enumerate(clean_users):
                    self.log_message(f"User {index} data: {json.dumps(user)}", 'ERROR')
                
                # Check for specific error types with enhanced messages
                if 'errorcode' in result:
                    error_code = result['errorcode']
                    if error_code == 'accesscontrol':
                        raise Exception(f"Moodle API Access Control Error: The API token doesn't have permission to create users. Please check the token permissions in Moodle admin panel. Attempted to create {user_count} users at {timestamp}.")
                    elif error_code == 'invalidparameter':
                        # Run diagnostic to identify specific issues
                        self.diagnose_invalid_parameter_error(clean_users, result)
                        
                        detailed_message = f"Moodle API Invalid Parameter Error: {result['message']}"
                        detailed_message += f" | Attempted to create {user_count} users at {timestamp}"
                        detailed_message += f" | Check user data for invalid characters, empty fields, or format issues"
                        raise Exception(detailed_message)
                    else:
                        raise Exception(f"Moodle API Error ({error_code}): {result['message']} | Attempted to create {user_count} users at {timestamp}")
                else:
                    raise Exception(f"Moodle API Error: {result['message']} | Attempted to create {user_count} users at {timestamp}")
            
            # Check for warnings
            if 'warnings' in result and result['warnings']:
                for warning in result['warnings']:
                    self.log_message(f"Moodle Warning: {warning['message']}", 'WARNING')
            
            return result
            
        except requests.exceptions.RequestException as e:
            raise Exception(f"Moodle HTTP Error: {e}")
    
    def test_moodle_connection(self) -> bool:
        """Test Moodle API connection and permissions"""
        self.log_message("Testing Moodle API connection...")
        
        # Test basic connection
        post_data = {
            'wstoken': self.moodle_config['api_token'],
            'wsfunction': 'core_webservice_get_site_info',
            'moodlewsrestformat': 'json'
        }
        
        try:
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=10,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            if 'exception' in result:
                raise Exception(f"Moodle API Error: {result['message']}")
            
            self.log_message(f"[SUCCESS] Moodle connection successful. Site: {result.get('sitename', 'Unknown')}")
            
            # Test user creation permissions
            self.log_message("Testing user creation permissions...")
            test_user = {
                'username': f'test_user_{int(time.time())}',
                'firstname': 'Test',
                'lastname': 'User',
                'email': 'test@example.com',
                'password': 'TestPassword123',
                'auth': 'manual'
            }
            
            post_data = {
                'wstoken': self.moodle_config['api_token'],
                'wsfunction': 'core_user_create_users',
                'moodlewsrestformat': 'json',
                'users[0][username]': test_user['username'],
                'users[0][firstname]': test_user['firstname'],
                'users[0][lastname]': test_user['lastname'],
                'users[0][email]': test_user['email'],
                'users[0][password]': test_user['password'],
                'users[0][auth]': test_user['auth']
            }
            
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=10,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            if 'exception' in result:
                if 'errorcode' in result and result['errorcode'] == 'accesscontrol':
                    raise Exception("❌ User creation permission denied. The API token needs 'core_user_create_users' permission.")
                else:
                    raise Exception(f"User creation test failed: {result['message']}")
            
            self.log_message("[SUCCESS] User creation permissions verified")
            
            # Clean up test user
            if result and len(result) > 0 and 'id' in result[0]:
                user_id = result[0]['id']
                delete_data = {
                    'wstoken': self.moodle_config['api_token'],
                    'wsfunction': 'core_user_delete_users',
                    'moodlewsrestformat': 'json',
                    'userids[0]': user_id
                }
                
                self.session.post(
                    self.moodle_config['api_url'],
                    data=delete_data,
                    timeout=10,
                    verify=self.config['verify_ssl']
                )
                
                self.log_message("[SUCCESS] Test user cleaned up")
            
            return True
            
        except requests.exceptions.RequestException as e:
            raise Exception(f"Moodle connection failed: {e}")
    
    def get_moodle_user_id(self, username: str) -> Optional[int]:
        """Get Moodle user ID by username"""
        post_data = {
            'wstoken': self.moodle_config['api_token'],
            'wsfunction': 'core_user_get_users_by_field',
            'moodlewsrestformat': 'json',
            'field': 'username',
            'values[0]': username
        }
        
        try:
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=10,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            # Handle None or empty response
            if not result:
                return None
            
            # Handle error response
            if isinstance(result, dict) and 'exception' in result:
                return None
            
            # Handle successful response (should be a list)
            if isinstance(result, list) and len(result) > 0:
                return result[0]['id']
            
            return None
            
        except requests.exceptions.RequestException:
            return None
    
    def get_moodle_courses(self) -> List[Dict[str, Any]]:
        """Get Moodle courses (subjects)"""
        post_data = {
            'wstoken': self.moodle_config['api_token'],
            'wsfunction': 'core_course_get_courses',
            'moodlewsrestformat': 'json'
        }
        
        try:
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=30,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            if 'exception' in result:
                raise Exception(f"Moodle API Error: {result['message']}")
            
            return result
            
        except requests.exceptions.RequestException as e:
            raise Exception(f"Moodle HTTP Error: {e}")
    
    def get_course_id_by_shortname(self, shortname: str) -> Optional[int]:
        """Get course ID by shortname"""
        try:
            courses = self.get_moodle_courses()
            for course in courses:
                if course.get('shortname') == shortname:
                    return course.get('id')
            return None
        except Exception as e:
            self.log_message(f"Error getting course ID for shortname {shortname}: {e}", 'ERROR')
            return None
    
    def enroll_user_in_course(self, user_id: int, course_id: int) -> bool:
        """Enroll user in Moodle course"""
        post_data = {
            'wstoken': self.moodle_config['api_token'],
            'wsfunction': 'enrol_manual_enrol_users',
            'moodlewsrestformat': 'json',
            'enrolments[0][roleid]': 5,  # Student role ID (usually 5)
            'enrolments[0][userid]': user_id,
            'enrolments[0][courseid]': course_id
        }
        
        try:
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=30,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            # Handle None response (successful enrollment returns None)
            if result is None:
                return True
            
            # Handle error response
            if isinstance(result, dict) and 'exception' in result:
                raise Exception(f"Moodle API Error: {result['message']}")
            
            # Any other response is considered success
            return True
            
        except requests.exceptions.RequestException as e:
            raise Exception(f"Moodle HTTP Error: {e}")
    
    def enroll_users_bulk(self, enrollments: List[Dict]) -> Dict[str, Any]:
        """Enroll multiple users in multiple courses at once for better performance"""
        if not enrollments:
            return {'success': True, 'message': 'No enrollments to process'}
        
        # Track API call
        if self.metrics:
            self.metrics.api_calls_made += 1
        
        # Prepare bulk enrollment data
        post_data = {
            'wstoken': self.moodle_config['api_token'],
            'wsfunction': 'enrol_manual_enrol_users',
            'moodlewsrestformat': 'json'
        }
        
        # Add all enrollments to the request
        for i, enrollment in enumerate(enrollments):
            post_data[f'enrolments[{i}][roleid]'] = 5  # Student role ID
            post_data[f'enrolments[{i}][userid]'] = enrollment['user_id']
            post_data[f'enrolments[{i}][courseid]'] = enrollment['course_id']
        
        try:
            self.log_message(f"📚 Bulk enrolling {len(enrollments)} users in courses...")
            response = self.session.post(
                self.moodle_config['api_url'],
                data=post_data,
                timeout=60,
                verify=self.config['verify_ssl']
            )
            response.raise_for_status()
            
            result = response.json()
            
            # Handle None response (successful enrollment returns None)
            if result is None:
                self.log_message(f"✅ Bulk enrollment successful: {len(enrollments)} enrollments")
                return {'success': True, 'enrollments': len(enrollments)}
            
            # Handle error response
            if isinstance(result, dict) and 'exception' in result:
                error_msg = f"Moodle API Error: {result['message']}"
                self.log_message(f"❌ {error_msg}", 'ERROR')
                raise Exception(error_msg)
            
            # Any other response is considered success
            self.log_message(f"✅ Bulk enrollment completed: {len(enrollments)} enrollments")
            return {'success': True, 'enrollments': len(enrollments)}
            
        except requests.exceptions.RequestException as e:
            error_msg = f"Moodle HTTP Error: {e}"
            self.log_message(f"❌ {error_msg}", 'ERROR')
            raise Exception(error_msg)
    
    def enroll_users_in_subjects(self, users: List[Dict[str, Any]]) -> Dict[str, int]:
        """Enroll users in their selected subjects with comprehensive error handling and debugging"""
        import traceback
        
        self.log_message("Starting course enrollment process...")
        
        # Get all Moodle courses
        moodle_courses = self.get_moodle_courses()
        self.log_message(f"Retrieved {len(moodle_courses)} courses from Moodle")
        
        # Use the configured subject to course shortname mapping
        subject_to_course_mapping = SUBJECT_TO_COURSE_MAPPING.copy()
        
        # Log the mapping at startup for verification
        self.log_message("Subject to Course Mapping:")
        for subject_id, course_shortname in subject_to_course_mapping.items():
            if course_shortname and isinstance(course_shortname, str) and course_shortname.strip():
                self.log_message(f"  Subject ID {subject_id} -> Course Shortname {course_shortname}")
        
        # Problem 2: Automatic detection of all unique subject IDs from CSV
        all_csv_subject_ids = set()
        for user in users:
            if user.get('subject_ids'):
                all_csv_subject_ids.update(user['subject_ids'])
        
        self.log_message(f"Found {len(all_csv_subject_ids)} unique subject IDs in CSV: {sorted(all_csv_subject_ids)}")
        
        # Problem 2: Compare CSV subject IDs with configured mappings
        configured_subject_ids = set(subject_to_course_mapping.keys())
        missing_subject_ids = all_csv_subject_ids - configured_subject_ids
        
        if missing_subject_ids:
            self.log_message(f"⚠️ Missing Subject IDs: {sorted(missing_subject_ids)}", 'WARNING')
            self.log_message("These subject IDs are in the CSV but not configured in SUBJECT_TO_COURSE_MAPPING", 'WARNING')
        
        # Initialize counters for enrollment summary
        enrollment_count = 0
        enrollment_errors = 0
        enrollment_skipped = 0
        successful_enrollments = []
        error_details = []
        skipped_details = []
        
        for user in users:
            try:
                # Problem 5: Log each user's processing with their subject IDs
                user_subject_ids = user.get('subject_ids', [])
                self.log_message(f"Processing user {user['username']} with subject IDs: {user_subject_ids}")
                
                # Get Moodle user ID using original username (users exist in Moodle with original usernames)
                original_username = self.get_original_username(user['username'])
                moodle_user_id = self.get_moodle_user_id(original_username)
                if not moodle_user_id:
                    error_msg = f"Could not find Moodle user ID for {original_username} (formatted: {user['username']})"
                    self.log_message(f"Warning: {error_msg}", 'WARNING')
                    error_details.append(f"User {user['username']}: {error_msg}")
                    enrollment_errors += 1
                    continue
                
                # Enroll in subjects with granular error handling
                if user_subject_ids:
                    for subject_id in user_subject_ids:
                        try:
                            # Problem 1: Use .get() method to safely get course shortname
                            course_shortname = subject_to_course_mapping.get(subject_id)
                            
                            # Problem 1: Comprehensive None validation before string operations
                            if not course_shortname or not isinstance(course_shortname, str) or not course_shortname.strip():
                                # Problem 4: Change from errors to warnings for unmapped subjects
                                skip_msg = f"⚠️ Skipping - No mapping for subject ID {subject_id}"
                                self.log_message(skip_msg, 'WARNING')
                                skipped_details.append(f"User {user['username']}: Subject ID {subject_id} - No mapping")
                                enrollment_skipped += 1
                                continue
                            
                            # Get course ID from shortname
                            course_id = self.get_course_id_by_shortname(course_shortname)
                            if course_id:
                                # Actually enroll the user
                                enrollment_result = self.enroll_user_in_course(moodle_user_id, course_id)
                                if enrollment_result:
                                    enrollment_count += 1
                                    success_msg = f"[SUCCESS] Enrolled user {user['username']} in course {course_shortname} (ID: {course_id})"
                                    self.log_message(success_msg)
                                    successful_enrollments.append(f"User {user['username']}: Subject ID {subject_id} -> {course_shortname}")
                                else:
                                    error_msg = f"Failed to enroll user {user['username']} in course {course_shortname} (ID: {course_id})"
                                    self.log_message(f"❌ {error_msg}", 'ERROR')
                                    error_details.append(f"User {user['username']}: {error_msg}")
                                    enrollment_errors += 1
                            else:
                                error_msg = f"Course shortname {course_shortname} not found in Moodle"
                                self.log_message(f"❌ {error_msg}", 'ERROR')
                                error_details.append(f"User {user['username']}: Subject ID {subject_id} -> {error_msg}")
                                enrollment_errors += 1
                        
                        except Exception as subject_error:
                            # Problem 3: Granular try-catch blocks around subject enrollment
                            error_msg = f"Error enrolling user {user['username']} in subject {subject_id}: {str(subject_error)}"
                            self.log_message(f"❌ {error_msg}", 'ERROR')
                            # Problem 3: Add traceback logging for detailed error information
                            self.log_message(f"Traceback: {traceback.format_exc()}", 'ERROR')
                            error_details.append(f"User {user['username']}: Subject ID {subject_id} - {str(subject_error)}")
                            enrollment_errors += 1
                
            except Exception as user_error:
                # Problem 3: Granular error handling for user-level errors
                error_msg = f"Error processing user {user['username']}: {str(user_error)}"
                self.log_message(f"❌ {error_msg}", 'ERROR')
                # Problem 3: Add traceback logging for detailed error information
                self.log_message(f"Traceback: {traceback.format_exc()}", 'ERROR')
                error_details.append(f"User {user['username']}: {str(user_error)}")
                enrollment_errors += 1
        
        # Problem 3: Add enrollment summary with three categories
        self.log_message("=" * 60)
        self.log_message("ENROLLMENT SUMMARY")
        self.log_message("=" * 60)
        self.log_message(f"✅ Successful enrollments: {enrollment_count}")
        self.log_message(f"❌ Errors: {enrollment_errors}")
        self.log_message(f"⚠️ Skipped (no mapping): {enrollment_skipped}")
        self.log_message("=" * 60)
        
        # Log detailed breakdown if there are any issues
        if successful_enrollments:
            self.log_message("SUCCESSFUL ENROLLMENTS:")
            for success in successful_enrollments[:10]:  # Limit to first 10 for readability
                self.log_message(f"  ✅ {success}")
            if len(successful_enrollments) > 10:
                self.log_message(f"  ... and {len(successful_enrollments) - 10} more")
        
        if error_details:
            self.log_message("ERRORS:")
            for error in error_details[:10]:  # Limit to first 10 for readability
                self.log_message(f"  ❌ {error}")
            if len(error_details) > 10:
                self.log_message(f"  ... and {len(error_details) - 10} more")
        
        if skipped_details:
            self.log_message("SKIPPED (NO MAPPING):")
            for skipped in skipped_details[:10]:  # Limit to first 10 for readability
                self.log_message(f"  ⚠️ {skipped}")
            if len(skipped_details) > 10:
                self.log_message(f"  ... and {len(skipped_details) - 10} more")
        
        self.log_message("=" * 60)
        
        return {
            'enrollments': enrollment_count, 
            'errors': enrollment_errors, 
            'skipped': enrollment_skipped,
            'successful_details': successful_enrollments,
            'error_details': error_details,
            'skipped_details': skipped_details
        }
    
    def filter_existing_users(self, users: List[Dict[str, Any]]) -> List[Dict[str, Any]]:
        """Filter out users that already exist in Moodle"""
        new_users = []
        
        for user in users:
            # Use original username format
            check_username = user['username'].lower().strip()
            
            # Check if user exists by username
            moodle_user_id = self.get_moodle_user_id(check_username)
            
            # If user doesn't exist (None result), add to new users
            if moodle_user_id is None:
                new_users.append(user)
            else:
                self.log_message(f"User {check_username} already exists in Moodle, skipping")
        
        return new_users
    
    def sync_to_moodle(self) -> bool:
        """Main sync function with comprehensive metrics and progress tracking"""
        try:
            self.log_message("🚀 Starting Moodle sync with comprehensive metrics tracking...")
            
            # Run health check if enabled
            if self.config.get('enable_health_checks', True):
                health_start = time.time()
                health_status = self.health_check()
                if self.metrics:
                    self.metrics.performance_breakdown['health_check'] = time.time() - health_start
                
                if health_status['overall_health'] == 'CRITICAL':
                    self.log_message("❌ Critical health issues detected, aborting sync", 'ERROR')
                    return False
                elif health_status['overall_health'] == 'WARNING':
                    self.log_message("⚠️ Health warnings detected, continuing with caution", 'WARNING')
            
            # Test Moodle connection first
            connection_start = time.time()
            try:
                self.test_moodle_connection()
                if self.metrics:
                    self.metrics.performance_breakdown['connection_test'] = time.time() - connection_start
            except Exception as e:
                self.log_message(f"Moodle connection test failed: {e}", 'ERROR')
                raise e
            
            # Download CSV
            csv_start = time.time()
            self.log_message("📥 Downloading CSV from Skolo-Kine...")
            csv_data = self.download_csv()
            if self.metrics:
                self.metrics.performance_breakdown['csv_download'] = time.time() - csv_start
            self.log_message(f"✅ CSV downloaded successfully ({len(csv_data)} bytes) in {time.time() - csv_start:.2f} seconds")
            
            # Parse CSV
            parse_start = time.time()
            self.log_message("🔍 Parsing CSV data...")
            users = self.parse_csv(csv_data)
            if self.metrics:
                self.metrics.performance_breakdown['csv_parse'] = time.time() - parse_start
                self.metrics.users_processed = len(users)
            self.log_message(f"✅ Parsed {len(users)} users from CSV in {time.time() - parse_start:.2f} seconds")
            
            if not users:
                self.log_message("No users found in CSV, skipping upload")
                return True
            
            # Filter out existing users
            filter_start = time.time()
            self.log_message("🔍 Checking for existing users in Moodle...")
            new_users = self.filter_existing_users(users)
            if self.metrics:
                self.metrics.performance_breakdown['user_check'] = time.time() - filter_start
                self.metrics.users_skipped = len(users) - len(new_users)
            self.log_message(f"📊 Found {len(new_users)} new users to upload (skipped {len(users) - len(new_users)} existing users)")
            
            if not new_users:
                self.log_message("No new users to upload")
                # Continue to enrollment process even if no new users
            
            # Upload to Moodle using parallel processing (only if there are new users)
            if new_users:
                upload_start = time.time()
                self.log_message("🚀 Uploading new users to Moodle using parallel processing...")
                
                # Use parallel processing for better performance
                upload_result = self.upload_users_parallel(new_users)
                
                if self.metrics:
                    self.metrics.performance_breakdown['user_upload'] = time.time() - upload_start
                    self.metrics.users_created = upload_result.get('successful_uploads', 0)
                
                self.log_message(f"✅ Parallel upload completed: {upload_result.get('successful_uploads', 0)} successful, {upload_result.get('failed_uploads', 0)} failed")
            else:
                self.log_message("Skipping user upload (no new users)")
            
            # Enroll users in their selected subjects (both new and existing)
            if users:
                enrollment_start = time.time()
                self.log_message("📚 Starting enrollment process for all users...")
                enrollment_result = self.enroll_users_in_subjects(users)
                if self.metrics:
                    self.metrics.performance_breakdown['enrollment'] = time.time() - enrollment_start
                    self.metrics.enrollments_successful = enrollment_result['enrollments']
                    self.metrics.enrollments_failed = enrollment_result['errors']
                    self.metrics.enrollments_skipped = enrollment_result['skipped']
                self.log_message(f"📚 Enrollment completed: {enrollment_result['enrollments']} enrollments, {enrollment_result['errors']} errors, {enrollment_result['skipped']} skipped")
            
            # Log comprehensive metrics
            if self.metrics:
                self.metrics.log_metrics(self)
            
            self.log_message("Moodle sync completed successfully")
            return True
            
        except Exception as e:
            self.log_message(f"Sync failed: {e}", 'ERROR')
            return False
    
    def sync_with_retry(self) -> bool:
        """Retry mechanism"""
        for attempt in range(1, self.config['max_retries'] + 1):
            self.log_message(f"Attempt {attempt} of {self.config['max_retries']}")
            
            if self.sync_to_moodle():
                return True
            
            if attempt < self.config['max_retries']:
                self.log_message(f"Waiting {self.config['retry_delay']} seconds before retry...")
                time.sleep(self.config['retry_delay'])
        
        self.log_message("All retry attempts failed", 'ERROR')
        return False

def main():
    """Deprecated entrypoint delegating to unified bot"""
    logger.info("=== Moodle Sync Bot (legacy) delegating to unified_sync_bot ===")
    try:
        from unified_sync_bot import main as unified_main
        unified_main()
    except Exception as e:
        logger.error(f"Delegation to unified_sync_bot failed: {e}")
        sys.exit(1)

if __name__ == "__main__":
    main()
