<?php
/**
 * Shared utilities for cron jobs
 * Prevents function redeclaration conflicts
 */

// Prevent direct access
if (!defined('CRON_UTILS_LOADED')) {
    define('CRON_UTILS_LOADED', true);
    
    // Load required files
    require_once __DIR__ . '/../config.php';
    require_once __DIR__ . '/../includes/functions.php';
    
    // Set execution time limit
    set_time_limit(300); // 5 minutes
    
    // Ensure logs directory exists
    if (!is_dir(__DIR__ . '/../logs')) {
        mkdir(__DIR__ . '/../logs', 0755, true);
    }
    
    /**
     * Log message to file and output
     */
    function logMessage($message, $logFile = null) {
        if ($logFile === null) {
            $logFile = __DIR__ . '/../logs/cron.log';
        }
        
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[$timestamp] $message" . PHP_EOL;
        
        // Ensure log file directory exists
        $logDir = dirname($logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
        echo $logEntry;
    }
    
    /**
     * Get database connection with error handling
     */
    function getCronDB() {
        try {
            $pdo = getDB();
            if (!$pdo) {
                throw new Exception("Database connection failed");
            }
            return $pdo;
        } catch (Exception $e) {
            logMessage("ERROR: Database connection failed - " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Check if a process is running
     */
    function isProcessRunning($processName) {
        $output = [];
        $returnCode = 0;
        exec("ps aux | grep '$processName' | grep -v grep", $output, $returnCode);
        return !empty($output);
    }
    
    /**
     * Get system memory usage
     */
    function getMemoryUsage() {
        $memUsage = memory_get_usage(true);
        $memPeak = memory_get_peak_usage(true);
        return [
            'current' => $memUsage,
            'peak' => $memPeak,
            'current_mb' => round($memUsage / 1024 / 1024, 2),
            'peak_mb' => round($memPeak / 1024 / 1024, 2)
        ];
    }
    
    /**
     * Get disk usage
     */
    function getDiskUsage($path) {
        $total = disk_total_space($path);
        $free = disk_free_space($path);
        $used = $total - $free;
        
        return [
            'total' => $total,
            'free' => $free,
            'used' => $used,
            'total_gb' => round($total / 1024 / 1024 / 1024, 2),
            'free_gb' => round($free / 1024 / 1024 / 1024, 2),
            'used_gb' => round($used / 1024 / 1024 / 1024, 2),
            'usage_percent' => round(($used / $total) * 100, 2)
        ];
    }
    
    /**
     * Clean old log files
     */
    function cleanOldLogs($logDir, $maxAge = 30) {
        $cleaned = 0;
        $files = glob($logDir . '/*.log');
        
        foreach ($files as $file) {
            if (filemtime($file) < (time() - ($maxAge * 24 * 60 * 60))) {
                if (unlink($file)) {
                    $cleaned++;
                }
            }
        }
        
        return $cleaned;
    }
    
    /**
     * Send alert email
     */
    function sendAlertEmail($subject, $message) {
        try {
            $mail = new PHPMailer\PHPMailer\PHPMailer(true);
            
            // SMTP configuration
            $mail->isSMTP();
            $mail->Host = SMTP_HOST;
            $mail->SMTPAuth = true;
            $mail->Username = SMTP_USERNAME;
            $mail->Password = SMTP_PASSWORD;
            $mail->SMTPSecure = SMTP_ENCRYPTION;
            $mail->Port = SMTP_PORT;
            
            // Recipients
            $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
            $mail->addAddress(SMTP_FROM_EMAIL); // Send to admin
            
            // Content
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $message;
            
            $mail->send();
            return true;
        } catch (Exception $e) {
            logMessage("ERROR: Failed to send alert email - " . $e->getMessage());
            return false;
        }
    }
}
?>
