<?php
/**
 * Payment Reminder Cron Job
 * Runs daily to:
 * 1. Send payment reminders (27 days after last payment)
 * 2. Send warnings (3 days after reminder)
 * 3. Suspend accounts (when grace period expires)
 * 
 * Schedule: Run daily at 8:00 AM
 * Cron: 0 8 * * * /usr/bin/php /path/to/cron/payment_reminder_cron.php
 */

// Prevent direct web access
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from the command line.\n");
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/payment_reminder_manager.php';

// Set up logging
$log_file = __DIR__ . '/../logs/payment_reminder_cron.log';
function logMessage($message) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[{$timestamp}] {$message}\n";
    file_put_contents($log_file, $log_entry, FILE_APPEND);
    echo $log_entry;
}

logMessage("========================================");
logMessage("Payment Reminder Cron Job Started");
logMessage("========================================");

try {
    $manager = new PaymentReminderManager();
    
    // =====================================================
    // STEP 1: Send Initial Payment Reminders
    // =====================================================
    logMessage("\n--- Step 1: Checking for students needing payment reminders ---");
    
    $students_need_reminders = $manager->getStudentsNeedingReminders();
    logMessage("Found " . count($students_need_reminders) . " students needing reminders");
    
    $reminders_sent = 0;
    $reminders_failed = 0;
    
    foreach ($students_need_reminders as $student) {
        logMessage("Processing reminder for student: {$student['full_name']} (ID: {$student['student_id']})");
        
        $result = $manager->sendPaymentReminder($student['student_id'], 'initial_reminder');
        
        if ($result['success']) {
            $reminders_sent++;
            logMessage("✓ Reminder sent successfully");
        } else {
            $reminders_failed++;
            logMessage("✗ Failed to send reminder: " . ($result['error'] ?? 'Unknown error'));
        }
    }
    
    logMessage("Reminders Summary: {$reminders_sent} sent, {$reminders_failed} failed");
    
    // =====================================================
    // STEP 2: Send Warning Notifications
    // =====================================================
    logMessage("\n--- Step 2: Checking for students needing warnings ---");
    
    $students_need_warnings = $manager->getStudentsNeedingWarnings();
    logMessage("Found " . count($students_need_warnings) . " students needing warnings");
    
    $warnings_sent = 0;
    $warnings_failed = 0;
    
    foreach ($students_need_warnings as $student) {
        logMessage("Processing warning for student: {$student['full_name']} (ID: {$student['student_id']})");
        
        $result = $manager->sendPaymentReminder($student['student_id'], 'warning');
        
        if ($result['success']) {
            $warnings_sent++;
            logMessage("✓ Warning sent successfully");
        } else {
            $warnings_failed++;
            logMessage("✗ Failed to send warning: " . ($result['error'] ?? 'Unknown error'));
        }
    }
    
    logMessage("Warnings Summary: {$warnings_sent} sent, {$warnings_failed} failed");
    
    // =====================================================
    // STEP 3: Suspend Overdue Accounts
    // =====================================================
    logMessage("\n--- Step 3: Checking for students to suspend ---");
    
    $students_to_suspend = $manager->getStudentsToSuspend();
    logMessage("Found " . count($students_to_suspend) . " students to suspend");
    
    $suspended = 0;
    $suspension_failed = 0;
    
    foreach ($students_to_suspend as $student) {
        logMessage("Processing suspension for student: {$student['full_name']} (ID: {$student['student_id']})");
        
        // Suspend in Moodle
        $result = $manager->suspendInMoodle($student['student_id'], 'payment_overdue');
        
        if ($result['success']) {
            $suspended++;
            logMessage("✓ Student suspended in Moodle successfully");
            
            // Send suspension notification
            $notification_result = $manager->sendPaymentReminder($student['student_id'], 'suspension_notice');
            
            if ($notification_result['success']) {
                logMessage("✓ Suspension notification sent");
            } else {
                logMessage("⚠ Failed to send suspension notification");
            }
            
            // Update payment schedule
            $pdo = getDB();
            $pdo->prepare("
                UPDATE student_payment_schedules 
                SET payment_status = 'suspended', 
                    suspended_at = NOW(),
                    suspension_reason = 'payment_overdue'
                WHERE student_id = ?
            ")->execute([$student['student_id']]);
            
        } else {
            $suspension_failed++;
            logMessage("✗ Failed to suspend student: " . ($result['error'] ?? 'Unknown error'));
        }
    }
    
    logMessage("Suspensions Summary: {$suspended} suspended, {$suspension_failed} failed");
    
    // =====================================================
    // STEP 4: Update Payment Statuses
    // =====================================================
    logMessage("\n--- Step 4: Updating payment statuses ---");
    
    $pdo = getDB();
    
    // Mark overdue payments
    $stmt = $pdo->query("
        UPDATE student_payment_schedules 
        SET payment_status = 'overdue'
        WHERE payment_status IN ('active', 'pending')
          AND next_due_date < CURDATE()
          AND suspended_at IS NULL
    ");
    $overdue_count = $stmt->rowCount();
    logMessage("Marked {$overdue_count} accounts as overdue");
    
    // Update suspension dates
    $stmt = $pdo->query("
        UPDATE sps
        INNER JOIN tenant_payment_settings tps ON sps.academy_reference = tps.academy_reference
        SET sps.suspension_date = DATE_ADD(sps.next_due_date, INTERVAL tps.days_before_suspension DAY)
        WHERE sps.payment_status = 'overdue'
          AND sps.suspension_date IS NULL
    ");
    $suspension_dates_updated = $stmt->rowCount();
    logMessage("Updated {$suspension_dates_updated} suspension dates");
    
    // =====================================================
    // FINAL SUMMARY
    // =====================================================
    logMessage("\n========================================");
    logMessage("CRON JOB SUMMARY");
    logMessage("========================================");
    logMessage("Initial Reminders: {$reminders_sent} sent, {$reminders_failed} failed");
    logMessage("Warnings: {$warnings_sent} sent, {$warnings_failed} failed");
    logMessage("Suspensions: {$suspended} suspended, {$suspension_failed} failed");
    logMessage("Status Updates: {$overdue_count} overdue, {$suspension_dates_updated} dates updated");
    logMessage("========================================");
    logMessage("Payment Reminder Cron Job Completed Successfully");
    logMessage("========================================\n");
    
} catch (Exception $e) {
    logMessage("CRITICAL ERROR: " . $e->getMessage());
    logMessage("Stack trace: " . $e->getTraceAsString());
    
    // Send error notification to admin
    if (defined('SMTP_FROM_EMAIL')) {
        $error_subject = "Payment Cron Job Failed";
        $error_body = "The payment reminder cron job failed with the following error:\n\n" . $e->getMessage();
        @mail(SMTP_FROM_EMAIL, $error_subject, $error_body);
    }
    
    exit(1);
}

exit(0);

