<?php
/**
 * Log Rotation Cron Job
 * Archives old logs and prevents disk space issues
 */

require_once __DIR__ . '/cron_utils.php';

// Set execution time limit
set_time_limit(300); // 5 minutes

// Log file
$logFile = __DIR__ . '/../logs/rotation.log';

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}



logMessage("=== LOG ROTATION STARTED ===");

try {
    $logsDir = __DIR__ . '/../logs';
    $archiveDir = $logsDir . '/archive';
    
    // Create archive directory if it doesn't exist
    if (!is_dir($archiveDir)) {
        mkdir($archiveDir, 0755, true);
        logMessage("Created archive directory: $archiveDir");
    }
    
    // Get all log files
    $logFiles = glob($logsDir . '/*.log');
    $rotatedCount = 0;
    $totalSize = 0;
    
    foreach ($logFiles as $logFile) {
        $fileName = basename($logFile);
        $fileSize = filesize($logFile);
        $totalSize += $fileSize;
        
        // Skip if file is too small (less than 1MB)
        if ($fileSize < 1048576) {
            logMessage("Skipping $fileName (size: " . round($fileSize/1024, 2) . "KB - too small)");
            continue;
        }
        
        // Create archive filename with timestamp
        $archiveFileName = date('Y-m-d_H-i-s') . '_' . $fileName;
        $archivePath = $archiveDir . '/' . $archiveFileName;
        
        // Move log file to archive
        if (rename($logFile, $archivePath)) {
            logMessage("Rotated $fileName (size: " . round($fileSize/1024/1024, 2) . "MB) to $archiveFileName");
            $rotatedCount++;
            
            // Create new empty log file
            touch($logFile);
            chmod($logFile, 0644);
        } else {
            logMessage("ERROR: Failed to rotate $fileName");
        }
    }
    
    // Clean up old archives (older than 30 days)
    $archiveFiles = glob($archiveDir . '/*.log');
    $cleanedCount = 0;
    $cutoffTime = time() - (30 * 24 * 60 * 60); // 30 days ago
    
    foreach ($archiveFiles as $archiveFile) {
        if (filemtime($archiveFile) < $cutoffTime) {
            if (unlink($archiveFile)) {
                $cleanedCount++;
                logMessage("Cleaned up old archive: " . basename($archiveFile));
            }
        }
    }
    
    // Compress large archives
    $largeArchives = glob($archiveDir . '/*.log');
    foreach ($largeArchives as $archiveFile) {
        if (filesize($archiveFile) > 10485760) { // 10MB
            $compressedFile = $archiveFile . '.gz';
            if (function_exists('gzopen')) {
                $fp_in = fopen($archiveFile, 'rb');
                $fp_out = gzopen($compressedFile, 'wb9');
                
                if ($fp_in && $fp_out) {
                    while (!feof($fp_in)) {
                        gzwrite($fp_out, fread($fp_in, 1024 * 1024));
                    }
                    fclose($fp_in);
                    gzclose($fp_out);
                    
                    unlink($archiveFile);
                    logMessage("Compressed large archive: " . basename($archiveFile));
                }
            }
        }
    }
    
    logMessage("Log rotation completed:");
    logMessage("- Files rotated: $rotatedCount");
    logMessage("- Old archives cleaned: $cleanedCount");
    logMessage("- Total size processed: " . round($totalSize/1024/1024, 2) . "MB");
    
    // Check disk space after rotation
    $diskFree = disk_free_space($logsDir);
    $diskTotal = disk_total_space($logsDir);
    $diskUsed = $diskTotal - $diskFree;
    $diskPercent = round(($diskUsed / $diskTotal) * 100, 2);
    
    logMessage("Disk usage after rotation: {$diskPercent}%");
    
    if ($diskPercent > 85) {
        logMessage("WARNING: Disk usage is still high after rotation");
    }
    
} catch (Exception $e) {
    logMessage("ERROR: Log rotation failed: " . $e->getMessage());
    exit(1);
}

logMessage("=== LOG ROTATION COMPLETED ===");
?>
