-- =====================================================
-- STUDENT SELF-REGISTRATION - APPLICATION SYSTEM
-- =====================================================
-- Stores student applications before creating full accounts
-- Allows tracking, approval, and payment before activation
-- =====================================================

DROP TABLE IF EXISTS student_application_subjects;
DROP TABLE IF EXISTS student_applications;

-- Main applications table
CREATE TABLE IF NOT EXISTS student_applications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    -- School identification
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Student information
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    full_name VARCHAR(255) NOT NULL,
    email VARCHAR(191) NOT NULL,
    phone VARCHAR(20) NULL,
    date_of_birth DATE NULL,
    gender ENUM('male', 'female', 'other') NULL,
    
    -- Academic information
    grade_id INT NOT NULL,
    academic_year VARCHAR(20) NULL,
    
    -- Parent/Guardian information
    parent_name VARCHAR(255) NULL,
    parent_email VARCHAR(191) NULL,
    parent_phone VARCHAR(20) NULL,
    parent_relationship VARCHAR(50) DEFAULT 'parent',
    
    -- Application status
    application_status ENUM('draft', 'submitted', 'verified', 'approved', 'rejected', 'completed') DEFAULT 'submitted',
    payment_status ENUM('unpaid', 'pending', 'paid', 'failed') DEFAULT 'unpaid',
    
    -- Verification
    email_verification_token VARCHAR(64) NULL,
    email_verified TINYINT(1) DEFAULT 0,
    email_verified_at DATETIME NULL,
    
    -- Payment details
    payment_amount DECIMAL(10,2) NULL,
    payment_reference VARCHAR(100) NULL,
    payment_method VARCHAR(50) NULL,
    payment_date DATETIME NULL,
    
    -- Account creation
    student_id INT NULL COMMENT 'Created student ID after approval',
    username VARCHAR(50) NULL COMMENT 'Generated username',
    password_plain VARCHAR(255) NULL COMMENT 'Temporary plain password for email',
    
    -- Admin actions
    approved_by INT NULL COMMENT 'Admin who approved',
    approved_at DATETIME NULL,
    rejection_reason TEXT NULL,
    admin_notes TEXT NULL,
    
    -- Timestamps
    submitted_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    completed_at DATETIME NULL,
    
    -- Indexes
    INDEX idx_academy (academy_reference),
    INDEX idx_email (email),
    INDEX idx_status (application_status),
    INDEX idx_payment_status (payment_status),
    INDEX idx_email_token (email_verification_token),
    INDEX idx_submitted (submitted_at),
    INDEX idx_grade (grade_id),
    
    FOREIGN KEY (grade_id) REFERENCES grades(id) ON DELETE RESTRICT,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE SET NULL
    -- FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) ON DELETE CASCADE

) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Stores student self-registration applications';

-- Subject selections for applications
CREATE TABLE IF NOT EXISTS student_application_subjects (
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    application_id INT NOT NULL,
    subject_id INT NOT NULL,
    
    -- Metadata
    added_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_application (application_id),
    INDEX idx_subject (subject_id),
    
    UNIQUE KEY unique_app_subject (application_id, subject_id),
    
    FOREIGN KEY (application_id) REFERENCES student_applications(id) ON DELETE CASCADE,
    FOREIGN KEY (subject_id) REFERENCES subjects(id) ON DELETE CASCADE

) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Selected subjects for each application';

-- =====================================================
-- VERIFICATION QUERIES
-- =====================================================

-- Check table structure
DESCRIBE student_applications;
DESCRIBE student_application_subjects;

-- Count applications by status
-- SELECT application_status, COUNT(*) as count 
-- FROM student_applications 
-- GROUP BY application_status;

-- Recent applications
-- SELECT * FROM student_applications 
-- ORDER BY submitted_at DESC LIMIT 10;

-- =====================================================
-- ROLLBACK
-- =====================================================

-- DROP TABLE IF EXISTS student_application_subjects;
-- DROP TABLE IF EXISTS student_applications;

-- =====================================================
-- COMPLETE!
-- =====================================================

