-- =====================================================
-- MULTI-TENANT SETUP FOR SKOLO-KINE
-- =====================================================
-- This adds the final pieces needed for multi-tenancy
-- Your system already has 60% of multi-tenant structure!
-- =====================================================

-- Step 1: Create Tenant Settings Table
-- Stores school-specific configuration
CREATE TABLE IF NOT EXISTS `tenant_settings` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `academy_reference` VARCHAR(50) NOT NULL,
    `setting_key` VARCHAR(100) NOT NULL,
    `setting_value` TEXT,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY `uk_academy_setting` (`academy_reference`, `setting_key`),
    KEY `idx_academy_ref` (`academy_reference`),
    CONSTRAINT `fk_tenant_settings_academy` FOREIGN KEY (`academy_reference`) 
        REFERENCES `academy_references`(`reference_code`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Step 2: Insert Default Settings for Kine Academy
INSERT INTO `tenant_settings` (`academy_reference`, `setting_key`, `setting_value`) VALUES
('KINE', 'school_name', 'Kine Academy'),
('KINE', 'school_email', 'info@kineacademy.com'),
('KINE', 'school_phone', '+268 7687 3207'),
('KINE', 'school_address', 'Eswatini'),
('KINE', 'primary_color', '#4F46E5'),
('KINE', 'secondary_color', '#10B981'),
('KINE', 'academic_year', '2024/2025'),
('KINE', 'email_from_name', 'Kine Academy'),
('KINE', 'logo_url', '/images/kine-logo.png')
ON DUPLICATE KEY UPDATE `updated_at` = CURRENT_TIMESTAMP;

-- Step 3: Ensure all academy_reference columns have indexes
ALTER TABLE `students` ADD INDEX IF NOT EXISTS `idx_students_academy_ref` (`academy_reference`);
ALTER TABLE `payments` ADD INDEX IF NOT EXISTS `idx_payments_academy_ref` (`academy_reference`);
ALTER TABLE `platform_fees` ADD INDEX IF NOT EXISTS `idx_platform_fees_academy_ref` (`academy_reference`);

-- Step 4: Set default academy_reference for any NULL values
UPDATE `students` SET `academy_reference` = 'KINE' WHERE `academy_reference` IS NULL;
UPDATE `payments` SET `academy_reference` = 'KINE' WHERE `academy_reference` IS NULL;

-- Step 5: Verify multi-tenant setup
SELECT 
    'Academy References' as table_name,
    COUNT(*) as count,
    GROUP_CONCAT(reference_code) as schools
FROM academy_references
WHERE is_active = 1

UNION ALL

SELECT 
    'Students per School' as table_name,
    COUNT(*) as count,
    CONCAT(academy_reference, ': ', COUNT(*)) as schools
FROM students
WHERE academy_reference IS NOT NULL
GROUP BY academy_reference

UNION ALL

SELECT 
    'Payments per School' as table_name,
    COUNT(*) as count,
    CONCAT(academy_reference, ': ', COUNT(*)) as schools
FROM payments
WHERE academy_reference IS NOT NULL
GROUP BY academy_reference;

-- =====================================================
-- VERIFICATION QUERIES
-- =====================================================

-- Check if multi-tenancy is ready
SELECT 
    'Multi-Tenant Status' as check_type,
    CASE 
        WHEN EXISTS (SELECT 1 FROM tenant_settings LIMIT 1) AND
             EXISTS (SELECT 1 FROM academy_references WHERE is_active = 1)
        THEN '✅ READY'
        ELSE '❌ NOT READY'
    END as status;

COMMIT;

-- =====================================================
-- NEXT STEPS
-- =====================================================
-- 1. Run this script: mysql < database/multi_tenant_setup.sql
-- 2. Create includes/school_context.php (see implementation plan)
-- 3. Add requireSchool() to all admin pages
-- 4. Update queries to filter by getCurrentSchool()
-- =====================================================

