-- =====================================================
-- VERIFY SUBJECTS TABLE FIX
-- =====================================================
-- Run this to verify that the subjects table is fixed
-- and ready for Eswatini curriculum import
-- =====================================================

-- =====================================================
-- 1. CHECK TABLE STRUCTURE
-- =====================================================

SELECT '=== TABLE STRUCTURE ===' as section;
DESCRIBE subjects;

SELECT '=== TABLE INDEXES ===' as section;
SHOW INDEXES FROM subjects;

-- =====================================================
-- 2. CHECK CONSTRAINTS
-- =====================================================

SELECT '=== CONSTRAINT VERIFICATION ===' as section;

-- Check for UNIQUE constraint on 'name' (should NOT exist)
SELECT 
    'name column constraint' as check_item,
    CASE 
        WHEN COUNT(*) = 0 THEN '✅ CORRECT: No UNIQUE constraint on name (allows duplicates)'
        ELSE '❌ WRONG: UNIQUE constraint exists on name (prevents duplicates)'
    END as status,
    COUNT(*) as constraint_count
FROM information_schema.TABLE_CONSTRAINTS
WHERE TABLE_SCHEMA = DATABASE()
AND TABLE_NAME = 'subjects'
AND CONSTRAINT_NAME = 'name'
AND CONSTRAINT_TYPE = 'UNIQUE';

-- Check for UNIQUE constraint on 'code' (should exist)
SELECT 
    'code column constraint' as check_item,
    CASE 
        WHEN COUNT(*) > 0 THEN '✅ CORRECT: UNIQUE constraint exists on code'
        ELSE '❌ WRONG: No UNIQUE constraint on code'
    END as status,
    COUNT(*) as constraint_count
FROM information_schema.TABLE_CONSTRAINTS
WHERE TABLE_SCHEMA = DATABASE()
AND TABLE_NAME = 'subjects'
AND CONSTRAINT_NAME IN ('unique_code', 'code')
AND CONSTRAINT_TYPE = 'UNIQUE';

-- =====================================================
-- 3. CHECK FOR DUPLICATE CODES (SHOULD BE NONE)
-- =====================================================

SELECT '=== DUPLICATE CODE CHECK ===' as section;

SELECT 
    CASE 
        WHEN COUNT(*) = 0 THEN '✅ PASS: No duplicate codes found'
        ELSE CONCAT('❌ FAIL: ', COUNT(*), ' duplicate codes exist')
    END as duplicate_check_result
FROM (
    SELECT code, COUNT(*) as count
    FROM subjects
    GROUP BY code
    HAVING count > 1
) as duplicates;

-- Show any duplicate codes
SELECT 
    code,
    COUNT(*) as duplicate_count,
    GROUP_CONCAT(id) as subject_ids,
    GROUP_CONCAT(name SEPARATOR ' | ') as subject_names
FROM subjects
GROUP BY code
HAVING COUNT(*) > 1;

-- =====================================================
-- 4. CHECK FOR DUPLICATE NAMES (SHOULD BE ALLOWED)
-- =====================================================

SELECT '=== DUPLICATE NAME CHECK (Should be allowed) ===' as section;

-- Show subjects with same name but different codes
SELECT 
    name,
    COUNT(*) as occurrences,
    COUNT(DISTINCT code) as unique_codes,
    GROUP_CONCAT(DISTINCT code ORDER BY code SEPARATOR ', ') as codes
FROM subjects
GROUP BY name
HAVING COUNT(*) > 1
ORDER BY occurrences DESC
LIMIT 20;

-- =====================================================
-- 5. CHECK SUBJECT COUNTS BY GRADE PREFIX
-- =====================================================

SELECT '=== SUBJECTS BY GRADE PREFIX ===' as section;

SELECT 
    CASE 
        WHEN code LIKE 'G4-%' THEN 'Grade 4'
        WHEN code LIKE 'G5-%' THEN 'Grade 5'
        WHEN code LIKE 'G6-%' THEN 'Grade 6'
        WHEN code LIKE 'G7-%' THEN 'Grade 7'
        WHEN code LIKE 'F1-%' THEN 'Form 1'
        WHEN code LIKE 'F2-%' THEN 'Form 2'
        WHEN code LIKE 'F3-%' THEN 'Form 3'
        WHEN code LIKE 'F4-%' THEN 'Form 4'
        WHEN code LIKE 'F5-%' THEN 'Form 5'
        ELSE 'Other'
    END as grade_level,
    COUNT(*) as subject_count
FROM subjects
GROUP BY grade_level
ORDER BY 
    CASE grade_level
        WHEN 'Grade 4' THEN 1
        WHEN 'Grade 5' THEN 2
        WHEN 'Grade 6' THEN 3
        WHEN 'Grade 7' THEN 4
        WHEN 'Form 1' THEN 5
        WHEN 'Form 2' THEN 6
        WHEN 'Form 3' THEN 7
        WHEN 'Form 4' THEN 8
        WHEN 'Form 5' THEN 9
        ELSE 10
    END;

-- =====================================================
-- 6. CHECK GRADE-SUBJECT MAPPINGS
-- =====================================================

SELECT '=== GRADE-SUBJECT MAPPINGS ===' as section;

SELECT 
    g.name as grade_name,
    COUNT(gs.subject_id) as subjects_linked,
    CASE 
        WHEN COUNT(gs.subject_id) = 0 THEN '⚠️ WARNING: No subjects linked'
        WHEN COUNT(gs.subject_id) < 5 THEN '⚠️ WARNING: Very few subjects'
        ELSE '✅ OK'
    END as status
FROM grades g
LEFT JOIN grade_subject gs ON g.id = gs.grade_id
GROUP BY g.id, g.name
ORDER BY g.name;

-- =====================================================
-- 7. SHOW SAMPLE SUBJECTS
-- =====================================================

SELECT '=== SAMPLE SUBJECTS (Mathematics across grades) ===' as section;

SELECT 
    name,
    code,
    CASE 
        WHEN code LIKE 'G4-%' THEN 'Grade 4'
        WHEN code LIKE 'G5-%' THEN 'Grade 5'
        WHEN code LIKE 'G6-%' THEN 'Grade 6'
        WHEN code LIKE 'G7-%' THEN 'Grade 7'
        WHEN code LIKE 'F1-%' THEN 'Form 1'
        WHEN code LIKE 'F2-%' THEN 'Form 2'
        WHEN code LIKE 'F3-%' THEN 'Form 3'
        WHEN code LIKE 'F4-%' THEN 'Form 4'
        WHEN code LIKE 'F5-%' THEN 'Form 5'
        ELSE 'Other'
    END as grade_level
FROM subjects
WHERE name LIKE '%Mathematics%'
OR name = 'Mathematics'
ORDER BY code;

-- =====================================================
-- 8. TOTAL SUBJECT COUNT
-- =====================================================

SELECT '=== TOTAL COUNTS ===' as section;

SELECT 
    COUNT(*) as total_subjects,
    COUNT(DISTINCT name) as unique_names,
    COUNT(DISTINCT code) as unique_codes,
    CASE 
        WHEN COUNT(*) = COUNT(DISTINCT code) THEN '✅ All codes are unique'
        ELSE '❌ Duplicate codes exist'
    END as code_uniqueness
FROM subjects;

-- =====================================================
-- FINAL SUMMARY
-- =====================================================

SELECT '=== FINAL SUMMARY ===' as section;

SELECT 
    CASE 
        WHEN (
            -- No UNIQUE on name
            (SELECT COUNT(*) FROM information_schema.TABLE_CONSTRAINTS
             WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'subjects'
             AND CONSTRAINT_NAME = 'name' AND CONSTRAINT_TYPE = 'UNIQUE') = 0
            AND
            -- UNIQUE exists on code
            (SELECT COUNT(*) FROM information_schema.TABLE_CONSTRAINTS
             WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'subjects'
             AND CONSTRAINT_NAME IN ('unique_code', 'code') AND CONSTRAINT_TYPE = 'UNIQUE') > 0
            AND
            -- No duplicate codes
            (SELECT COUNT(*) FROM (
                SELECT code FROM subjects GROUP BY code HAVING COUNT(*) > 1
            ) as dups) = 0
        ) THEN '✅✅✅ ALL CHECKS PASSED - Ready for curriculum import!'
        ELSE '❌ SOME CHECKS FAILED - Review results above'
    END as overall_status;

-- =====================================================
-- NEXT STEPS
-- =====================================================

SELECT '=== NEXT STEPS ===' as section;

SELECT 
    CASE 
        WHEN (SELECT COUNT(*) FROM subjects WHERE code LIKE 'G4-%') = 0 
        THEN 'Re-import ESWATINI_CURRICULUM_SUBJECTS.sql to add all subjects'
        ELSE 'Subjects already imported. Check if mappings are complete.'
    END as action_needed;

-- =====================================================

