# Skolo-Kine System Functions Documentation

## Overview
This document provides comprehensive documentation for all PHP functions in the Skolo-Kine learning management system. Each function is documented with its purpose, parameters, return values, and usage examples.

## Table of Contents
1. [Session Management](#session-management)
2. [CSRF Protection](#csrf-protection)
3. [Student Management](#student-management)
4. [Moodle Integration](#moodle-integration)
5. [Payment System](#payment-system)
6. [Email System](#email-system)
7. [Admin Functions](#admin-functions)
8. [Dashboard Functions](#dashboard-functions)
9. [Subject Management](#subject-management)
10. [Utility Functions](#utility-functions)

---

## Session Management

### `startSession()`
**Purpose**: Ensures a PHP session is started if not already active.
**Parameters**: None
**Returns**: void
**Usage**: Called automatically by other functions that need session access.

```php
startSession(); // Starts session if not already started
```

### `generateCSRFToken()`
**Purpose**: Generates a CSRF token for form protection.
**Parameters**: None
**Returns**: string - 64-character hex token
**Usage**: Used in forms to prevent CSRF attacks.

```php
$token = generateCSRFToken();
echo '<input type="hidden" name="csrf_token" value="' . $token . '">';
```

### `validateCSRFToken($token)`
**Purpose**: Validates a CSRF token against the session token.
**Parameters**: 
- `$token` (string) - The token to validate
**Returns**: bool - true if valid, false otherwise
**Usage**: Used to validate form submissions.

```php
if (validateCSRFToken($_POST['csrf_token'])) {
    // Process form
}
```

---

## Student Management

### `getStudentCount()`
**Purpose**: Gets the total number of students in the system.
**Parameters**: None
**Returns**: int - Number of students
**Usage**: Used for statistics and limit checking.

```php
$count = getStudentCount();
echo "Total students: $count";
```

### `isStudentLimitReached()`
**Purpose**: Checks if the system has reached the maximum student limit.
**Parameters**: None
**Returns**: bool - true if limit reached, false otherwise
**Usage**: Used before creating new students.

```php
if (isStudentLimitReached()) {
    throw new Exception('Student limit reached');
}
```

### `getGrades()`
**Purpose**: Retrieves all grades from the database.
**Parameters**: None
**Returns**: array - Array of grade objects
**Usage**: Used in forms and displays.

```php
$grades = getGrades();
foreach ($grades as $grade) {
    echo $grade['name'];
}
```

### `getSubjectsForGrade($gradeId)`
**Purpose**: Gets all subjects assigned to a specific grade.
**Parameters**: 
- `$gradeId` (int) - The grade ID
**Returns**: array - Array of subject objects
**Usage**: Used in subject selection forms.

```php
$subjects = getSubjectsForGrade(1);
```

### `getAllSubjects()`
**Purpose**: Retrieves all subjects from the database.
**Parameters**: None
**Returns**: array - Array of all subjects
**Usage**: Used in admin interfaces and reports.

```php
$subjects = getAllSubjects();
```

### `getStudents($limit, $offset, $includeSuspended)`
**Purpose**: Retrieves students with optional pagination and filtering.
**Parameters**: 
- `$limit` (int|null) - Maximum number of students to return
- `$offset` (int) - Number of students to skip
- `$includeSuspended` (bool) - Whether to include suspended students
**Returns**: array - Array of student objects
**Usage**: Used in admin dashboards and reports.

```php
$students = getStudents(20, 0, false); // Get first 20 active students
```

### `getStudentById($id)`
**Purpose**: Retrieves a specific student by ID.
**Parameters**: 
- `$id` (int) - Student ID
**Returns**: array|false - Student object or false if not found
**Usage**: Used for student details and editing.

```php
$student = getStudentById(123);
if ($student) {
    echo $student['first_name'];
}
```

### `createStudent($data)`
**Purpose**: Creates a new student with auto-generated username and password.
**Parameters**: 
- `$data` (array) - Student data including first_name, last_name, email, phone, grade_id, subject_ids
**Returns**: int - New student ID
**Usage**: Used in registration and admin student creation.

```php
$studentId = createStudent([
    'first_name' => 'John',
    'last_name' => 'Doe',
    'email' => 'john@example.com',
    'phone' => '26812345678',
    'grade_id' => 1,
    'subject_ids' => [1, 2, 3]
]);
```

### `updateStudent($id, $data)`
**Purpose**: Updates an existing student's information.
**Parameters**: 
- `$id` (int) - Student ID
- `$data` (array) - Updated student data
**Returns**: bool - true if successful, false otherwise
**Usage**: Used in admin student editing.

```php
updateStudent(123, ['first_name' => 'Jane']);
```

### `deleteStudent($id)`
**Purpose**: Soft deletes a student (marks as deleted).
**Parameters**: 
- `$id` (int) - Student ID
**Returns**: bool - true if successful, false otherwise
**Usage**: Used in admin student management.

```php
deleteStudent(123);
```

### `permanentlyDeleteStudent($id)`
**Purpose**: Permanently removes a student and all related data.
**Parameters**: 
- `$id` (int) - Student ID
**Returns**: bool - true if successful, false otherwise
**Usage**: Used for complete student removal.

```php
permanentlyDeleteStudent(123);
```

### `restoreStudent($id, $subjectIds)`
**Purpose**: Restores a soft-deleted student.
**Parameters**: 
- `$id` (int) - Student ID
- `$subjectIds` (array|null) - Optional subject IDs to assign
**Returns**: bool - true if successful, false otherwise
**Usage**: Used in admin student restoration.

```php
restoreStudent(123, [1, 2, 3]);
```

---

## Moodle Integration

### `triggerMoodleSyncBot()`
**Purpose**: Triggers the Moodle synchronization bot.
**Parameters**: None
**Returns**: bool - true if triggered successfully
**Usage**: Used after student changes to sync with Moodle.

```php
if (triggerMoodleSyncBot()) {
    echo "Bot triggered successfully";
}
```

### `callMoodleAPI($function, $params)`
**Purpose**: Makes API calls to Moodle.
**Parameters**: 
- `$function` (string) - Moodle API function name
- `$params` (array) - API parameters
**Returns**: array - API response
**Usage**: Used for all Moodle API interactions.

```php
$result = callMoodleAPI('core_user_create_users', [
    'users' => [['username' => 'test', 'firstname' => 'Test']]
]);
```

### `enrollUserInMoodleCourse($userId, $courseId)`
**Purpose**: Enrolls a user in a Moodle course.
**Parameters**: 
- `$userId` (int) - Moodle user ID
- `$courseId` (int) - Moodle course ID
**Returns**: bool - true if successful, false otherwise
**Usage**: Used for course enrollment.

```php
enrollUserInMoodleCourse(123, 456);
```

### `suspendStudentInMoodle($student)`
**Purpose**: Suspends a student in Moodle.
**Parameters**: 
- `$student` (array) - Student data
**Returns**: bool - true if successful, false otherwise
**Usage**: Used for student suspension.

```php
suspendStudentInMoodle($student);
```

### `restoreStudentInMoodle($student)`
**Purpose**: Restores a suspended student in Moodle.
**Parameters**: 
- `$student` (array) - Student data
**Returns**: bool - true if successful, false otherwise
**Usage**: Used for student restoration.

```php
restoreStudentInMoodle($student);
```

---

## Payment System

### `ensurePaymentsTables()`
**Purpose**: Creates payment-related database tables if they don't exist.
**Parameters**: None
**Returns**: void
**Usage**: Called during system initialization.

```php
ensurePaymentsTables();
```

### `momo_get_access_token(&$error)`
**Purpose**: Gets MTN MoMo access token.
**Parameters**: 
- `&$error` (string|null) - Reference to store error message
**Returns**: string|false - Access token or false on error
**Usage**: Used for MTN MoMo API authentication.

```php
$token = momo_get_access_token($error);
if (!$token) {
    echo "Error: $error";
}
```

### `momo_request_to_pay($token, $referenceId, $amount, $currency, $msisdn, $externalId, $payerMessage, $payeeNote)`
**Purpose**: Initiates a payment request with MTN MoMo.
**Parameters**: 
- `$token` (string) - Access token
- `$referenceId` (string) - Unique reference ID
- `$amount` (string) - Payment amount
- `$currency` (string) - Currency code
- `$msisdn` (string) - Phone number
- `$externalId` (string) - External reference
- `$payerMessage` (string) - Message for payer
- `$payeeNote` (string) - Note for payee
**Returns**: array - API response
**Usage**: Used to initiate payments.

```php
$result = momo_request_to_pay($token, 'REF123', '10.00', 'SZL', '26812345678', 'EXT123', 'Payment for subjects', 'Student payment');
```

### `momo_get_request_status($token, $referenceId)`
**Purpose**: Gets the status of a payment request.
**Parameters**: 
- `$token` (string) - Access token
- `$referenceId` (string) - Reference ID
**Returns**: array - Payment status
**Usage**: Used to check payment status.

```php
$status = momo_get_request_status($token, 'REF123');
```

### `createPaymentRecord($student_id, $amount, $reference_id, $subject_ids)`
**Purpose**: Creates a payment record in the database.
**Parameters**: 
- `$student_id` (int) - Student ID
- `$amount` (float) - Payment amount
- `$reference_id` (string) - Reference ID
- `$subject_ids` (array) - Subject IDs
**Returns**: int - Payment ID
**Usage**: Used to record payments.

```php
$paymentId = createPaymentRecord(123, 10.00, 'REF123', [1, 2, 3]);
```

### `updatePaymentStatus($payment_id, $status, $reference_id)`
**Purpose**: Updates a payment's status.
**Parameters**: 
- `$payment_id` (int) - Payment ID
- `$status` (string) - New status
- `$reference_id` (string|null) - Reference ID
**Returns**: bool - true if successful, false otherwise
**Usage**: Used to update payment status.

```php
updatePaymentStatus(123, 'SUCCESSFUL', 'REF123');
```

### `getStudentPaymentStatus($studentId)`
**Purpose**: Gets a student's payment status.
**Parameters**: 
- `$studentId` (int) - Student ID
**Returns**: array - Payment status information
**Usage**: Used in admin dashboard and payment checks.

```php
$status = getStudentPaymentStatus(123);
if ($status['has_paid']) {
    echo "Student has paid";
}
```

---

## Email System

### `sendEmailVerification($email, $firstName, $token)`
**Purpose**: Sends email verification email.
**Parameters**: 
- `$email` (string) - Email address
- `$firstName` (string) - Student's first name
- `$token` (string) - Verification token
**Returns**: bool - true if sent successfully, false otherwise
**Usage**: Used during student registration.

```php
sendEmailVerification('student@example.com', 'John', 'TOKEN123');
```

### `sendStudentCredentials($email, $firstName, $username, $password)`
**Purpose**: Sends student login credentials.
**Parameters**: 
- `$email` (string) - Email address
- `$firstName` (string) - Student's first name
- `$username` (string) - Username
- `$password` (string) - Password
**Returns**: bool - true if sent successfully, false otherwise
**Usage**: Used after email verification.

```php
sendStudentCredentials('student@example.com', 'John', 'USER123', 'PASS123');
```

### `sendPaymentSuccessEmail($email, $firstName, $amount, $currency, $subjects, $referenceId)`
**Purpose**: Sends payment success notification.
**Parameters**: 
- `$email` (string) - Email address
- `$firstName` (string) - Student's first name
- `$amount` (float) - Payment amount
- `$currency` (string) - Currency
- `$subjects` (array) - Subject names
- `$referenceId` (string) - Payment reference
**Returns**: bool - true if sent successfully, false otherwise
**Usage**: Used after successful payments.

```php
sendPaymentSuccessEmail('student@example.com', 'John', 10.00, 'SZL', ['Math', 'English'], 'REF123');
```

### `sendPaymentFailureEmail($email, $firstName, $amount, $currency, $referenceId)`
**Purpose**: Sends payment failure notification.
**Parameters**: 
- `$email` (string) - Email address
- `$firstName` (string) - Student's first name
- `$amount` (float) - Payment amount
- `$currency` (string) - Currency
- `$referenceId` (string) - Payment reference
**Returns**: bool - true if sent successfully, false otherwise
**Usage**: Used after failed payments.

```php
sendPaymentFailureEmail('student@example.com', 'John', 10.00, 'SZL', 'REF123');
```

---

## Admin Functions

### `isAdminLoggedIn()`
**Purpose**: Checks if admin is logged in.
**Parameters**: None
**Returns**: bool - true if logged in, false otherwise
**Usage**: Used for admin access control.

```php
if (isAdminLoggedIn()) {
    // Admin area
}
```

### `requireAdminLogin()`
**Purpose**: Redirects to login if admin not logged in.
**Parameters**: None
**Returns**: void
**Usage**: Used at the top of admin pages.

```php
requireAdminLogin(); // Redirects if not logged in
```

### `adminLogin($username, $password)`
**Purpose**: Authenticates admin login.
**Parameters**: 
- `$username` (string) - Admin username
- `$password` (string) - Admin password
**Returns**: bool - true if successful, false otherwise
**Usage**: Used in admin login form.

```php
if (adminLogin('admin', 'password')) {
    // Login successful
}
```

### `adminLogout()`
**Purpose**: Logs out admin user.
**Parameters**: None
**Returns**: void
**Usage**: Used in admin logout.

```php
adminLogout();
```

---

## Dashboard Functions

### `getDashboardStats()`
**Purpose**: Gets statistics for admin dashboard.
**Parameters**: None
**Returns**: array - Dashboard statistics
**Usage**: Used to populate admin dashboard.

```php
$stats = getDashboardStats();
echo "Total students: " . $stats['total_students'];
```

---

## Subject Management

### `createSubject($name, $code)`
**Purpose**: Creates a new subject.
**Parameters**: 
- `$name` (string) - Subject name
- `$code` (string) - Subject code
**Returns**: int - Subject ID
**Usage**: Used in subject management.

```php
$subjectId = createSubject('Mathematics', 'MATH100');
```

### `updateSubject($id, $name, $code)`
**Purpose**: Updates an existing subject.
**Parameters**: 
- `$id` (int) - Subject ID
- `$name` (string) - New subject name
- `$code` (string) - New subject code
**Returns**: bool - true if successful, false otherwise
**Usage**: Used in subject editing.

```php
updateSubject(123, 'Advanced Math', 'MATH200');
```

### `deleteSubject($id)`
**Purpose**: Deletes a subject.
**Parameters**: 
- `$id` (int) - Subject ID
**Returns**: bool - true if successful, false otherwise
**Usage**: Used in subject management.

```php
deleteSubject(123);
```

---

## Utility Functions

### `generateSecurePasswordForMoodle($lastname)`
**Purpose**: Generates a secure password for Moodle.
**Parameters**: 
- `$lastname` (string) - Student's last name
**Returns**: string - Generated password
**Usage**: Used for Moodle password generation.

```php
$password = generateSecurePasswordForMoodle('Doe');
```

### `formatUsernameForMoodle($username)`
**Purpose**: Formats username for Moodle compatibility.
**Parameters**: 
- `$username` (string) - Username to format
**Returns**: string - Formatted username
**Usage**: Used for Moodle username formatting.

```php
$moodleUsername = formatUsernameForMoodle('2024120101');
```

### `uuidv4()`
**Purpose**: Generates a UUID v4.
**Parameters**: None
**Returns**: string - UUID v4
**Usage**: Used for unique identifiers.

```php
$uuid = uuidv4();
```

---

## Error Handling

All functions use PHP exceptions for error handling. Common exceptions:
- `Exception` - General errors
- `PDOException` - Database errors
- `MailException` - Email errors

## Security Considerations

1. **CSRF Protection**: All forms use CSRF tokens
2. **SQL Injection**: All queries use prepared statements
3. **XSS Protection**: All output is escaped
4. **Password Security**: Passwords are hashed using `password_hash()`
5. **Session Security**: Sessions are properly managed

## Performance Considerations

1. **Database Indexing**: Ensure proper indexes on frequently queried columns
2. **Caching**: Consider caching for frequently accessed data
3. **Pagination**: Use pagination for large datasets
4. **Connection Pooling**: Use connection pooling for high-traffic scenarios

## Testing Recommendations

1. **Unit Tests**: Test each function individually
2. **Integration Tests**: Test function interactions
3. **Security Tests**: Test for vulnerabilities
4. **Performance Tests**: Test under load
5. **User Acceptance Tests**: Test user workflows

---

*This documentation is generated automatically and should be updated when functions are modified.*
