<?php
/**
 * Enhanced Moodle Integration
 * Improved error handling and API management
 */

// Prevent direct access
if (!defined('MOODLE_INTEGRATION_LOADED')) {
    define('MOODLE_INTEGRATION_LOADED', true);
    
    /**
     * Enhanced Moodle API call with better error handling
     */
    function callMoodleAPIEnhanced($function, $params = []) {
        try {
            // Validate required constants
            if (!defined('MOODLE_API_URL') || !defined('MOODLE_API_TOKEN')) {
                throw new Exception('Moodle API configuration missing');
            }
            
            $url = MOODLE_API_URL . '?wstoken=' . MOODLE_API_TOKEN . 
                   '&wsfunction=' . $function . '&moodlewsrestformat=json';
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Skolo-Kine/1.0');
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                throw new Exception("CURL Error: $error");
            }
            
            if ($httpCode !== 200) {
                throw new Exception("HTTP Error: $httpCode");
            }
            
            $data = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("JSON Decode Error: " . json_last_error_msg());
            }
            
            // Check for Moodle API errors
            if (isset($data['exception'])) {
                throw new Exception("Moodle API Error: " . $data['message']);
            }
            
            return $data;
            
        } catch (Exception $e) {
            error_log("Moodle API Error: " . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Enhanced user creation in Moodle
     */
    function createMoodleUser($userData) {
        try {
            $result = callMoodleAPIEnhanced('core_user_create_users', [
                'users' => [$userData]
            ]);
            
            if (isset($result['error'])) {
                throw new Exception($result['error']);
            }
            
            if (isset($result[0]['id'])) {
                return $result[0]['id'];
            }
            
            throw new Exception("User creation failed - no ID returned");
            
        } catch (Exception $e) {
            error_log("Moodle User Creation Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Enhanced user enrollment in Moodle
     */
    function enrollMoodleUser($userId, $courseId, $roleId = 5) {
        try {
            $result = callMoodleAPIEnhanced('enrol_manual_enrol_users', [
                'enrolments' => [[
                    'roleid' => $roleId,
                    'userid' => $userId,
                    'courseid' => $courseId
                ]]
            ]);
            
            if (isset($result['error'])) {
                throw new Exception($result['error']);
            }
            
            return true;
            
        } catch (Exception $e) {
            error_log("Moodle User Enrollment Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Enhanced user suspension in Moodle
     */
    function suspendMoodleUser($userId) {
        try {
            $result = callMoodleAPIEnhanced('core_user_update_users', [
                'users' => [[
                    'id' => $userId,
                    'suspended' => 1
                ]]
            ]);
            
            if (isset($result['error'])) {
                throw new Exception($result['error']);
            }
            
            return true;
            
        } catch (Exception $e) {
            error_log("Moodle User Suspension Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Enhanced user restoration in Moodle
     */
    function restoreMoodleUser($userId) {
        try {
            $result = callMoodleAPIEnhanced('core_user_update_users', [
                'users' => [[
                    'id' => $userId,
                    'suspended' => 0
                ]]
            ]);
            
            if (isset($result['error'])) {
                throw new Exception($result['error']);
            }
            
            return true;
            
        } catch (Exception $e) {
            error_log("Moodle User Restoration Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get Moodle user by username
     */
    function getMoodleUserByUsername($username) {
        try {
            $result = callMoodleAPIEnhanced('core_user_get_users', [
                'criteria' => [[
                    'key' => 'username',
                    'value' => $username
                ]]
            ]);
            
            if (isset($result['error'])) {
                throw new Exception($result['error']);
            }
            
            if (isset($result['users']) && !empty($result['users'])) {
                return $result['users'][0];
            }
            
            return null;
            
        } catch (Exception $e) {
            error_log("Moodle User Lookup Error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get Moodle courses
     */
    function getMoodleCourses() {
        try {
            $result = callMoodleAPIEnhanced('core_course_get_courses');
            
            if (isset($result['error'])) {
                throw new Exception($result['error']);
            }
            
            return $result;
            
        } catch (Exception $e) {
            error_log("Moodle Courses Error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Enhanced bot trigger with lock management
     */
    function triggerMoodleSyncBotEnhanced() {
        $lockFile = __DIR__ . '/../bot/temp/sync.lock';
        $lockTimeout = 300; // 5 minutes
        
        // Check if bot is already running
        if (file_exists($lockFile)) {
            $lockTime = filemtime($lockFile);
            if (time() - $lockTime < $lockTimeout) {
                error_log("Moodle sync bot is already running, skipping trigger");
                return false;
            } else {
                // Remove stale lock
                unlink($lockFile);
            }
        }
        
        // Create lock file
        file_put_contents($lockFile, time());
        
        try {
            // Trigger the bot
            $botPath = __DIR__ . '/../bot/unified_sync_bot.py';
            if (file_exists($botPath)) {
                $command = "python " . escapeshellarg($botPath) . " > /dev/null 2>&1 &";
                exec($command);
                return true;
            } else {
                throw new Exception("Bot script not found");
            }
        } catch (Exception $e) {
            error_log("Bot trigger error: " . $e->getMessage());
            // Remove lock file on error
            if (file_exists($lockFile)) {
                unlink($lockFile);
            }
            return false;
        }
    }
    
    /**
     * Test Moodle API connection
     */
    function testMoodleConnection() {
        try {
            $result = callMoodleAPIEnhanced('core_webservice_get_site_info');
            
            if (isset($result['error'])) {
                return ['success' => false, 'error' => $result['error']];
            }
            
            return ['success' => true, 'data' => $result];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}