<?php
/**
 * Parent Portal Functions
 * Helper functions for parent portal features
 */

/**
 * Check if parent is logged in
 */
function isParentLoggedIn() {
    startSession();
    return isset($_SESSION['parent_logged_in']) && $_SESSION['parent_logged_in'] === true;
}

/**
 * Require parent login
 */
function requireParentLogin() {
    if (!isParentLoggedIn()) {
        header('Location: ../parent_login.php');
        exit;
    }
}

/**
 * Get parent's children
 */
function getParentChildren($parent_id) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT s.*, g.name as grade_name, ps.relationship, ps.is_primary
        FROM students s
        JOIN parent_students ps ON s.id = ps.student_id
        JOIN grades g ON s.grade_id = g.id
        WHERE ps.parent_id = ?
        ORDER BY s.full_name
    ");
    $stmt->execute([$parent_id]);
    return $stmt->fetchAll();
}

/**
 * Get student payments for parent
 */
function getStudentPayments($student_id, $parent_id) {
    // Verify parent has access to this student
    if (!parentHasAccessToStudent($parent_id, $student_id)) {
        return [];
    }
    
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT * FROM payments 
        WHERE student_id = ? 
        ORDER BY created_at DESC
    ");
    $stmt->execute([$student_id]);
    return $stmt->fetchAll();
}

/**
 * Check if parent has access to student
 */
function parentHasAccessToStudent($parent_id, $student_id) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT COUNT(*) FROM parent_students 
        WHERE parent_id = ? AND student_id = ?
    ");
    $stmt->execute([$parent_id, $student_id]);
    return $stmt->fetchColumn() > 0;
}

/**
 * Get parent notifications
 */
function getParentNotifications($parent_id, $unread_only = false) {
    $pdo = getDB();
    $sql = "SELECT * FROM parent_notifications WHERE parent_id = ?";
    if ($unread_only) {
        $sql .= " AND is_read = FALSE";
    }
    $sql .= " ORDER BY created_at DESC LIMIT 50";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$parent_id]);
    return $stmt->fetchAll();
}

/**
 * Mark notification as read
 */
function markNotificationRead($notification_id, $parent_id) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        UPDATE parent_notifications 
        SET is_read = TRUE, read_at = NOW() 
        WHERE id = ? AND parent_id = ?
    ");
    return $stmt->execute([$notification_id, $parent_id]);
}

/**
 * Send notification to parent
 */
function sendParentNotification($parent_id, $title, $message, $type = 'general', $student_id = null) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        INSERT INTO parent_notifications (parent_id, student_id, title, message, type, created_at)
        VALUES (?, ?, ?, ?, ?, NOW())
    ");
    return $stmt->execute([$parent_id, $student_id, $title, $message, $type]);
}

/**
 * Log parent activity
 */
function logParentActivity($parent_id, $action, $description = '', $student_id = null) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            INSERT INTO parent_activity_log (parent_id, action, description, student_id, ip_address, user_agent, created_at)
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([
            $parent_id,
            $action,
            $description,
            $student_id,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ]);
        return true;
    } catch (Exception $e) {
        error_log("Failed to log parent activity: " . $e->getMessage());
        return false;
    }
}

/**
 * Get payment summary for parent
 */
function getParentPaymentSummary($parent_id) {
    $children = getParentChildren($parent_id);
    $summary = [
        'total_children' => count($children),
        'active' => 0,
        'pending' => 0,
        'suspended' => 0,
        'total_paid' => 0,
        'total_pending' => 0
    ];
    
    $pdo = getDB();
    foreach ($children as $child) {
        // Count by status
        if ($child['payment_status'] === 'active') {
            $summary['active']++;
        } elseif ($child['payment_status'] === 'pending') {
            $summary['pending']++;
        } else {
            $summary['suspended']++;
        }
        
        // Calculate totals
        $stmt = $pdo->prepare("
            SELECT 
                SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END) as paid,
                SUM(CASE WHEN status = 'pending' THEN amount ELSE 0 END) as pending
            FROM payments WHERE student_id = ?
        ");
        $stmt->execute([$child['id']]);
        $result = $stmt->fetch();
        
        $summary['total_paid'] += $result['paid'] ?? 0;
        $summary['total_pending'] += $result['pending'] ?? 0;
    }
    
    return $summary;
}

/**
 * Link parent to student
 */
function linkParentToStudent($parent_id, $student_id, $relationship = 'parent', $is_primary = false) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            INSERT INTO parent_students (parent_id, student_id, relationship, is_primary)
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE relationship = ?, is_primary = ?
        ");
        return $stmt->execute([$parent_id, $student_id, $relationship, $is_primary, $relationship, $is_primary]);
    } catch (Exception $e) {
        error_log("Failed to link parent to student: " . $e->getMessage());
        return false;
    }
}