<?php
/**
 * Tenant Backup System
 * 
 * Handles automated backups for individual tenants
 * - Database backups (tenant data only)
 * - File backups (ZIP archives)
 * - Full backups (database + files)
 * - Restore capabilities
 * 
 * @package Multi-Tenant System
 * @version 1.0
 */

class TenantBackup {
    
    /**
     * Database connection
     * @var PDO
     */
    private $db;
    
    /**
     * Directory manager
     * @var TenantDirectoryManager
     */
    private $dirManager;
    
    /**
     * Constructor
     */
    public function __construct($db, $dirManager = null) {
        $this->db = $db;
        $this->dirManager = $dirManager;
        
        if ($this->dirManager === null) {
            require_once __DIR__ . '/tenant_directory_manager.php';
            $this->dirManager = new TenantDirectoryManager($db);
        }
    }
    
    /**
     * Backup tenant database (only tenant's data)
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Result with backup file info
     */
    public function backupTenantDatabase($tenant_id) {
        try {
            // Validate tenant
            if (!$this->validateTenant($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            // Get backup path
            $backupPath = $this->dirManager->getTenantBackupPath($tenant_id, 'database');
            if (!$backupPath || !file_exists($backupPath)) {
                throw new Exception("Backup directory not found");
            }
            
            // Generate backup filename
            $timestamp = date('Y-m-d_His');
            $filename = "{$timestamp}_database.sql";
            $filepath = $backupPath . '/' . $filename;
            
            // Start SQL output
            $sql = "-- Tenant Database Backup\n";
            $sql .= "-- Tenant ID: {$tenant_id}\n";
            $sql .= "-- Date: " . date('Y-m-d H:i:s') . "\n";
            $sql .= "-- Generated by: TenantBackup System\n\n";
            $sql .= "SET FOREIGN_KEY_CHECKS = 0;\n\n";
            
            // Get all tables that have tenant data
            $tables = $this->getTenantTables();
            
            foreach ($tables as $table => $tenant_column) {
                $sql .= $this->exportTableData($table, $tenant_column, $tenant_id);
            }
            
            $sql .= "\nSET FOREIGN_KEY_CHECKS = 1;\n";
            
            // Write to file
            file_put_contents($filepath, $sql);
            
            // Compress with gzip
            $gzFilepath = $filepath . '.gz';
            $this->compressFile($filepath, $gzFilepath);
            
            // Delete uncompressed file
            unlink($filepath);
            
            // Calculate checksum
            $checksum = md5_file($gzFilepath);
            $filesize = filesize($gzFilepath);
            
            // Log backup
            $this->logBackup($tenant_id, 'database', $filename . '.gz', $filesize, $checksum);
            
            return [
                'success' => true,
                'filename' => $filename . '.gz',
                'path' => $gzFilepath,
                'size' => $filesize,
                'size_formatted' => $this->formatBytes($filesize),
                'checksum' => $checksum,
                'timestamp' => $timestamp
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Backup tenant files (ZIP or TAR.GZ archive)
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Result with backup file info
     */
    public function backupTenantFiles($tenant_id) {
        try {
            // Validate tenant
            if (!$this->validateTenant($tenant_id)) {
                throw new Exception("Invalid tenant ID");
            }
            
            // Check if ZIP extension is available, otherwise use tar
            if (!class_exists('ZipArchive')) {
                return $this->backupTenantFilesWithTar($tenant_id);
            }
            
            // Get paths
            $tenantRoot = $this->dirManager->getTenantRootPath($tenant_id);
            $backupPath = $this->dirManager->getTenantBackupPath($tenant_id, 'files');
            
            if (!$tenantRoot || !file_exists($tenantRoot)) {
                throw new Exception("Tenant directory not found");
            }
            
            if (!$backupPath || !file_exists($backupPath)) {
                throw new Exception("Backup directory not found");
            }
            
            // Generate backup filename
            $timestamp = date('Y-m-d_His');
            $filename = "{$timestamp}_files.zip";
            $filepath = $backupPath . '/' . $filename;
            
            // Create ZIP archive
            $zip = new ZipArchive();
            if ($zip->open($filepath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
                throw new Exception("Failed to create ZIP archive");
            }
            
            // Add files to ZIP (exclude backups directory to avoid recursion)
            $this->addDirectoryToZip($zip, $tenantRoot . '/uploads', 'uploads');
            $this->addDirectoryToZip($zip, $tenantRoot . '/logs', 'logs');
            
            // Add config.json
            if (file_exists($tenantRoot . '/config.json')) {
                $zip->addFile($tenantRoot . '/config.json', 'config.json');
            }
            
            // Close ZIP
            $fileCount = $zip->numFiles;
            $zip->close();
            
            // Calculate checksum
            $checksum = md5_file($filepath);
            $filesize = filesize($filepath);
            
            // Log backup
            $this->logBackup($tenant_id, 'files', $filename, $filesize, $checksum);
            
            return [
                'success' => true,
                'filename' => $filename,
                'path' => $filepath,
                'size' => $filesize,
                'size_formatted' => $this->formatBytes($filesize),
                'file_count' => $fileCount,
                'checksum' => $checksum,
                'timestamp' => $timestamp
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Full tenant backup (database + files + config)
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Result with backup info
     */
    public function fullTenantBackup($tenant_id) {
        try {
            // Backup database
            $dbBackup = $this->backupTenantDatabase($tenant_id);
            if (!$dbBackup['success']) {
                throw new Exception("Database backup failed: " . $dbBackup['error']);
            }
            
            // Backup files
            $fileBackup = $this->backupTenantFiles($tenant_id);
            if (!$fileBackup['success']) {
                throw new Exception("File backup failed: " . $fileBackup['error']);
            }
            
            // Create full backup (use PharData for compatibility)
            $backupPath = $this->dirManager->getTenantBackupPath($tenant_id);
            $timestamp = date('Y-m-d_His');
            $fullBackupFilename = "{$timestamp}_full_backup.tar.gz";
            $fullBackupPath = $backupPath . '/' . $fullBackupFilename;
            
            // Use PharData (works without ZIP extension)
            $tarPath = $backupPath . '/' . $timestamp . '_full_backup.tar';
            $phar = new PharData($tarPath);
            
            // Add database backup
            $phar->addFile($dbBackup['path'], 'database/' . $dbBackup['filename']);
            
            // Add files backup
            $phar->addFile($fileBackup['path'], 'files/' . $fileBackup['filename']);
            
            // Add metadata
            $metadata = [
                'tenant_id' => $tenant_id,
                'backup_date' => date('Y-m-d H:i:s'),
                'database_backup' => $dbBackup['filename'],
                'files_backup' => $fileBackup['filename'],
                'database_size' => $dbBackup['size'],
                'files_size' => $fileBackup['size'],
                'file_count' => $fileBackup['file_count'] ?? 0,
                'checksum_db' => $dbBackup['checksum'],
                'checksum_files' => $fileBackup['checksum']
            ];
            
            $phar->addFromString('backup_info.json', json_encode($metadata, JSON_PRETTY_PRINT));
            
            // Compress with gzip
            $phar->compress(Phar::GZ);
            unset($phar);
            
            // Remove uncompressed tar
            if (file_exists($tarPath)) {
                unlink($tarPath);
            }
            
            $filesize = filesize($fullBackupPath);
            $checksum = md5_file($fullBackupPath);
            
            // Log backup
            $this->logBackup($tenant_id, 'full', $fullBackupFilename, $filesize, $checksum);
            
            return [
                'success' => true,
                'filename' => $fullBackupFilename,
                'path' => $fullBackupPath,
                'size' => $filesize,
                'size_formatted' => $this->formatBytes($filesize),
                'checksum' => $checksum,
                'database_backup' => $dbBackup,
                'files_backup' => $fileBackup,
                'timestamp' => $timestamp
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Schedule tenant backup
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $schedule Schedule type (daily, weekly, monthly)
     * @return array Result
     */
    public function scheduleTenantBackup($tenant_id, $schedule = 'daily') {
        try {
            // Validate schedule
            $validSchedules = ['daily', 'weekly', 'monthly'];
            if (!in_array($schedule, $validSchedules)) {
                throw new Exception("Invalid schedule. Use: daily, weekly, or monthly");
            }
            
            // Update tenant config
            $config = $this->dirManager->getTenantConfig($tenant_id);
            if ($config === false) {
                throw new Exception("Tenant configuration not found");
            }
            
            $config['backup_schedule'] = $schedule;
            $config['backup_enabled'] = true;
            $config['last_backup_schedule_update'] = date('Y-m-d H:i:s');
            
            $this->dirManager->createTenantConfig($tenant_id, $config);
            
            return [
                'success' => true,
                'schedule' => $schedule,
                'message' => "Backup scheduled: {$schedule}"
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * List all backups for tenant
     * 
     * @param string $tenant_id Tenant identifier
     * @return array List of backups
     */
    public function listTenantBackups($tenant_id) {
        try {
            $backupPath = $this->dirManager->getTenantBackupPath($tenant_id);
            
            if (!$backupPath || !file_exists($backupPath)) {
                return [
                    'success' => true,
                    'backups' => []
                ];
            }
            
            $backups = [];
            
            // Scan backup directories
            foreach (['database', 'files', ''] as $subdir) {
                $scanPath = $backupPath . ($subdir ? '/' . $subdir : '');
                
                if (!file_exists($scanPath)) {
                    continue;
                }
                
                $files = scandir($scanPath);
                foreach ($files as $file) {
                    if ($file === '.' || $file === '..' || $file === 'index.php') {
                        continue;
                    }
                    
                    $fullPath = $scanPath . '/' . $file;
                    if (is_file($fullPath)) {
                        $backups[] = [
                            'filename' => $file,
                            'type' => $subdir ?: 'full',
                            'path' => $fullPath,
                            'size' => filesize($fullPath),
                            'size_formatted' => $this->formatBytes(filesize($fullPath)),
                            'date' => date('Y-m-d H:i:s', filemtime($fullPath)),
                            'checksum' => md5_file($fullPath)
                        ];
                    }
                }
            }
            
            // Sort by date (newest first)
            usort($backups, function($a, $b) {
                return strcmp($b['date'], $a['date']);
            });
            
            return [
                'success' => true,
                'backups' => $backups,
                'count' => count($backups)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete specific backup
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $backup_filename Backup filename
     * @return array Result
     */
    public function deleteTenantBackup($tenant_id, $backup_filename) {
        try {
            // Find backup file
            $backups = $this->listTenantBackups($tenant_id);
            
            $found = false;
            foreach ($backups['backups'] as $backup) {
                if ($backup['filename'] === $backup_filename) {
                    if (unlink($backup['path'])) {
                        $found = true;
                        break;
                    } else {
                        throw new Exception("Failed to delete backup file");
                    }
                }
            }
            
            if (!$found) {
                throw new Exception("Backup file not found");
            }
            
            return [
                'success' => true,
                'message' => 'Backup deleted successfully'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Cleanup old backups (keep only X most recent)
     * 
     * @param string $tenant_id Tenant identifier
     * @param int $keep_count Number of backups to keep
     * @return array Result
     */
    public function cleanupOldBackups($tenant_id, $keep_count = 10) {
        try {
            $backups = $this->listTenantBackups($tenant_id);
            
            if (!$backups['success']) {
                throw new Exception("Failed to list backups");
            }
            
            $deleted = 0;
            $totalSize = 0;
            
            // Keep only the most recent backups
            $toDelete = array_slice($backups['backups'], $keep_count);
            
            foreach ($toDelete as $backup) {
                if (unlink($backup['path'])) {
                    $deleted++;
                    $totalSize += $backup['size'];
                }
            }
            
            return [
                'success' => true,
                'deleted' => $deleted,
                'kept' => min($keep_count, count($backups['backups'])),
                'space_freed' => $totalSize,
                'space_freed_formatted' => $this->formatBytes($totalSize)
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Restore tenant from backup
     * 
     * @param string $tenant_id Tenant identifier
     * @param string $backup_filename Backup file to restore from
     * @return array Result
     */
    public function restoreTenantBackup($tenant_id, $backup_filename) {
        try {
            // Find backup
            $backups = $this->listTenantBackups($tenant_id);
            $backup = null;
            
            foreach ($backups['backups'] as $b) {
                if ($b['filename'] === $backup_filename) {
                    $backup = $b;
                    break;
                }
            }
            
            if (!$backup) {
                throw new Exception("Backup file not found");
            }
            
            // Determine backup type and restore accordingly
            if ($backup['type'] === 'database') {
                return $this->restoreDatabaseBackup($tenant_id, $backup['path']);
            } elseif ($backup['type'] === 'files') {
                return $this->restoreFilesBackup($tenant_id, $backup['path']);
            } elseif ($backup['type'] === 'full') {
                return $this->restoreFullBackup($tenant_id, $backup['path']);
            } else {
                throw new Exception("Unknown backup type");
            }
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    // ========================================================================
    // PRIVATE HELPER METHODS
    // ========================================================================
    
    /**
     * Backup tenant files using TAR (fallback when ZIP not available)
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Result with backup file info
     */
    private function backupTenantFilesWithTar($tenant_id) {
        try {
            // Get paths
            $tenantRoot = $this->dirManager->getTenantRootPath($tenant_id);
            $backupPath = $this->dirManager->getTenantBackupPath($tenant_id, 'files');
            
            if (!$tenantRoot || !file_exists($tenantRoot)) {
                throw new Exception("Tenant directory not found");
            }
            
            if (!$backupPath || !file_exists($backupPath)) {
                throw new Exception("Backup directory not found");
            }
            
            // Generate backup filename
            $timestamp = date('Y-m-d_His');
            $filename = "{$timestamp}_files.tar.gz";
            $filepath = $backupPath . '/' . $filename;
            
            // Create TAR archive using system command
            $excludes = "--exclude='backups' --exclude='*.log'";
            $command = sprintf(
                "cd %s && tar -czf %s {$excludes} uploads/ logs/ config.json 2>&1",
                escapeshellarg($tenantRoot),
                escapeshellarg($filepath)
            );
            
            $output = [];
            $returnCode = 0;
            exec($command, $output, $returnCode);
            
            if ($returnCode !== 0 || !file_exists($filepath)) {
                // Fallback: Use PHP's PharData (available without ZIP extension)
                return $this->backupTenantFilesWithPhar($tenant_id);
            }
            
            // Calculate checksum and file count
            $checksum = md5_file($filepath);
            $filesize = filesize($filepath);
            
            // Estimate file count (tar doesn't provide this easily)
            $fileCount = $this->countFiles($tenantRoot . '/uploads');
            
            // Log backup
            $this->logBackup($tenant_id, 'files', $filename, $filesize, $checksum);
            
            return [
                'success' => true,
                'filename' => $filename,
                'path' => $filepath,
                'size' => $filesize,
                'size_formatted' => $this->formatBytes($filesize),
                'file_count' => $fileCount,
                'checksum' => $checksum,
                'timestamp' => $timestamp,
                'method' => 'tar'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Backup using PharData (pure PHP, no extensions needed)
     * 
     * @param string $tenant_id Tenant identifier
     * @return array Result with backup file info
     */
    private function backupTenantFilesWithPhar($tenant_id) {
        try {
            // Get paths
            $tenantRoot = $this->dirManager->getTenantRootPath($tenant_id);
            $backupPath = $this->dirManager->getTenantBackupPath($tenant_id, 'files');
            
            // Generate backup filename
            $timestamp = date('Y-m-d_His');
            $tarFilename = "{$timestamp}_files.tar";
            $tarPath = $backupPath . '/' . $tarFilename;
            $gzFilename = $tarFilename . '.gz';
            $gzPath = $backupPath . '/' . $gzFilename;
            
            // Create TAR archive with PharData
            $phar = new PharData($tarPath);
            
            // Add directories
            if (file_exists($tenantRoot . '/uploads')) {
                $phar->buildFromDirectory($tenantRoot . '/uploads', '/^(?!.*\/backups\/).*$/');
            }
            
            if (file_exists($tenantRoot . '/logs')) {
                $phar->buildFromDirectory($tenantRoot . '/logs');
            }
            
            // Add config
            if (file_exists($tenantRoot . '/config.json')) {
                $phar->addFile($tenantRoot . '/config.json', 'config.json');
            }
            
            // Compress with gzip
            $phar->compress(Phar::GZ);
            unset($phar);
            
            // Remove uncompressed tar
            if (file_exists($tarPath)) {
                unlink($tarPath);
            }
            
            // Get file info
            $checksum = md5_file($gzPath);
            $filesize = filesize($gzPath);
            $fileCount = $this->countFiles($tenantRoot . '/uploads');
            
            // Log backup
            $this->logBackup($tenant_id, 'files', $gzFilename, $filesize, $checksum);
            
            return [
                'success' => true,
                'filename' => $gzFilename,
                'path' => $gzPath,
                'size' => $filesize,
                'size_formatted' => $this->formatBytes($filesize),
                'file_count' => $fileCount,
                'checksum' => $checksum,
                'timestamp' => $timestamp,
                'method' => 'phar'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Count files in directory
     */
    private function countFiles($directory) {
        if (!is_dir($directory)) {
            return 0;
        }
        
        $count = 0;
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $count++;
            }
        }
        
        return $count;
    }
    
    /**
     * Get tables that contain tenant data
     */
    private function getTenantTables() {
        return [
            'students' => 'academy_reference',
            'teachers' => 'academy_reference',
            'payments' => 'academy_reference',
            'tenant_files' => 'tenant_id',
            'tenant_file_access_log' => 'tenant_id',
            'tenant_storage_quotas' => 'tenant_id',
            'file_operations_log' => 'tenant_id'
        ];
    }
    
    /**
     * Export table data for tenant
     */
    private function exportTableData($table, $tenant_column, $tenant_id) {
        try {
            $sql = "-- Table: {$table}\n";
            
            // Check if table exists
            $stmt = $this->db->query("SHOW TABLES LIKE '{$table}'");
            if ($stmt->rowCount() === 0) {
                $sql .= "-- Table does not exist, skipping\n\n";
                return $sql;
            }
            
            // Get table structure
            $stmt = $this->db->query("SHOW CREATE TABLE `{$table}`");
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $sql .= $row['Create Table'] . ";\n\n";
            
            // Get data for this tenant
            $stmt = $this->db->prepare("SELECT * FROM `{$table}` WHERE `{$tenant_column}` = ?");
            $stmt->execute([$tenant_id]);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (empty($rows)) {
                $sql .= "-- No data for this tenant\n\n";
                return $sql;
            }
            
            // Generate INSERT statements
            foreach ($rows as $row) {
                $columns = array_keys($row);
                $values = array_map(function($val) {
                    return $val === null ? 'NULL' : $this->db->quote($val);
                }, array_values($row));
                
                $sql .= "INSERT INTO `{$table}` (`" . implode('`, `', $columns) . "`) VALUES (" . implode(', ', $values) . ");\n";
            }
            
            $sql .= "\n";
            
            return $sql;
            
        } catch (PDOException $e) {
            return "-- Error exporting {$table}: " . $e->getMessage() . "\n\n";
        }
    }
    
    /**
     * Compress file with gzip
     */
    private function compressFile($source, $destination) {
        $gz = gzopen($destination, 'wb9');
        $fp = fopen($source, 'rb');
        
        while (!feof($fp)) {
            gzwrite($gz, fread($fp, 1024 * 512));
        }
        
        fclose($fp);
        gzclose($gz);
    }
    
    /**
     * Add directory to ZIP recursively
     */
    private function addDirectoryToZip($zip, $directory, $zipPath = '') {
        if (!is_dir($directory)) {
            return;
        }
        
        $files = scandir($directory);
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..' || $file === 'index.php' || $file === '.htaccess') {
                continue;
            }
            
            $fullPath = $directory . '/' . $file;
            $zipFilePath = $zipPath ? $zipPath . '/' . $file : $file;
            
            if (is_dir($fullPath)) {
                $zip->addEmptyDir($zipFilePath);
                $this->addDirectoryToZip($zip, $fullPath, $zipFilePath);
            } elseif (is_file($fullPath)) {
                $zip->addFile($fullPath, $zipFilePath);
            }
        }
    }
    
    /**
     * Restore database backup
     */
    private function restoreDatabaseBackup($tenant_id, $backup_path) {
        try {
            // Decompress if needed
            if (substr($backup_path, -3) === '.gz') {
                $tempPath = sys_get_temp_dir() . '/restore_' . time() . '.sql';
                
                $gz = gzopen($backup_path, 'rb');
                $fp = fopen($tempPath, 'wb');
                
                while (!gzeof($gz)) {
                    fwrite($fp, gzread($gz, 4096));
                }
                
                gzclose($gz);
                fclose($fp);
                
                $sqlContent = file_get_contents($tempPath);
                unlink($tempPath);
            } else {
                $sqlContent = file_get_contents($backup_path);
            }
            
            // Execute SQL
            $this->db->exec($sqlContent);
            
            return [
                'success' => true,
                'message' => 'Database restored successfully'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Restore files backup
     */
    private function restoreFilesBackup($tenant_id, $backup_path) {
        try {
            $tenantRoot = $this->dirManager->getTenantRootPath($tenant_id);
            
            $zip = new ZipArchive();
            if ($zip->open($backup_path) !== true) {
                throw new Exception("Failed to open backup file");
            }
            
            $zip->extractTo($tenantRoot);
            $zip->close();
            
            return [
                'success' => true,
                'message' => 'Files restored successfully'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Restore full backup
     */
    private function restoreFullBackup($tenant_id, $backup_path) {
        try {
            $tempDir = sys_get_temp_dir() . '/restore_' . time();
            mkdir($tempDir, 0755, true);
            
            // Extract full backup
            $zip = new ZipArchive();
            if ($zip->open($backup_path) !== true) {
                throw new Exception("Failed to open full backup");
            }
            
            $zip->extractTo($tempDir);
            $zip->close();
            
            // Restore database
            if (file_exists($tempDir . '/database')) {
                $dbFiles = scandir($tempDir . '/database');
                foreach ($dbFiles as $file) {
                    if (substr($file, -6) === '.sql.gz' || substr($file, -4) === '.sql') {
                        $dbRestore = $this->restoreDatabaseBackup($tenant_id, $tempDir . '/database/' . $file);
                        if (!$dbRestore['success']) {
                            throw new Exception("Database restore failed");
                        }
                    }
                }
            }
            
            // Restore files
            if (file_exists($tempDir . '/files')) {
                $fileFiles = scandir($tempDir . '/files');
                foreach ($fileFiles as $file) {
                    if (substr($file, -4) === '.zip') {
                        $fileRestore = $this->restoreFilesBackup($tenant_id, $tempDir . '/files/' . $file);
                        if (!$fileRestore['success']) {
                            throw new Exception("File restore failed");
                        }
                    }
                }
            }
            
            // Cleanup temp
            $this->deleteDirectoryRecursive($tempDir);
            
            return [
                'success' => true,
                'message' => 'Full backup restored successfully'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Log backup operation
     */
    private function logBackup($tenant_id, $type, $filename, $size, $checksum) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO file_operations_log 
                (tenant_id, operation, message, metadata, created_at)
                VALUES (?, ?, ?, ?, NOW())
            ");
            
            $metadata = json_encode([
                'backup_type' => $type,
                'filename' => $filename,
                'size' => $size,
                'checksum' => $checksum
            ]);
            
            $stmt->execute([
                $tenant_id,
                'backup_created',
                "Backup created: {$filename}",
                $metadata
            ]);
        } catch (PDOException $e) {
            // Silent fail - logging is optional
        }
    }
    
    /**
     * Validate tenant
     */
    private function validateTenant($tenant_id) {
        return preg_match('/^[a-zA-Z0-9_-]{3,50}$/', $tenant_id);
    }
    
    /**
     * Format bytes
     */
    private function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    /**
     * Delete directory recursively
     */
    private function deleteDirectoryRecursive($dir) {
        if (!file_exists($dir)) {
            return true;
        }
        
        if (!is_dir($dir)) {
            return unlink($dir);
        }
        
        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }
            
            if (!$this->deleteDirectoryRecursive($dir . '/' . $item)) {
                return false;
            }
        }
        
        return rmdir($dir);
    }
}

