<?php
/**
 * Parent Payment Page
 * Allows parents to select subjects and payment period for their children
 */

require_once '../includes/functions.php';
require_once '../includes/parent_functions.php';
requireParentLogin();

$parent_id = $_SESSION['parent_id'];
$student_id = (int)($_GET['student_id'] ?? 0);

// Verify parent has access to this student
if (!parentHasAccessToStudent($parent_id, $student_id)) {
    header('Location: dashboard.php');
    exit;
}

$pdo = getDB();

// Get student details
$stmt = $pdo->prepare("
    SELECT s.*, g.name as grade_name 
    FROM students s
    JOIN grades g ON s.grade_id = g.id
    WHERE s.id = ?
");
$stmt->execute([$student_id]);
$student = $stmt->fetch();

if (!$student) {
    header('Location: dashboard.php');
    exit;
}

// Get all subjects available for this student's grade
$stmt = $pdo->prepare("
    SELECT s.id, s.name, s.code 
    FROM subjects s
    INNER JOIN grade_subject gs ON s.id = gs.subject_id
    WHERE gs.grade_id = ?
    ORDER BY s.name
");
$stmt->execute([$student['grade_id']]);
$subjects = $stmt->fetchAll();

// Get currently enrolled subjects to mark them
$stmt = $pdo->prepare("
    SELECT subject_id FROM student_subject WHERE student_id = ?
");
$stmt->execute([$student_id]);
$enrolled_subjects = array_column($stmt->fetchAll(), 'subject_id');

// Get school settings for pricing
$stmt = $pdo->prepare("
    SELECT setting_key, setting_value 
    FROM tenant_settings 
    WHERE academy_reference = ?
");
$stmt->execute([$student['academy_reference']]);
$settings = [];
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get school name for display
$schoolName = $settings['school_name'] ?? 'School';

// School-specific pricing - Each school sets their own fees
$price_per_subject = floatval($settings['subject_fee'] ?? 350);
$yearly_discount = floatval($settings['yearly_discount'] ?? 10); // 10% discount for yearly

// Get currency symbol (default SZL = E)
$currency_symbol = $settings['currency_symbol'] ?? 'E';

// Check if card payments are enabled for this school
$card_payments_enabled = ($settings['card_payments_enabled'] ?? 0) == 1;
$payment_gateway = $settings['payment_gateway'] ?? 'paygate';

$error = '';
$success = '';

// Handle payment submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_payment'])) {
    $selected_subjects = $_POST['subjects'] ?? [];
    $payment_period = $_POST['payment_period'] ?? 'monthly';
    $payment_method = $_POST['payment_method'] ?? 'momo';
    $momo_number = trim($_POST['momo_number'] ?? '');
    
    if (empty($selected_subjects)) {
        $error = "Please select at least one subject.";
    } elseif ($payment_method === 'momo' && empty($momo_number)) {
        $error = "Please enter your MoMo number.";
    } else {
        // Calculate total amount
        $subject_count = count($selected_subjects);
        $base_amount = $price_per_subject * $subject_count;
        
        if ($payment_period === 'yearly') {
            $months = 12;
            $total_before_discount = $base_amount * $months;
            $discount_amount = $total_before_discount * ($yearly_discount / 100);
            $total_amount = $total_before_discount - $discount_amount;
        } else {
            $total_amount = $base_amount;
        }
        
        // Store payment details in session for processing page
        $_SESSION['pending_payment'] = [
            'student_id' => $student_id,
            'subjects' => $selected_subjects,
            'payment_period' => $payment_period,
            'subject_count' => $subject_count,
            'price_per_subject' => $price_per_subject,
            'total_amount' => $total_amount,
            'payment_method' => $payment_method,
            'momo_number' => $momo_number,
            'parent_id' => $parent_id,
            'payment_gateway' => $payment_gateway,
            'currency_symbol' => $currency_symbol
        ];
        
        // Redirect to payment processing
        header('Location: process_payment.php');
        exit;
    }
}

logParentActivity($parent_id, 'VIEW_PAYMENT', "Viewed payment page for student: {$student['full_name']}", $student_id);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Make Payment - Parent Portal</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #667eea;
            --secondary: #764ba2;
        }
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
        }
        .navbar {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%) !important;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.08);
        }
        .subject-card {
            cursor: pointer;
            transition: all 0.3s;
            border: 2px solid #e0e0e0;
        }
        .subject-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(102,126,234,0.2);
        }
        .subject-card.selected {
            border-color: var(--primary);
            background: linear-gradient(135deg, rgba(102,126,234,0.1) 0%, rgba(118,75,162,0.1) 100%);
        }
        .subject-card input[type="checkbox"] {
            width: 20px;
            height: 20px;
        }
        .price-summary {
            position: sticky;
            top: 20px;
        }
        .payment-period-option {
            cursor: pointer;
            transition: all 0.3s;
        }
        .payment-period-option:hover {
            transform: scale(1.02);
        }
        .payment-period-option input[type="radio"]:checked + label {
            border-color: var(--primary);
            background: linear-gradient(135deg, rgba(102,126,234,0.15) 0%, rgba(118,75,162,0.15) 100%);
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar navbar-dark sticky-top">
        <div class="container-fluid">
            <a href="dashboard.php" class="navbar-brand">
                <i class="bi bi-arrow-left me-2"></i>Back to Dashboard
            </a>
            <div class="d-flex align-items-center gap-2">
                <span class="text-white">
                    <i class="bi bi-person-circle me-2"></i><?= htmlspecialchars($_SESSION['parent_name']) ?>
                </span>
                <a href="../logout.php?parent=1" class="btn btn-light btn-sm">
                    <i class="bi bi-box-arrow-right me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container my-4">
        <!-- Student Info Header -->
        <div class="card mb-4">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <h4 class="mb-1">
                            <i class="bi bi-person-badge text-primary me-2"></i>
                            Payment for: <?= htmlspecialchars($student['full_name']) ?>
                        </h4>
                        <p class="text-muted mb-0">
                            Grade: <?= htmlspecialchars($student['grade_name']) ?> | 
                            Username: <?= htmlspecialchars($student['username']) ?>
                        </p>
                    </div>
                    <div>
                        <?php if ($student['payment_status'] === 'active'): ?>
                            <span class="badge bg-success fs-6">Currently Active</span>
                        <?php else: ?>
                            <span class="badge bg-warning fs-6">Payment Required</span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="bi bi-exclamation-triangle-fill me-2"></i><?= htmlspecialchars($error) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <form method="POST" id="paymentForm">
            <div class="row">
                <!-- Left Column: Subject Selection -->
                <div class="col-lg-8 mb-4">
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0"><i class="bi bi-book me-2"></i>Select Subjects</h5>
                        </div>
                        <div class="card-body">
                            <p class="text-muted mb-3">
                                <i class="bi bi-info-circle me-1"></i>
                                These are the subjects available for <?= htmlspecialchars($student['grade_name']) ?>. 
                                Select the subjects you want to pay for. Already enrolled subjects are marked.
                            </p>
                            
                            <?php if (empty($subjects)): ?>
                                <div class="alert alert-warning">
                                    <i class="bi bi-exclamation-triangle me-2"></i>
                                    <strong>No subjects available.</strong> 
                                    No subjects have been configured for <?= htmlspecialchars($student['grade_name']) ?> yet.
                                </div>
                            <?php else: ?>
                                <div class="row g-3">
                                    <?php foreach ($subjects as $subject): ?>
                                        <?php $is_enrolled = in_array($subject['id'], $enrolled_subjects); ?>
                                        <div class="col-md-6">
                                            <div class="subject-card card h-100 <?= $is_enrolled ? 'selected' : '' ?>" 
                                                 onclick="toggleSubject(this, <?= $subject['id'] ?>)">
                                                <div class="card-body">
                                                    <div class="d-flex align-items-center">
                                                        <input type="checkbox" 
                                                               name="subjects[]" 
                                                               value="<?= $subject['id'] ?>"
                                                               id="subject_<?= $subject['id'] ?>"
                                                               <?= $is_enrolled ? 'checked' : '' ?>
                                                               onclick="event.stopPropagation()"
                                                               onchange="updateTotal()">
                                                        <label for="subject_<?= $subject['id'] ?>" 
                                                               class="ms-3 mb-0 flex-grow-1"
                                                               style="cursor: pointer;">
                                                            <strong><?= htmlspecialchars($subject['name']) ?></strong>
                                                            <br>
                                                            <small class="text-muted"><?= htmlspecialchars($subject['code']) ?></small>
                                                        </label>
                                                        <div class="text-end">
                                                            <strong class="text-primary fs-5">
                                                                <?= $currency_symbol ?><?= number_format($price_per_subject, 2) ?>
                                                            </strong>
                                                            <br>
                                                            <small class="text-muted">/month</small>
                                                        </div>
                                                    </div>
                                                    <?php if ($is_enrolled): ?>
                                                        <div class="mt-2">
                                                            <span class="badge bg-success">
                                                                <i class="bi bi-check-circle me-1"></i>Currently Enrolled
                                                            </span>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Payment Period Selection -->
                    <div class="card mt-4">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="mb-0"><i class="bi bi-calendar-check me-2"></i>Payment Period</h5>
                        </div>
                        <div class="card-body">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <div class="payment-period-option">
                                        <input type="radio" 
                                               name="payment_period" 
                                               value="monthly" 
                                               id="monthly" 
                                               checked
                                               onchange="updateTotal()"
                                               class="d-none">
                                        <label for="monthly" class="card h-100 mb-0" style="cursor: pointer;">
                                            <div class="card-body text-center">
                                                <i class="bi bi-calendar-month text-primary fs-1"></i>
                                                <h5 class="mt-3">Monthly Payment</h5>
                                                <p class="text-muted mb-0">Pay for one month</p>
                                                <div class="mt-2">
                                                    <span class="badge bg-light text-dark">Most Flexible</span>
                                                </div>
                                            </div>
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="payment-period-option">
                                        <input type="radio" 
                                               name="payment_period" 
                                               value="yearly" 
                                               id="yearly"
                                               onchange="updateTotal()"
                                               class="d-none">
                                        <label for="yearly" class="card h-100 mb-0" style="cursor: pointer;">
                                            <div class="card-body text-center">
                                                <i class="bi bi-calendar-range text-success fs-1"></i>
                                                <h5 class="mt-3">Yearly Payment</h5>
                                                <p class="text-muted mb-0">Pay for 12 months</p>
                                                <div class="mt-2">
                                                    <span class="badge bg-success">
                                                        Save <?= $yearly_discount ?>%!
                                                    </span>
                                                </div>
                                            </div>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Payment Method Selection -->
                    <div class="card mt-4">
                        <div class="card-header bg-info text-white">
                            <h5 class="mb-0"><i class="bi bi-wallet2 me-2"></i>Payment Method</h5>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <label class="form-label">Choose Payment Method <span class="text-danger">*</span></label>
                                <div class="row g-3">
                                    <!-- MoMo Option -->
                                    <div class="col-md-6">
                                        <input type="radio" name="payment_method" value="momo" id="method_momo" 
                                               checked onchange="togglePaymentFields()" class="d-none">
                                        <label for="method_momo" class="card h-100 mb-0" style="cursor: pointer;">
                                            <div class="card-body text-center">
                                                <i class="bi bi-phone text-primary fs-1"></i>
                                                <h6 class="mt-3">MTN Mobile Money</h6>
                                                <p class="text-muted small mb-0">Pay with your MoMo wallet</p>
                                                <div class="mt-2">
                                                    <span class="badge bg-primary">Most Popular</span>
                                                </div>
                                            </div>
                                        </label>
                                    </div>
                                    
                                    <!-- Card Option -->
                                    <?php if ($card_payments_enabled): ?>
                                    <div class="col-md-6">
                                        <input type="radio" name="payment_method" value="card" id="method_card" 
                                               onchange="togglePaymentFields()" class="d-none">
                                        <label for="method_card" class="card h-100 mb-0" style="cursor: pointer;">
                                            <div class="card-body text-center">
                                                <i class="bi bi-credit-card text-success fs-1"></i>
                                                <h6 class="mt-3">Credit / Debit Card</h6>
                                                <p class="text-muted small mb-0">Visa, Mastercard accepted</p>
                                                <div class="mt-2">
                                                    <span class="badge bg-success">Secure</span>
                                                </div>
                                            </div>
                                        </label>
                                    </div>
                                    <?php else: ?>
                                    <div class="col-md-6">
                                        <div class="card h-100 bg-light">
                                            <div class="card-body text-center">
                                                <i class="bi bi-credit-card text-muted fs-1"></i>
                                                <h6 class="mt-3 text-muted">Credit / Debit Card</h6>
                                                <p class="text-muted small mb-0">Not available yet</p>
                                                <div class="mt-2">
                                                    <span class="badge bg-secondary">Coming Soon</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <!-- MoMo Number (shown when MoMo selected) -->
                            <div id="momo_fields" class="mb-3">
                                <label for="momo_number" class="form-label">
                                    <i class="bi bi-phone me-1"></i>MoMo Number <span class="text-danger">*</span>
                                </label>
                                <div class="input-group input-group-lg">
                                    <span class="input-group-text">+268</span>
                                    <input type="tel" 
                                           class="form-control" 
                                           id="momo_number" 
                                           name="momo_number"
                                           placeholder="7687 3207"
                                           pattern="[0-9]{8}">
                                </div>
                                <small class="text-muted">
                                    <i class="bi bi-info-circle me-1"></i>
                                    Enter your MTN MoMo number (8 digits)
                                </small>
                            </div>
                            
                            <!-- Card Info (shown when Card selected) -->
                            <div id="card_fields" style="display: none;">
                                <div class="alert alert-info">
                                    <i class="bi bi-shield-check me-2"></i>
                                    <strong>Secure Card Payment</strong><br>
                                    You'll be redirected to our secure payment gateway to enter your card details.
                                    Your card information is never stored on our servers.
                                </div>
                                <div class="text-center">
                                    <img src="../images/payment-cards.png" alt="Accepted Cards" style="max-height: 40px;" 
                                         onerror="this.outerHTML='<small class=\'text-muted\'>Visa, Mastercard, American Express accepted</small>'">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Right Column: Price Summary -->
                <div class="col-lg-4">
                    <div class="price-summary">
                        <div class="card">
                            <div class="card-header bg-dark text-white">
                                <h5 class="mb-0"><i class="bi bi-calculator me-2"></i>Payment Summary</h5>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <small class="text-muted">Selected Subjects</small>
                                    <h4 id="selected_count" class="mb-0">0</h4>
                                </div>
                                
                                <div class="mb-3">
                                    <small class="text-muted">Price per Subject (<?= htmlspecialchars($schoolName) ?>)</small>
                                    <h5 id="price_per_subject" class="mb-0"><?= $currency_symbol ?><?= number_format($price_per_subject, 2) ?></h5>
                                </div>
                                
                                <div class="mb-3" id="yearly_info" style="display: none;">
                                    <small class="text-muted">Payment Period</small>
                                    <h6 class="mb-0">12 Months (Yearly)</h6>
                                    <small class="text-success">
                                        <i class="bi bi-tag-fill me-1"></i><?= $yearly_discount ?>% Discount Applied
                                    </small>
                                </div>
                                
                                <hr>
                                
                                <div class="mb-3">
                                    <div class="d-flex justify-content-between">
                                        <span>Subtotal:</span>
                                        <strong id="subtotal"><?= $currency_symbol ?>0.00</strong>
                                    </div>
                                </div>
                                
                                <div class="mb-3" id="discount_row" style="display: none;">
                                    <div class="d-flex justify-content-between text-success">
                                        <span>Discount (<?= $yearly_discount ?>%):</span>
                                        <strong id="discount">-<?= $currency_symbol ?>0.00</strong>
                                    </div>
                                </div>
                                
                                <div class="bg-primary text-white p-3 rounded">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span class="fs-5">Total Amount:</span>
                                        <strong class="fs-3" id="total_amount"><?= $currency_symbol ?>0.00</strong>
                                    </div>
                                </div>
                                
                                <button type="submit" 
                                        name="submit_payment" 
                                        class="btn btn-success btn-lg w-100 mt-3"
                                        id="submit_btn"
                                        disabled>
                                    <i class="bi bi-credit-card me-2"></i>Proceed to Payment
                                </button>
                                
                                <div class="text-center mt-3">
                                    <small class="text-muted">
                                        <i class="bi bi-shield-check me-1"></i>
                                        Secure payment via MTN MoMo
                                    </small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const pricePerSubject = <?= $price_per_subject ?>;
        const yearlyDiscount = <?= $yearly_discount ?>;
        const currencySymbol = '<?= $currency_symbol ?>';
        
        // Toggle payment method fields
        function togglePaymentFields() {
            const paymentMethod = document.querySelector('input[name="payment_method"]:checked').value;
            const momoFields = document.getElementById('momo_fields');
            const cardFields = document.getElementById('card_fields');
            const momoInput = document.getElementById('momo_number');
            
            if (paymentMethod === 'momo') {
                momoFields.style.display = 'block';
                cardFields.style.display = 'none';
                momoInput.required = true;
            } else {
                momoFields.style.display = 'none';
                cardFields.style.display = 'block';
                momoInput.required = false;
            }
            
            // Update card selection styling
            document.querySelectorAll('input[name="payment_method"]').forEach(radio => {
                const label = document.querySelector(`label[for="${radio.id}"]`);
                if (radio.checked) {
                    label.style.borderColor = 'var(--primary)';
                    label.style.background = 'linear-gradient(135deg, rgba(102,126,234,0.1) 0%, rgba(118,75,162,0.1) 100%)';
                } else {
                    label.style.borderColor = '#dee2e6';
                    label.style.background = 'white';
                }
            });
        }
        
        function toggleSubject(card, subjectId) {
            const checkbox = card.querySelector('input[type="checkbox"]');
            checkbox.checked = !checkbox.checked;
            
            if (checkbox.checked) {
                card.classList.add('selected');
            } else {
                card.classList.remove('selected');
            }
            
            updateTotal();
        }
        
        function updateTotal() {
            const selectedCheckboxes = document.querySelectorAll('input[name="subjects[]"]:checked');
            const count = selectedCheckboxes.length;
            const paymentPeriod = document.querySelector('input[name="payment_period"]:checked').value;
            
            // Update count
            document.getElementById('selected_count').textContent = count;
            
            // Calculate amounts
            let subtotal = pricePerSubject * count;
            let total = subtotal;
            
            if (paymentPeriod === 'yearly') {
                subtotal = subtotal * 12;
                const discount = subtotal * (yearlyDiscount / 100);
                total = subtotal - discount;
                
                // Show yearly info
                document.getElementById('yearly_info').style.display = 'block';
                document.getElementById('discount_row').style.display = 'block';
                document.getElementById('discount').textContent = '-E' + discount.toFixed(2);
            } else {
                // Hide yearly info
                document.getElementById('yearly_info').style.display = 'none';
                document.getElementById('discount_row').style.display = 'none';
            }
            
            // Update display with school's currency
            document.getElementById('subtotal').textContent = currencySymbol + subtotal.toFixed(2);
            document.getElementById('total_amount').textContent = currencySymbol + total.toFixed(2);
            
            if (paymentPeriod === 'yearly') {
                const discount = subtotal * (yearlyDiscount / 100);
                document.getElementById('discount').textContent = '-' + currencySymbol + discount.toFixed(2);
            }
            
            // Enable/disable submit button
            document.getElementById('submit_btn').disabled = (count === 0);
        }
        
        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            updateTotal();
            togglePaymentFields();
        });
    </script>
</body>
</html>

