<?php
/**
 * Parent Login Page
 * Authentication for parent portal access
 */

require_once 'includes/functions.php';
require_once 'includes/parent_functions.php';

// Get school from URL (for branded login pages via clean URLs)
$url_school_code = $_GET['school'] ?? '';
$school_branding = null;

if ($url_school_code) {
    $pdo = getDB();
    $stmt = $pdo->prepare("
        SELECT ar.reference_code, ar.academy_name, ts.setting_key, ts.setting_value
        FROM academy_references ar
        LEFT JOIN tenant_settings ts ON ar.reference_code = ts.academy_reference
        WHERE ar.reference_code = ? AND ar.is_active = 1
    ");
    $stmt->execute([strtoupper($url_school_code)]);
    $results = $stmt->fetchAll();
    
    if (!empty($results)) {
        $school_branding = [
            'code' => $results[0]['reference_code'],
            'name' => $results[0]['academy_name']
        ];
        foreach ($results as $row) {
            if ($row['setting_key']) {
                $school_branding[$row['setting_key']] = $row['setting_value'];
            }
        }
    }
}

// Redirect if already logged in
if (isParentLoggedIn()) {
    header('Location: parent/dashboard.php');
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $remember_me = isset($_POST['remember_me']);
    
    if (empty($email) || empty($password)) {
        $error = "Please enter both email and password.";
    } else {
        $pdo = getDB();
        
        try {
            // Get parent by email
            $stmt = $pdo->prepare("SELECT * FROM parents WHERE email = ?");
            $stmt->execute([$email]);
            $parent = $stmt->fetch();
            
            if (!$parent) {
                $error = "Invalid email or password.";
                
                // Log failed attempt
                try {
                    $stmt = $pdo->prepare("
                        INSERT INTO parent_activity_log (parent_id, action, description, ip_address, user_agent, created_at)
                        VALUES (0, 'FAILED_LOGIN', ?, ?, ?, NOW())
                    ");
                    $stmt->execute([
                        "Failed login attempt for: $email",
                        $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                        $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
                    ]);
                } catch (Exception $e) {
                    error_log("Failed to log failed login: " . $e->getMessage());
                }
            } elseif ($parent['locked_until'] && strtotime($parent['locked_until']) > time()) {
                $locked_minutes = ceil((strtotime($parent['locked_until']) - time()) / 60);
                $error = "Account is locked due to multiple failed login attempts. Please try again in $locked_minutes minutes.";
            } elseif (!password_verify($password, $parent['password_hash'])) {
                // Increment login attempts
                $attempts = ($parent['login_attempts'] ?? 0) + 1;
                
                if ($attempts >= 5) {
                    // Lock account for 30 minutes
                    $stmt = $pdo->prepare("
                        UPDATE parents 
                        SET login_attempts = ?, locked_until = DATE_ADD(NOW(), INTERVAL 30 MINUTE)
                        WHERE id = ?
                    ");
                    $stmt->execute([$attempts, $parent['id']]);
                    $error = "Account locked due to too many failed attempts. Please try again in 30 minutes.";
                } else {
                    $stmt = $pdo->prepare("UPDATE parents SET login_attempts = ? WHERE id = ?");
                    $stmt->execute([$attempts, $parent['id']]);
                    $remaining = 5 - $attempts;
                    $error = "Invalid email or password. $remaining attempts remaining.";
                }
                
                // Log failed attempt
                logParentActivity($parent['id'], 'FAILED_LOGIN', "Failed login attempt");
            } else {
                // Successful login!
                
                // Reset login attempts and update last login
                $stmt = $pdo->prepare("
                    UPDATE parents 
                    SET login_attempts = 0, locked_until = NULL, last_login = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$parent['id']]);
                
                // Set session variables
                $_SESSION['parent_logged_in'] = true;
                $_SESSION['parent_id'] = $parent['id'];
                $_SESSION['parent_name'] = $parent['full_name'];
                $_SESSION['parent_email'] = $parent['email'];
                $_SESSION['academy_reference'] = $parent['academy_reference'];
                
                // Remember me functionality
                if ($remember_me) {
                    // Set cookie for 30 days
                    $token = bin2hex(random_bytes(32));
                    setcookie('parent_remember', $token, time() + (30 * 24 * 60 * 60), '/', '', true, true);
                    
                    // Store token in database (you'd need to add a remember_token column)
                    // For now, we'll skip this implementation
                }
                
                // Log successful login
                logParentActivity($parent['id'], 'LOGIN', 'Successful login');
                
                // Redirect to dashboard
                header('Location: parent/dashboard.php');
                exit;
            }
        } catch (PDOException $e) {
            error_log("Parent login error: " . $e->getMessage());
            $error = "An error occurred during login. Please try again.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Parent Login - School Management System</title>

      <!-- Favicon -->
      <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">

    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 20px 0;
        }
        .login-container {
            max-width: 450px;
            margin: 0 auto;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 1rem 1rem 0 0 !important;
            padding: 2rem;
        }
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            padding: 0.75rem;
            font-size: 1.1rem;
            font-weight: 600;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102,126,234,0.4);
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102,126,234,0.25);
        }
        .form-check-input:checked {
            background-color: #667eea;
            border-color: #667eea;
        }
        .link-primary {
            color: #667eea !important;
        }
        .link-primary:hover {
            color: #764ba2 !important;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="login-container">
            <div class="card">
                <div class="card-header text-center">
                    <h2 class="mb-2"><i class="bi bi-person-circle"></i> Parent Login</h2>
                    <p class="mb-0">Access your child's information</p>
                </div>
                <div class="card-body p-4">
                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <i class="bi bi-exclamation-triangle-fill me-2"></i><?= htmlspecialchars($error) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST" action="">
                        <div class="mb-3">
                            <label for="email" class="form-label">Email Address</label>
                            <div class="input-group input-group-lg">
                                <span class="input-group-text"><i class="bi bi-envelope"></i></span>
                                <input type="email" class="form-control" id="email" name="email" 
                                       value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" 
                                       placeholder="parent@example.com" required autofocus>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="password" class="form-label">Password</label>
                            <div class="input-group input-group-lg">
                                <span class="input-group-text"><i class="bi bi-lock"></i></span>
                                <input type="password" class="form-control" id="password" name="password" 
                                       placeholder="Enter your password" required>
                                <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                    <i class="bi bi-eye" id="eyeIcon"></i>
                                </button>
                            </div>
                        </div>
                        
                        <div class="mb-3 d-flex justify-content-between align-items-center">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="remember_me" name="remember_me">
                                <label class="form-check-label" for="remember_me">
                                    Remember me
                                </label>
                            </div>
                            <a href="parent_forgot_password.php" class="link-primary text-decoration-none">
                                Forgot password?
                            </a>
                        </div>
                        
                        <button type="submit" class="btn btn-primary btn-lg w-100 mb-3">
                            <i class="bi bi-box-arrow-in-right me-2"></i>Login
                        </button>
                        
                        <div class="text-center">
                            <p class="mb-0">
                                Don't have an account? 
                                <a href="parent_register.php" class="link-primary text-decoration-none fw-bold">Register here</a>
                            </p>
                        </div>
                    </form>
                </div>
            </div>
            
            <div class="text-center mt-4">
                <a href="index.php" class="text-white text-decoration-none">
                    <i class="bi bi-arrow-left me-2"></i>Back to Home
                </a>
            </div>
            
            <div class="card mt-3 bg-white bg-opacity-25 border-0">
                <div class="card-body text-white text-center py-2">
                    <small>
                        <i class="bi bi-shield-check me-1"></i>
                        Your data is secure and encrypted
                    </small>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle password visibility
        document.getElementById('togglePassword').addEventListener('click', function() {
            const password = document.getElementById('password');
            const eyeIcon = document.getElementById('eyeIcon');
            
            if (password.type === 'password') {
                password.type = 'text';
                eyeIcon.classList.remove('bi-eye');
                eyeIcon.classList.add('bi-eye-slash');
            } else {
                password.type = 'password';
                eyeIcon.classList.remove('bi-eye-slash');
                eyeIcon.classList.add('bi-eye');
            }
        });
    </script>
</body>
</html>

