<?php
/**
 * Parent Registration Page
 * Allows parents to create an account and link to their children
 */

session_start();
require_once 'includes/functions.php';
require_once 'includes/parent_functions.php';

$error = '';
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $pdo = getDB();
    
    $full_name = trim($_POST['full_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $student_username = trim($_POST['student_username'] ?? '');
    $relationship = $_POST['relationship'] ?? 'parent';
    
    // Validation
    if (empty($full_name) || empty($email) || empty($phone) || empty($password) || empty($student_username)) {
        $error = "All fields are required.";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "Please enter a valid email address.";
    } elseif (strlen($password) < 6) {
        $error = "Password must be at least 6 characters long.";
    } elseif ($password !== $confirm_password) {
        $error = "Passwords do not match.";
    } else {
        try {
            // Find student
            $stmt = $pdo->prepare("SELECT id, academy_reference FROM students WHERE username = ?");
            $stmt->execute([$student_username]);
            $student = $stmt->fetch();
            
            if (!$student) {
                $error = "Student not found. Please check the username and try again.";
            } else {
                // Check if parent already exists
                $stmt = $pdo->prepare("SELECT id FROM parents WHERE email = ? AND academy_reference = ?");
                $stmt->execute([$email, $student['academy_reference']]);
                $existing_parent = $stmt->fetch();
                
                if ($existing_parent) {
                    $error = "An account with this email already exists for this school.";
                } else {
                    // Create parent account
                    $password_hash = password_hash($password, PASSWORD_DEFAULT);
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO parents (full_name, email, phone, password_hash, academy_reference, created_at)
                        VALUES (?, ?, ?, ?, ?, NOW())
                    ");
                    $stmt->execute([$full_name, $email, $phone, $password_hash, $student['academy_reference']]);
                    $parent_id = $pdo->lastInsertId();
                    
                    // Link parent to student
                    $stmt = $pdo->prepare("
                        INSERT INTO parent_students (parent_id, student_id, relationship, is_primary)
                        VALUES (?, ?, ?, TRUE)
                    ");
                    $stmt->execute([$parent_id, $student['id'], $relationship]);
                    
                    // Send welcome notification
                    sendParentNotification(
                        $parent_id,
                        'Welcome to the Parent Portal!',
                        "Your account has been created successfully. You can now view your child's information and make payments online.",
                        'general',
                        $student['id']
                    );
                    
                    // Log activity
                    logParentActivity($parent_id, 'REGISTRATION', 'Parent account created');
                    
                    // Automatically log in the parent
                    $_SESSION['parent_logged_in'] = true;
                    $_SESSION['parent_id'] = $parent_id;
                    $_SESSION['parent_name'] = $full_name;
                    $_SESSION['parent_email'] = $email;
                    $_SESSION['academy_reference'] = $student['academy_reference'];
                    
                    // Log the auto-login
                    logParentActivity($parent_id, 'AUTO_LOGIN', 'Automatically logged in after registration');
                    
                    // Redirect to dashboard
                    header('Location: parent/dashboard.php');
                    exit;
                }
            }
        } catch (PDOException $e) {
            error_log("Parent registration error: " . $e->getMessage());
            $error = "An error occurred during registration. Please try again.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Parent Registration - School Management System</title>

      <!-- Favicon -->
      <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">

    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 20px 0;
        }
        .registration-container {
            max-width: 500px;
            margin: 0 auto;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 1rem 1rem 0 0 !important;
            padding: 2rem;
        }
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102,126,234,0.4);
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102,126,234,0.25);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="registration-container">
            <div class="card">
                <div class="card-header text-center">
                    <h2 class="mb-2"><i class="bi bi-person-plus-fill"></i> Parent Registration</h2>
                    <p class="mb-0">Create an account to manage your child's education</p>
                </div>
                <div class="card-body p-4">
                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <i class="bi bi-exclamation-triangle-fill me-2"></i><?= htmlspecialchars($error) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($success): ?>
                        <div class="alert alert-success text-center" role="alert">
                            <div class="spinner-border text-success mb-3" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <h5><i class="bi bi-check-circle-fill me-2"></i>Account Created Successfully!</h5>
                            <p class="mb-0">Redirecting to your dashboard...</p>
                        </div>
                    <?php else: ?>
                    
                    <form method="POST" action="">
                        <div class="mb-3">
                            <label for="full_name" class="form-label">Full Name <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-person"></i></span>
                                <input type="text" class="form-control" id="full_name" name="full_name" 
                                       value="<?= htmlspecialchars($_POST['full_name'] ?? '') ?>" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="email" class="form-label">Email Address <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-envelope"></i></span>
                                <input type="email" class="form-control" id="email" name="email" 
                                       value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required>
                            </div>
                            <small class="text-muted">You'll use this to login</small>
                        </div>
                        
                        <div class="mb-3">
                            <label for="phone" class="form-label">Phone Number <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text">+268</span>
                                <input type="tel" class="form-control" id="phone" name="phone" 
                                       value="<?= htmlspecialchars($_POST['phone'] ?? '') ?>" 
                                       placeholder="7687 3207" required>
                            </div>
                            <small class="text-muted">For payment notifications</small>
                        </div>
                        
                        <div class="mb-3">
                            <label for="student_username" class="form-label">Student Username <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-person-badge"></i></span>
                                <input type="text" class="form-control" id="student_username" name="student_username" 
                                       value="<?= htmlspecialchars($_POST['student_username'] ?? '') ?>" 
                                       onblur="checkStudent()" required>
                                <button class="btn btn-outline-secondary" type="button" onclick="checkStudent()" id="check_btn">
                                    <i class="bi bi-search"></i> Check
                                </button>
                            </div>
                            <small class="text-muted">Enter your child's username and click "Check" to verify</small>
                            
                            <!-- Student Info Display (Success) -->
                            <div id="student_info" style="display: none;" class="mt-3">
                                <div class="alert alert-success mb-0">
                                    <i class="bi bi-check-circle me-2"></i>
                                    <strong>Student Found!</strong>
                                    <div class="mt-2" style="border-left: 3px solid #10b981; padding-left: 15px;">
                                        <div class="mb-1"><strong>Student Name:</strong> <span id="student_name" class="fw-bold"></span></div>
                                        <div class="mb-1"><strong>Registering for:</strong> <span id="student_school" class="badge bg-primary fs-6"></span></div>
                                        <div><strong>Grade:</strong> <span id="student_grade"></span></div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Student Not Found (Error) -->
                            <div id="student_error" style="display: none;" class="mt-3">
                                <div class="alert alert-danger mb-0">
                                    <i class="bi bi-x-circle me-2"></i>
                                    <strong>Student not found!</strong>
                                    <p class="mb-0 mt-2" id="error_message"></p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="relationship" class="form-label">Relationship <span class="text-danger">*</span></label>
                            <select class="form-select" id="relationship" name="relationship" required>
                                <option value="parent">Parent</option>
                                <option value="mother">Mother</option>
                                <option value="father">Father</option>
                                <option value="guardian">Guardian</option>
                                <option value="grandparent">Grandparent</option>
                                <option value="other">Other</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label for="password" class="form-label">Password <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-lock"></i></span>
                                <input type="password" class="form-control" id="password" name="password" 
                                       minlength="6" required>
                            </div>
                            <small class="text-muted">At least 6 characters</small>
                        </div>
                        
                        <div class="mb-4">
                            <label for="confirm_password" class="form-label">Confirm Password <span class="text-danger">*</span></label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-lock-fill"></i></span>
                                <input type="password" class="form-control" id="confirm_password" name="confirm_password" 
                                       minlength="6" required>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-primary btn-lg w-100 mb-3">
                            <i class="bi bi-check-circle me-2"></i>Create Account
                        </button>
                        
                        <div class="text-center">
                            <p class="mb-0">
                                Already have an account? 
                                <a href="parent_login.php" class="text-decoration-none fw-bold">Login here</a>
                            </p>
                        </div>
                    </form>
                    
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="text-center mt-4">
                <a href="index.php" class="text-white text-decoration-none">
                    <i class="bi bi-arrow-left me-2"></i>Back to Home
                </a>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Check if student exists and show school info
        async function checkStudent() {
            const username = document.getElementById('student_username').value.trim();
            const infoDiv = document.getElementById('student_info');
            const errorDiv = document.getElementById('student_error');
            const checkBtn = document.getElementById('check_btn');
            
            // Hide previous results
            infoDiv.style.display = 'none';
            errorDiv.style.display = 'none';
            
            if (!username) {
                return;
            }
            
            // Show loading on button
            checkBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span>';
            checkBtn.disabled = true;
            
            try {
                // Call API to check student
                const response = await fetch('check_student.php?username=' + encodeURIComponent(username));
                const data = await response.json();
                
                if (data.success) {
                    // Student found - show info
                    document.getElementById('student_name').textContent = data.student.full_name;
                    document.getElementById('student_school').textContent = data.student.school_name + ' (' + data.student.academy_reference + ')';
                    document.getElementById('student_grade').textContent = data.student.grade_name;
                    infoDiv.style.display = 'block';
                } else {
                    // Student not found - show error
                    document.getElementById('error_message').textContent = data.message || 'Please check the username and try again.';
                    errorDiv.style.display = 'block';
                }
            } catch (error) {
                // Network error
                document.getElementById('error_message').textContent = 'Error checking student. Please try again.';
                errorDiv.style.display = 'block';
            } finally {
                // Reset button
                checkBtn.innerHTML = '<i class="bi bi-search"></i> Check';
                checkBtn.disabled = false;
            }
        }
    </script>
</body>
</html>
