<?php
/**
 * Card Payment Return Handler (PayGate)
 * Handles the return from PayGate after card payment
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/wallet_functions.php';

startSession();

// Get PayGate response parameters
$pay_request_id = $_GET['PAY_REQUEST_ID'] ?? null;
$reference = $_GET['REFERENCE'] ?? null;

if (!$pay_request_id || !$reference) {
    header('Location: ../../register.php?error=invalid_return');
    exit;
}

try {
    $pdo = getDB();
    
    // Query PayGate for transaction status
    $query_data = [
        'PAYGATE_ID' => PAYGATE_ID,
        'PAY_REQUEST_ID' => $pay_request_id,
        'REFERENCE' => $reference
    ];
    
    // Generate checksum
    $checksum_string = '';
    foreach ($query_data as $value) {
        $checksum_string .= $value;
    }
    $checksum_string .= PAYGATE_SECRET;
    $query_data['CHECKSUM'] = md5($checksum_string);
    
    // Query PayGate
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://secure.paygate.co.za/payweb3/query.trans');
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($query_data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200 || !$response) {
        throw new Exception('Failed to query payment status');
    }
    
    // Parse response
    parse_str($response, $result);
    
    // Verify checksum
    $verify_checksum = '';
    foreach ($result as $key => $value) {
        if ($key !== 'CHECKSUM' && $key !== 'CHECKSUM_MD5') {
            $verify_checksum .= $value;
        }
    }
    $verify_checksum .= PAYGATE_SECRET;
    
    if (md5($verify_checksum) !== ($result['CHECKSUM'] ?? '')) {
        throw new Exception('Invalid checksum in response');
    }
    
    // Get payment record
    $stmt = $pdo->prepare("SELECT * FROM payments WHERE reference_id = ?");
    $stmt->execute([$reference]);
    $payment = $stmt->fetch();
    
    if (!$payment) {
        throw new Exception('Payment record not found');
    }
    
    // PayGate status codes:
    // 1 = Approved
    // 2 = Declined
    // 4 = Cancelled
    $transaction_status = $result['TRANSACTION_STATUS'] ?? null;
    
    if ($transaction_status == '1') {
        // Payment successful
        $new_status = 'SUCCESSFUL';
        
        // Update payment
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET status = 'SUCCESSFUL',
                momo_status = 'APPROVED',
                transaction_id = ?,
                card_type = ?,
                card_last4 = ?,
                raw_response = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $result['TRANSACTION_ID'] ?? null,
            $result['PAY_METHOD_DETAIL'] ?? null,
            substr($result['PAY_METHOD_DETAIL'] ?? '', -4),
            json_encode($result),
            $payment['id']
        ]);
        
        // Update card transaction
        $stmt = $pdo->prepare("
            UPDATE card_transactions 
            SET transaction_id = ?,
                card_type = ?,
                card_last4 = ?,
                auth_code = ?,
                status_code = ?,
                status_description = 'Approved',
                raw_response = ?,
                updated_at = NOW()
            WHERE payment_id = ?
        ");
        $stmt->execute([
            $result['TRANSACTION_ID'] ?? null,
            $result['PAY_METHOD'] ?? null,
            substr($result['PAY_METHOD_DETAIL'] ?? '', -4),
            $result['AUTH_CODE'] ?? null,
            $transaction_status,
            json_encode($result),
            $payment['id']
        ]);
        
        // Update student payment status
        $stmt = $pdo->prepare("
            UPDATE students 
            SET payment_status = 'PAID',
                payment_date = NOW(),
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$payment['student_id']]);
        
        // Credit admin wallet (deduct platform fee)
        creditAdminWallet($payment['id'], PLATFORM_ADMIN_ID, $academy_reference ?? 'KINE');
        
        // Clear session
        unset($_SESSION['pending_payment_student_id']);
        unset($_SESSION['pending_payment_amount']);
        
        // Redirect to success page
        header('Location: ../../success.php?payment=card&reference=' . urlencode($reference));
        exit;
        
    } else {
        // Payment failed or cancelled
        $status_desc = 'Declined';
        if ($transaction_status == '4') {
            $status_desc = 'Cancelled';
        }
        
        // Update payment
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET status = 'FAILED',
                momo_status = ?,
                error_message = ?,
                raw_response = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $status_desc,
            $result['RESULT_DESC'] ?? 'Payment declined',
            json_encode($result),
            $payment['id']
        ]);
        
        // Update card transaction
        $stmt = $pdo->prepare("
            UPDATE card_transactions 
            SET status_code = ?,
                status_description = ?,
                raw_response = ?,
                updated_at = NOW()
            WHERE payment_id = ?
        ");
        $stmt->execute([
            $transaction_status,
            $status_desc,
            json_encode($result),
            $payment['id']
        ]);
        
        // Redirect to error page
        header('Location: ../../register.php?payment_error=card_declined');
        exit;
    }
    
} catch (Exception $e) {
    error_log("Card payment return error: " . $e->getMessage());
    header('Location: ../../register.php?payment_error=system_error');
    exit;
}
?>

