<?php
session_start();
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';

// Check if student is logged in or if student_id is provided
if (!isset($_SESSION['student_id']) && !isset($_GET['student_id'])) {
    header('Location: ../login.php');
    exit;
}

$student_id = $_SESSION['student_id'] ?? $_GET['student_id'];

// Get student details
$student = getStudentById($student_id);
if (!$student) {
    die('Student not found');
}

// Get selected subjects from student's registration
$pdo = getDB();
$selected_subjects = [];
$stmt = $pdo->prepare('SELECT subject_id FROM student_subject WHERE student_id = ?');
$stmt->execute([$student_id]);
$selected_subjects = $stmt->fetchAll(PDO::FETCH_COLUMN);

// If no subjects from registration, check URL parameters
if (empty($selected_subjects)) {
    if (isset($_GET['subjects'])) {
        $selected_subjects = explode(',', $_GET['subjects']);
    } elseif (isset($_POST['subjects'])) {
        $selected_subjects = $_POST['subjects'];
    }
}

if (empty($selected_subjects)) {
    die('No subjects selected for payment. Please register for subjects first.');
}

// Get subject details and calculate total
$subjects = getSubjectsByIds($selected_subjects);
$total_amount = count($subjects) * SUBJECT_PRICE_SZL;

// Handle payment request
$payment_error = null;
$payment_success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['initiate_payment'])) {
    try {
        // Generate payment reference
        $payment_ref = 'PAY-' . $student_id . '-' . time() . '-' . uniqid();
        
        // Create payment record
        $payment_id = createPaymentRecord($student_id, $total_amount, $payment_ref, $selected_subjects);
        
        if ($payment_id) {
            // Initiate MTN MoMo payment
            $err = null;
            $token = momo_get_access_token($err);
            
            if ($token) {
                $amount = number_format($total_amount, 2, '.', '');
                $currency = MOMO_CURRENCY;
                $msisdn = $student['phone'] ?? '26876873207'; // Use student's registered phone number
                $external_id = 'EXT-' . $payment_id;
                
                [$code, $response] = momo_request_to_pay(
                    $token, 
                    $payment_ref, 
                    $amount, 
                    $currency, 
                    $msisdn, 
                    $external_id,
                    'Payment for ' . count($subjects) . ' subject(s)',
                    'Skolo-Kine Education Payment'
                );
                
                if ($code === 202) {
                    // Payment request created successfully
                    updatePaymentStatus($payment_id, 'PENDING', $payment_ref);
                    $payment_success = true;
                } else {
                    $payment_error = 'Payment request failed: ' . $response;
                }
            } else {
                $payment_error = 'Failed to get payment token: ' . $err;
            }
        } else {
            $payment_error = 'Failed to create payment record';
        }
    } catch (Exception $e) {
        $payment_error = 'Payment error: ' . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment - Skolo-Kine</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .payment-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        .subject-item {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
            border-left: 4px solid #667eea;
        }
        .total-amount {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            border-radius: 15px;
            padding: 20px;
            text-align: center;
        }
        .btn-pay {
            background: linear-gradient(135deg, #28a745, #20c997);
            border: none;
            border-radius: 25px;
            padding: 15px 40px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 1px;
            transition: all 0.3s ease;
        }
        .btn-pay:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(40, 167, 69, 0.3);
        }
        .momo-logo {
            width: 40px;
            height: 40px;
            background: #ff6b35;
            border-radius: 50%;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="payment-card p-4">
                    <div class="text-center mb-4">
                        <h2 class="text-primary mb-3">
                            <i class="fas fa-credit-card me-2"></i>
                            Complete Your Payment
                        </h2>
                        <p class="text-muted">Pay securely with MTN MoMo Pay</p>
                    </div>

                    <?php if ($payment_error): ?>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <?php echo htmlspecialchars($payment_error); ?>
                        </div>
                    <?php endif; ?>

                    <?php if ($payment_success): ?>
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle me-2"></i>
                            Payment request sent successfully! Please check your phone for the MTN MoMo payment prompt.
                        </div>
                    <?php else: ?>
                        <form method="POST">
                            <div class="row">
                                <div class="col-md-8">
                                    <h5 class="mb-3">Selected Subjects</h5>
                                    <?php foreach ($subjects as $subject): ?>
                                        <div class="subject-item">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <h6 class="mb-1"><?php echo htmlspecialchars($subject['name']); ?></h6>
                                                    <small class="text-muted">Grade <?php echo htmlspecialchars($subject['grade']); ?></small>
                                                </div>
                                                <span class="badge bg-primary"><?php echo MOMO_CURRENCY; ?> <?php echo number_format(SUBJECT_PRICE_SZL, 2); ?></span>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                                
                                <div class="col-md-4">
                                    <div class="total-amount">
                                        <h4>Total Amount</h4>
                                        <h2 class="mb-3"><?php echo MOMO_CURRENCY; ?> <?php echo number_format($total_amount, 2); ?></h2>
                                        <p class="mb-0"><?php echo count($subjects); ?> subject(s)</p>
                                    </div>
                                </div>
                            </div>

                            <div class="mt-4">
                                <h5 class="mb-3">Payment Method</h5>
                                <div class="d-flex align-items-center p-3 bg-light rounded">
                                    <div class="momo-logo">M</div>
                                    <div>
                                        <h6 class="mb-1">MTN MoMo Pay</h6>
                                        <small class="text-muted">Secure mobile money payment</small>
                                    </div>
                                </div>
                                <div class="alert alert-info mt-3">
                                    <i class="fas fa-info-circle me-2"></i>
                                    <strong>Payment will be sent to:</strong> <?php echo htmlspecialchars($student['phone']); ?><br>
                                    <small>Make sure this number has an active MTN MoMo account. You will receive a payment prompt on your phone.</small>
                                </div>
                            </div>

                            <div class="mt-4">
                                <h5 class="mb-3">Student Information</h5>
                                <div class="row">
                                    <div class="col-md-6">
                                        <p><strong>Name:</strong> <?php echo htmlspecialchars($student['firstname'] . ' ' . $student['lastname']); ?></p>
                                    </div>
                                    <div class="col-md-6">
                                        <p><strong>Email:</strong> <?php echo htmlspecialchars($student['email']); ?></p>
                                    </div>
                                </div>
                            </div>

                            <div class="text-center mt-4">
                                <button type="submit" name="initiate_payment" class="btn btn-pay btn-lg">
                                    <i class="fas fa-mobile-alt me-2"></i>
                                    Pay with MTN MoMo
                                </button>
                            </div>
                        </form>
                    <?php endif; ?>

                    <div class="text-center mt-4">
                        <a href="../dashboard.php" class="btn btn-outline-secondary">
                            <i class="fas fa-arrow-left me-2"></i>
                            Back to Dashboard
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>