<?php
/**
 * Quick Multi-Tenant System Test
 * Run this to verify everything is working
 */

require_once 'config.php';

echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Skolo-Kine Multi-Tenant Quick Test                       ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n\n";

$pdo = getDB();
$allPassed = true;

// Test 1: Check Tables
echo "Test 1: Checking database tables...\n";
$requiredTables = ['academy_references', 'tenant_settings', 'admin_wallet', 'platform_fees'];
foreach ($requiredTables as $table) {
    try {
        $result = $pdo->query("SHOW TABLES LIKE '$table'");
        if ($result->fetch()) {
            echo "  ✓ $table exists\n";
        } else {
            echo "  ✗ $table MISSING\n";
            $allPassed = false;
        }
    } catch (PDOException $e) {
        echo "  ✗ Error checking $table: " . $e->getMessage() . "\n";
        $allPassed = false;
    }
}
echo "\n";

// Test 2: Check Columns
echo "Test 2: Checking required columns...\n";
$columnChecks = [
    'students' => 'academy_reference',
    'payments' => 'academy_reference',
    'payments' => 'payment_method',
    'students' => 'moodle_status'
];

foreach ($columnChecks as $table => $column) {
    try {
        $result = $pdo->query("SHOW COLUMNS FROM $table LIKE '$column'");
        if ($result->fetch()) {
            echo "  ✓ $table.$column exists\n";
        } else {
            echo "  ✗ $table.$column MISSING\n";
            $allPassed = false;
        }
    } catch (PDOException $e) {
        echo "  ✗ Error: " . $e->getMessage() . "\n";
        $allPassed = false;
    }
}
echo "\n";

// Test 3: Check Schools
echo "Test 3: Checking schools...\n";
try {
    $stmt = $pdo->query("SELECT reference_code, academy_name, is_active FROM academy_references");
    $schools = $stmt->fetchAll();
    
    if (count($schools) > 0) {
        echo "  ✓ Found " . count($schools) . " school(s):\n";
        foreach ($schools as $school) {
            $status = $school['is_active'] ? '✓ Active' : '✗ Inactive';
            echo "    - {$school['academy_name']} ({$school['reference_code']}) - $status\n";
        }
    } else {
        echo "  ⚠ No schools found\n";
        echo "    Creating default KINE academy...\n";
        
        $pdo->exec("
            INSERT IGNORE INTO academy_references (admin_id, academy_name, reference_code, is_active)
            VALUES (1, 'Kine Academy', 'KINE', 1)
        ");
        echo "    ✓ KINE academy created\n";
    }
} catch (PDOException $e) {
    echo "  ✗ Error: " . $e->getMessage() . "\n";
    $allPassed = false;
}
echo "\n";

// Test 4: Check Students
echo "Test 4: Checking students assignment...\n";
try {
    $stmt = $pdo->query("
        SELECT academy_reference, COUNT(*) as count 
        FROM students 
        WHERE deleted_at IS NULL 
        GROUP BY academy_reference
    ");
    $studentCounts = $stmt->fetchAll();
    
    if (count($studentCounts) > 0) {
        echo "  ✓ Students per school:\n";
        foreach ($studentCounts as $row) {
            $school = $row['academy_reference'] ?? 'NULL';
            echo "    - $school: {$row['count']} students\n";
        }
    } else {
        echo "  ⚠ No students found\n";
    }
    
    // Check for NULL academy_reference
    $stmt = $pdo->query("SELECT COUNT(*) FROM students WHERE academy_reference IS NULL");
    $nullCount = $stmt->fetchColumn();
    if ($nullCount > 0) {
        echo "  ⚠ $nullCount students without school assignment\n";
        echo "    Fixing...\n";
        $pdo->exec("UPDATE students SET academy_reference = 'KINE' WHERE academy_reference IS NULL");
        echo "    ✓ Assigned to KINE\n";
    } else {
        echo "  ✓ All students have school assignment\n";
    }
} catch (PDOException $e) {
    echo "  ✗ Error: " . $e->getMessage() . "\n";
    $allPassed = false;
}
echo "\n";

// Test 5: Check File Structure
echo "Test 5: Checking key files...\n";
$requiredFiles = [
    'includes/school_context.php',
    'select_school.php',
    'super_admin/dashboard.php',
    'super_admin/schools/create.php',
    'admin/settings/school_settings.php'
];

foreach ($requiredFiles as $file) {
    if (file_exists($file)) {
        echo "  ✓ $file exists\n";
    } else {
        echo "  ✗ $file MISSING\n";
        $allPassed = false;
    }
}
echo "\n";

// Summary
echo "╔════════════════════════════════════════════════════════════╗\n";
echo "║  Test Summary                                              ║\n";
echo "╚════════════════════════════════════════════════════════════╝\n";

if ($allPassed) {
    echo "✅ ALL TESTS PASSED!\n\n";
    echo "Your multi-tenant system is ready to use!\n\n";
    echo "Next steps:\n";
    echo "1. Open: http://localhost/Multi-Tanent/admin/login.php\n";
    echo "2. Login with: admin / admin123\n";
    echo "3. Select a school\n";
    echo "4. Start testing!\n\n";
    echo "For detailed testing: See LOCALHOST_TESTING_GUIDE.md\n\n";
    exit(0);
} else {
    echo "⚠️  Some tests failed!\n\n";
    echo "Please check the errors above.\n";
    echo "Try running: php database/run_all_missing_columns.php\n\n";
    exit(1);
}

