<?php
require_once __DIR__ . '/includes/functions.php';

if (isset($_GET['action']) && $_GET['action'] === 'subjects') {
	$gradeId = (int)($_GET['grade_id'] ?? 0);
	$subjects = $gradeId ? getSubjectsForGrade($gradeId) : [];
	header('Content-Type: application/json');
	echo json_encode($subjects);
	exit;
}

if (isset($_GET['action']) && $_GET['action'] === 'validate_email') {
	$email = trim($_GET['email'] ?? '');
	header('Content-Type: application/json');
	
	if (empty($email)) {
		echo json_encode(['valid' => false, 'message' => 'Email is required']);
		exit;
	}
	
	// Validate email format and existence
	$emailValidation = validateEmailExists($email);
	
	// If email format is valid, also check availability
	if ($emailValidation['valid']) {
		$isAvailable = checkEmailAvailability($email);
		if (!$isAvailable) {
			$emailValidation = ['valid' => false, 'message' => 'This email address is already registered'];
		}
	}
	
	echo json_encode($emailValidation);
	exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
	if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
		http_response_code(400);
		die('Invalid CSRF token');
	}
	try {
		$password = (string)($_POST['password'] ?? '');
		$confirm = (string)($_POST['password_confirm'] ?? '');
		if ($password === '' || strlen($password) < 8) {
			throw new Exception('Password must be at least 8 characters.');
		}
		if ($password !== $confirm) {
			throw new Exception('Passwords do not match.');
		}
		$data = [
			'first_name' => trim($_POST['first_name'] ?? ''),
			'last_name' => trim($_POST['last_name'] ?? ''),
			'email' => trim($_POST['email'] ?? ''),
			'phone' => trim($_POST['phone'] ?? ''),
			'grade_id' => (int)($_POST['grade_id'] ?? 0),
			'subject_ids' => array_filter(array_map('intval', (array)($_POST['subject_ids'] ?? []))),
			'password_plain' => $password
		];
		if ($data['first_name'] === '' || $data['last_name'] === '' || $data['email'] === '' || $data['phone'] === '' || !$data['grade_id']) {
			throw new Exception('All fields are required.');
		}
		
		// Validate email format and existence
		$emailValidation = validateEmailExists($data['email']);
		if (!$emailValidation['valid']) {
			throw new Exception('Email validation failed: ' . $emailValidation['message']);
		}
		
		// Validate phone number format (Swaziland format: 268XXXXXXXX)
		$phoneRegex = '/^268[0-9]{8}$/';
		if (!preg_match($phoneRegex, $data['phone'])) {
			throw new Exception('Please enter a valid Swaziland phone number (format: 268XXXXXXXX, e.g., 26876873207).');
		}
		
		// Check if email is already registered
		if (!checkEmailAvailability($data['email'])) {
			throw new Exception('This email address is already registered. Please use a different email or contact support if you believe this is an error.');
		}
		
		// Check if Grade 7 is selected (only Grade 7 is available for now)
		$pdo = getDB();
		$stmt = $pdo->prepare("SELECT name FROM grades WHERE id = ?");
		$stmt->execute([$data['grade_id']]);
		$grade = $stmt->fetch();
		
		if (!$grade || $grade['name'] !== 'Grade 7') {
			throw new Exception('Only Grade 7 registration is available at the moment.');
		}
		$studentId = createStudent($data);
		
		// Calculate payment amount (number of subjects × price)
		$subject_count = count($data['subject_ids']);
		$payment_amount = $subject_count * SUBJECT_PRICE;
		
		// Store payment info in session for payment selection page
		session_start();
		$_SESSION['pending_payment_student_id'] = $studentId;
		$_SESSION['pending_payment_amount'] = $payment_amount;
		$_SESSION['pending_payment_subjects'] = $data['subject_ids'];
		
		// Redirect to payment selection page
		header('Location: payment_selection.php');
		exit;
	} catch (Throwable $e) {
		header('Content-Type: text/html; charset=utf-8');
		echo '<!doctype html><html><body style="font-family:system-ui,Segoe UI,Arial">';
		echo '<h3>Registration failed</h3>';
		echo '<p style="color:#b00020">' . htmlspecialchars($e->getMessage()) . '</p>';
		echo '<p><a href="index.php">Back to form</a></p>';
		echo '</body></html>';
		exit;
	}
}

http_response_code(405);
header('Allow: POST');
echo 'Method Not Allowed';
