<?php
/**
 * Complete Moodle Workflow Test
 * Tests the entire student sync, enrollment, and suspension workflow
 */

session_start();
require_once 'config.php';
require_once 'includes/functions.php';
require_once 'includes/moodle_enrollment_handler.php';
require_once 'includes/moodle_suspension_handler.php';

// Security check
if (!isset($_SESSION['admin_id']) && !isset($_SESSION['user_id'])) {
    $secret_key = $_GET['key'] ?? '';
    if ($secret_key !== 'test123') {
        die('Access denied. Login as admin or use ?key=test123');
    }
}

$pdo = getDB();
$test_log = [];
$overall_success = true;

function logTest($title, $success, $message, $data = null) {
    global $test_log, $overall_success;
    $test_log[] = [
        'title' => $title,
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('H:i:s')
    ];
    if (!$success) {
        $overall_success = false;
    }
}

echo "<!DOCTYPE html>
<html lang='en'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Complete Moodle Workflow Test</title>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css' rel='stylesheet'>
    <link href='https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css' rel='stylesheet'>
    <style>
        body { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); min-height: 100vh; padding: 20px; }
        .test-container { background: white; border-radius: 15px; padding: 30px; max-width: 1000px; margin: 0 auto; box-shadow: 0 10px 40px rgba(0,0,0,0.2); }
        .step-box { border-left: 4px solid #667eea; padding: 15px; margin: 15px 0; background: #f8f9fa; border-radius: 5px; }
        .step-success { border-left-color: #28a745; background: #f1f9f3; }
        .step-failed { border-left-color: #dc3545; background: #fdf4f4; }
        .log-entry { padding: 10px; margin: 5px 0; border-radius: 5px; font-family: monospace; font-size: 12px; }
        .code-output { background: #f8f9fa; border: 1px solid #dee2e6; padding: 10px; border-radius: 5px; max-height: 300px; overflow-y: auto; font-family: monospace; font-size: 11px; }
    </style>
</head>
<body>
<div class='container'>
<div class='test-container'>";

echo "<h1 class='text-center mb-4'><i class='bi bi-mortarboard-fill' style='color: #667eea;'></i> Complete Moodle Workflow Test</h1>";
echo "<p class='text-center text-muted mb-4'>Testing: User Sync → Course Enrollment → Suspension → Unsuspension</p>";
echo "<hr>";

// =====================================================
// STEP 1: Check Moodle Connection
// =====================================================
echo "<h4><i class='bi bi-1-circle-fill text-primary'></i> Testing Moodle Connection</h4>";

$handler = new MoodleSuspensionHandler();
$test_result = $handler->callMoodleAPI('core_webservice_get_site_info');

if (isset($test_result['sitename'])) {
    logTest('Moodle Connection', true, 'Connected to: ' . $test_result['sitename'], $test_result);
    echo "<div class='step-box step-success'>";
    echo "<i class='bi bi-check-circle-fill text-success'></i> <strong>Connected Successfully!</strong><br>";
    echo "Site: " . htmlspecialchars($test_result['sitename']) . "<br>";
    echo "Version: " . htmlspecialchars($test_result['release'] ?? 'Unknown') . "<br>";
    echo "Users: " . number_format($test_result['usercount'] ?? 0);
    echo "</div>";
} else {
    logTest('Moodle Connection', false, 'Failed to connect', $test_result);
    echo "<div class='step-box step-failed'>";
    echo "<i class='bi bi-x-circle-fill text-danger'></i> <strong>Connection Failed!</strong><br>";
    echo "Error: " . ($test_result['message'] ?? 'Unknown error');
    echo "</div>";
    $overall_success = false;
}

// =====================================================
// STEP 2: Check Students in Database
// =====================================================
echo "<h4 class='mt-4'><i class='bi bi-2-circle-fill text-primary'></i> Checking Students in Database</h4>";

$stmt = $pdo->query("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN moodle_user_id IS NOT NULL THEN 1 ELSE 0 END) as synced,
        SUM(CASE WHEN moodle_user_id IS NULL THEN 1 ELSE 0 END) as not_synced
    FROM students
");
$student_stats = $stmt->fetch();

echo "<div class='step-box step-success'>";
echo "<strong>Database Status:</strong><br>";
echo "Total Students: <strong>{$student_stats['total']}</strong><br>";
echo "Synced to Moodle: <strong class='text-success'>{$student_stats['synced']}</strong><br>";
echo "Not Synced: <strong class='text-warning'>{$student_stats['not_synced']}</strong>";
echo "</div>";

logTest('Student Database Check', true, "Total: {$student_stats['total']}, Synced: {$student_stats['synced']}", $student_stats);

// =====================================================
// STEP 3: Verify Synced Students in Moodle
// =====================================================
echo "<h4 class='mt-4'><i class='bi bi-3-circle-fill text-primary'></i> Verifying Students Exist in Moodle</h4>";

$stmt = $pdo->query("
    SELECT id, full_name, username, moodle_user_id, academy_reference 
    FROM students 
    WHERE moodle_user_id IS NOT NULL 
    LIMIT 5
");
$synced_students = $stmt->fetchAll();

$verified_count = 0;
$failed_count = 0;

foreach ($synced_students as $student) {
    $result = $handler->callMoodleAPI('core_user_get_users_by_field', [
        'field' => 'id',
        'values' => [(int)$student['moodle_user_id']]
    ]);
    
    if (!empty($result) && !isset($result['exception'])) {
        $moodle_user = $result[0];
        $verified_count++;
        echo "<div class='log-entry bg-light'>";
        echo "<i class='bi bi-check-circle text-success'></i> ";
        echo "<strong>{$student['full_name']}</strong> → ";
        echo "Moodle User: {$moodle_user['username']} (ID: {$moodle_user['id']})";
        if (isset($moodle_user['suspended']) && $moodle_user['suspended']) {
            echo " <span class='badge bg-warning'>SUSPENDED</span>";
        }
        echo "</div>";
    } else {
        $failed_count++;
        echo "<div class='log-entry bg-danger text-white'>";
        echo "<i class='bi bi-x-circle'></i> ";
        echo "<strong>{$student['full_name']}</strong> → NOT FOUND in Moodle!";
        echo "</div>";
    }
}

echo "<div class='step-box " . ($failed_count > 0 ? 'step-failed' : 'step-success') . " mt-2'>";
echo "<strong>Verification Result:</strong> {$verified_count} verified, {$failed_count} failed";
echo "</div>";

logTest('Verify Students in Moodle', $failed_count === 0, "Verified: {$verified_count}, Failed: {$failed_count}");

// =====================================================
// STEP 4: Test Suspension Feature
// =====================================================
echo "<h4 class='mt-4'><i class='bi bi-4-circle-fill text-primary'></i> Testing Suspension Feature</h4>";

// Get a student to test with
$stmt = $pdo->query("
    SELECT * FROM students 
    WHERE moodle_user_id IS NOT NULL 
    AND suspended = 0
    LIMIT 1
");
$test_student = $stmt->fetch();

if ($test_student) {
    echo "<p><strong>Test Student:</strong> {$test_student['full_name']} (ID: {$test_student['id']}, Moodle ID: {$test_student['moodle_user_id']})</p>";
    
    // Test Suspension
    echo "<div class='mt-3'><strong>Testing Suspension...</strong></div>";
    $suspend_result = $handler->suspendUser($test_student['id'], 'test_suspension');
    
    if ($suspend_result['success']) {
        echo "<div class='step-box step-success'>";
        echo "<i class='bi bi-check-circle-fill text-success'></i> <strong>Suspension Successful!</strong><br>";
        echo "Student suspended in Moodle<br>";
        echo "Database updated: suspended = 1";
        echo "</div>";
        logTest('Suspend Student', true, 'Successfully suspended student in Moodle');
        
        // Verify in Moodle
        $verify_result = $handler->callMoodleAPI('core_user_get_users_by_field', [
            'field' => 'id',
            'values' => [(int)$test_student['moodle_user_id']]
        ]);
        
        if (!empty($verify_result) && $verify_result[0]['suspended'] == 1) {
            echo "<div class='alert alert-success mt-2'>";
            echo "<i class='bi bi-check-circle-fill'></i> <strong>Verified in Moodle:</strong> User is suspended";
            echo "</div>";
        }
        
        // Wait a moment, then unsuspend
        sleep(1);
        
        echo "<div class='mt-3'><strong>Testing Unsuspension...</strong></div>";
        $unsuspend_result = $handler->unsuspendUser($test_student['id']);
        
        if ($unsuspend_result['success']) {
            echo "<div class='step-box step-success'>";
            echo "<i class='bi bi-check-circle-fill text-success'></i> <strong>Unsuspension Successful!</strong><br>";
            echo "Student unsuspended in Moodle<br>";
            echo "Database updated: suspended = 0";
            echo "</div>";
            logTest('Unsuspend Student', true, 'Successfully unsuspended student in Moodle');
            
            // Verify in Moodle
            $verify_result2 = $handler->callMoodleAPI('core_user_get_users_by_field', [
                'field' => 'id',
                'values' => [(int)$test_student['moodle_user_id']]
            ]);
            
            if (!empty($verify_result2) && $verify_result2[0]['suspended'] == 0) {
                echo "<div class='alert alert-success mt-2'>";
                echo "<i class='bi bi-check-circle-fill'></i> <strong>Verified in Moodle:</strong> User is active (not suspended)";
                echo "</div>";
            }
        } else {
            echo "<div class='step-box step-failed'>";
            echo "<i class='bi bi-x-circle-fill text-danger'></i> <strong>Unsuspension Failed!</strong><br>";
            echo "Error: " . htmlspecialchars($unsuspend_result['error']);
            echo "</div>";
            logTest('Unsuspend Student', false, $unsuspend_result['error']);
        }
        
    } else {
        echo "<div class='step-box step-failed'>";
        echo "<i class='bi bi-x-circle-fill text-danger'></i> <strong>Suspension Failed!</strong><br>";
        echo "Error: " . htmlspecialchars($suspend_result['error']);
        echo "</div>";
        logTest('Suspend Student', false, $suspend_result['error']);
    }
    
} else {
    echo "<div class='alert alert-warning'>";
    echo "<i class='bi bi-exclamation-triangle'></i> No students found with Moodle sync. Please sync students first.";
    echo "</div>";
    logTest('Suspension Test', false, 'No synced students available for testing');
}

// =====================================================
// STEP 5: Check Subject-Course Mappings
// =====================================================
echo "<h4 class='mt-4'><i class='bi bi-5-circle-fill text-primary'></i> Checking Subject-Course Mappings</h4>";

$stmt = $pdo->query("
    SELECT 
        COUNT(*) as total_mappings,
        SUM(CASE WHEN auto_enroll = 1 THEN 1 ELSE 0 END) as auto_enroll_enabled,
        SUM(CASE WHEN sync_enabled = 1 THEN 1 ELSE 0 END) as sync_enabled
    FROM subject_moodle_mapping
");
$mapping_stats = $stmt->fetch();

if ($mapping_stats['total_mappings'] > 0) {
    echo "<div class='step-box step-success'>";
    echo "<strong>Mappings Found:</strong> {$mapping_stats['total_mappings']}<br>";
    echo "<strong>Auto-Enroll Enabled:</strong> {$mapping_stats['auto_enroll_enabled']}<br>";
    echo "<strong>Sync Enabled:</strong> {$mapping_stats['sync_enabled']}";
    echo "</div>";
    
    // Show some mappings
    $stmt = $pdo->query("
        SELECT s.name as subject_name, s.code as subject_code,
               smm.moodle_course_shortname, smm.moodle_course_id,
               smm.moodle_course_idnumber, smm.auto_enroll, smm.sync_enabled
        FROM subject_moodle_mapping smm
        JOIN subjects s ON smm.subject_id = s.id
        LIMIT 5
    ");
    $mappings = $stmt->fetchAll();
    
    if (!empty($mappings)) {
        echo "<table class='table table-sm mt-2'>";
        echo "<thead><tr><th>Subject</th><th>Moodle Course</th><th>Auto-Enroll</th><th>Sync</th></tr></thead>";
        echo "<tbody>";
        foreach ($mappings as $map) {
            $course_display = htmlspecialchars($map['moodle_course_shortname'] ?? 'N/A');
            if (!empty($map['moodle_course_id'])) {
                $course_display .= " (ID: " . htmlspecialchars($map['moodle_course_id']) . ")";
            }
            echo "<tr>";
            echo "<td>" . htmlspecialchars($map['subject_name']) . "</td>";
            echo "<td>{$course_display}</td>";
            echo "<td>" . ($map['auto_enroll'] ? '✓' : '✗') . "</td>";
            echo "<td>" . ($map['sync_enabled'] ? '✓' : '✗') . "</td>";
            echo "</tr>";
        }
        echo "</tbody></table>";
    }
    
    logTest('Subject-Course Mappings', true, "Found {$mapping_stats['total_mappings']} mappings");
} else {
    echo "<div class='step-box step-failed'>";
    echo "<i class='bi bi-exclamation-triangle text-warning'></i> ";
    echo "<strong>No mappings found!</strong> Configure subject-course mappings in admin dashboard.";
    echo "</div>";
    logTest('Subject-Course Mappings', false, 'No mappings configured');
}

// =====================================================
// STEP 6: Test Enrollment Handler
// =====================================================
echo "<h4 class='mt-4'><i class='bi bi-6-circle-fill text-primary'></i> Testing Enrollment Handler</h4>";

if ($test_student && $mapping_stats['total_mappings'] > 0) {
    try {
        // Get subjects for test student
        $stmt = $pdo->prepare("
            SELECT subject_id FROM student_subject WHERE student_id = ?
        ");
        $stmt->execute([$test_student['id']]);
        $subject_ids = array_column($stmt->fetchAll(), 'subject_id');
        
        if (!empty($subject_ids)) {
            $enrollment_handler = new MoodleEnrollmentHandler($test_student['academy_reference'], $pdo);
            $enroll_result = $enrollment_handler->enrollStudentInMultipleSubjects($test_student['id'], $subject_ids);
            
            echo "<div class='step-box step-success'>";
            echo "<i class='bi bi-check-circle-fill text-success'></i> <strong>Enrollment Handler Working!</strong><br>";
            echo "Student: {$test_student['full_name']}<br>";
            echo "Subjects to enroll: " . count($subject_ids);
            echo "</div>";
            logTest('Enrollment Handler', true, 'Successfully processed enrollment');
        } else {
            echo "<div class='alert alert-warning'>";
            echo "<i class='bi bi-info-circle'></i> Test student has no subjects assigned";
            echo "</div>";
        }
    } catch (Exception $e) {
        echo "<div class='step-box step-failed'>";
        echo "<i class='bi bi-x-circle-fill text-danger'></i> <strong>Enrollment Error!</strong><br>";
        echo "Error: " . htmlspecialchars($e->getMessage());
        echo "</div>";
        logTest('Enrollment Handler', false, $e->getMessage());
    }
} else {
    echo "<div class='alert alert-info'>";
    echo "<i class='bi bi-info-circle'></i> Skipped - No synced students or mappings available";
    echo "</div>";
}

// =====================================================
// STEP 7: Check Moodle Sync Logs
// =====================================================
echo "<h4 class='mt-4'><i class='bi bi-7-circle-fill text-primary'></i> Checking Moodle Sync Logs</h4>";

try {
    $stmt = $pdo->query("
        SELECT 
            COUNT(*) as total_logs,
            SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) as successful,
            SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
            MAX(created_at) as last_sync
        FROM moodle_sync_log
    ");
    $log_stats = $stmt->fetch();
} catch (Exception $e) {
    // Table might not exist
    $log_stats = ['total_logs' => 0, 'successful' => 0, 'failed' => 0, 'last_sync' => null];
    echo "<div class='alert alert-warning'>";
    echo "<i class='bi bi-info-circle'></i> Moodle sync log table not found. This is normal if you haven't run any syncs yet.";
    echo "</div>";
}

if ($log_stats['total_logs'] > 0) {
    echo "<div class='step-box step-success'>";
    echo "<strong>Sync History:</strong><br>";
    echo "Total Syncs: {$log_stats['total_logs']}<br>";
    echo "Successful: <span class='text-success'>{$log_stats['successful']}</span><br>";
    echo "Failed: <span class='text-danger'>{$log_stats['failed']}</span><br>";
    echo "Last Sync: " . ($log_stats['last_sync'] ? date('M j, Y g:i A', strtotime($log_stats['last_sync'])) : 'Never');
    echo "</div>";
    
    // Show recent logs
    try {
        $stmt = $pdo->query("
            SELECT * FROM moodle_sync_log 
            ORDER BY created_at DESC 
            LIMIT 5
        ");
        $recent_logs = $stmt->fetchAll();
    } catch (Exception $e) {
        $recent_logs = [];
    }
    
    echo "<details class='mt-2'>";
    echo "<summary class='btn btn-sm btn-outline-secondary'>View Recent Sync Logs</summary>";
    echo "<div class='code-output mt-2'>";
    foreach ($recent_logs as $log) {
        echo date('Y-m-d H:i:s', strtotime($log['created_at'])) . " | ";
        echo "Type: {$log['sync_type']} | ";
        echo "Student: {$log['student_id']} | ";
        echo "Status: {$log['status']}";
        if ($log['error_message']) {
            echo " | Error: {$log['error_message']}";
        }
        echo "\n";
    }
    echo "</div></details>";
    
    logTest('Sync Logs', true, "Total logs: {$log_stats['total_logs']}, Success rate: " . round(($log_stats['successful'] / $log_stats['total_logs']) * 100, 1) . "%");
} else {
    echo "<div class='alert alert-warning'>";
    echo "<i class='bi bi-info-circle'></i> No sync logs found. Run the Moodle sync bot to populate logs.";
    echo "</div>";
    logTest('Sync Logs', true, 'No logs yet (bot not run)');
}

// =====================================================
// STEP 8: Check Suspension Logs
// =====================================================
echo "<h4 class='mt-4'><i class='bi bi-8-circle-fill text-primary'></i> Checking Suspension System</h4>";

$stmt = $pdo->query("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active_suspensions,
        SUM(CASE WHEN status = 'lifted' THEN 1 ELSE 0 END) as lifted_suspensions,
        SUM(CASE WHEN moodle_suspended = 1 THEN 1 ELSE 0 END) as moodle_synced
    FROM suspension_logs
");
$suspension_stats = $stmt->fetch();

echo "<div class='step-box step-success'>";
echo "<strong>Suspension System Status:</strong><br>";
echo "Total Suspensions: {$suspension_stats['total']}<br>";
echo "Active: {$suspension_stats['active_suspensions']}<br>";
echo "Lifted: {$suspension_stats['lifted_suspensions']}<br>";
echo "Moodle Synced: {$suspension_stats['moodle_synced']}";
echo "</div>";

logTest('Suspension System', true, "Suspension logs tracking active");

// =====================================================
// FINAL SUMMARY
// =====================================================
echo "<hr class='my-4'>";
echo "<h3 class='text-center mb-4'>";
if ($overall_success) {
    echo "<i class='bi bi-check-circle-fill text-success'></i> All Systems Operational!";
} else {
    echo "<i class='bi bi-exclamation-triangle-fill text-warning'></i> Some Issues Detected";
}
echo "</h3>";

// Summary cards
echo "<div class='row'>";
$pass_count = 0;
$fail_count = 0;
foreach ($test_log as $log) {
    if ($log['success']) $pass_count++;
    else $fail_count++;
}

echo "<div class='col-md-4 text-center'>";
echo "<h2 class='text-primary'>" . count($test_log) . "</h2>";
echo "<p class='text-muted'>Total Tests</p>";
echo "</div>";

echo "<div class='col-md-4 text-center'>";
echo "<h2 class='text-success'>{$pass_count}</h2>";
echo "<p class='text-muted'>Passed</p>";
echo "</div>";

echo "<div class='col-md-4 text-center'>";
echo "<h2 class='text-danger'>{$fail_count}</h2>";
echo "<p class='text-muted'>Failed</p>";
echo "</div>";
echo "</div>";

// Recommendations
echo "<div class='card mt-4'>";
echo "<div class='card-body'>";
echo "<h5><i class='bi bi-lightbulb'></i> Recommendations</h5>";
echo "<ul>";

if ($student_stats['not_synced'] > 0) {
    echo "<li><strong>Action Required:</strong> {$student_stats['not_synced']} students not synced to Moodle. Run the sync bot or trigger manual sync.</li>";
}

if ($mapping_stats['total_mappings'] == 0) {
    echo "<li><strong>Setup Needed:</strong> Configure subject-course mappings in Admin → Moodle Mapping</li>";
}

if ($log_stats['total_logs'] == 0) {
    echo "<li><strong>Info:</strong> No sync history yet. This is normal for new installations.</li>";
}

if ($overall_success && $student_stats['synced'] > 0) {
    echo "<li class='text-success'><strong>✓ System Ready:</strong> Moodle integration is working correctly!</li>";
}

echo "</ul>";
echo "</div>";
echo "</div>";

// Admin Actions
echo "<div class='card mt-4'>";
echo "<div class='card-body'>";
echo "<h5><i class='bi bi-tools'></i> Available Admin Actions</h5>";
echo "<div class='d-grid gap-2'>";
echo "<a href='admin/settings/manage_student_payments.php' class='btn btn-primary'>";
echo "<i class='bi bi-people'></i> Manage Student Payments (Suspend/Unsuspend)</a>";
echo "<a href='admin/moodle/configure_mapping.php' class='btn btn-outline-primary'>";
echo "<i class='bi bi-diagram-3'></i> Configure Subject-Course Mappings</a>";
echo "<a href='test_moodle_functions.php?key=test123' class='btn btn-outline-secondary'>";
echo "<i class='bi bi-clipboard-check'></i> View Detailed API Tests</a>";
echo "</div>";
echo "</div>";
echo "</div>";

// Delete warning
echo "<div class='alert alert-danger mt-4'>";
echo "<i class='bi bi-shield-exclamation'></i> <strong>Security Warning:</strong> ";
echo "Delete this file after testing: <code>test_moodle_complete_workflow.php</code>";
echo "</div>";

echo "</div></div>";
echo "<script src='https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js'></script>";
echo "</body></html>";
?>

