<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';
// Ensure session exists for CSRF token in CLI
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}

// Establish base URL (allow override via env)
$base = getenv('TEST_BASE_URL') ?: (defined('PUBLIC_BASE_URL') ? PUBLIC_BASE_URL : null);

// Helper function for HTTP requests
function httpJson($url, $headers = [], $method = 'GET', $body = null) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    // keep and send cookies for CSRF/session continuity
    $cookie = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'momo_test_cookie.txt';
    curl_setopt($ch, CURLOPT_COOKIEJAR, $cookie);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $cookie);
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($body !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
        }
    }
    $resp = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err = curl_error($ch);
    curl_close($ch);
    return [$code, $resp, $err];
}

echo "=== MTN MoMo Payment System Production Test ===\n\n";

// 1) Test access token retrieval
echo "1. Testing access token retrieval...\n";
list($code, $resp, $err) = httpJson(MOMO_COLLECTION_BASE_URL . '/token/', [
    'Authorization: Basic ' . base64_encode(MOMO_API_USER_ID . ':' . MOMO_API_KEY),
    'Ocp-Apim-Subscription-Key: ' . MOMO_COLLECTION_SUBSCRIPTION_KEY
], 'POST');

$data = $code === 200 ? json_decode($resp, true) : null;
$hasToken = isset($data['access_token']) ? 'yes' : 'no';
echo "   TOKEN: code=$code hasToken=$hasToken\n";

if ($err) {
    echo "   CURL Error: $err\n";
}

if ($code !== 200) {
    echo "   Response: $resp\n";
}

// 2) Test payment request (dry-run with test data)
echo "\n2. Testing payment request...\n";

// fetch CSRF token via HTTP to bind session cookie
$csrfUrl = rtrim($base, '/') . '/tools/csrf.php';
list($csrfCode, $csrfResp, $csrfErr) = httpJson($csrfUrl);
$csrfData = $csrfCode === 200 ? json_decode($csrfResp, true) : null;
$csrfToken = $csrfData['csrf_token'] ?? generateCSRFToken();

// Get a test student ID
try {
    $pdo = getDB();
    $stmt = $pdo->query("SELECT id FROM students LIMIT 1");
    $student = $stmt->fetch();
    $testStudentId = $student ? $student['id'] : 1;
} catch (Exception $e) {
    $testStudentId = 1;
}

$base = getenv('TEST_BASE_URL') ?: PUBLIC_BASE_URL;
$testUrl = rtrim($base, '/') . '/payments/mtn/request.php';
$post = [
    'csrf_token' => $csrfToken,
    'student_id' => $testStudentId,
    'amount' => '1.00',
    'phone' => '26870000000', // Test phone number
    'description' => 'Test payment for production verification'
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $testUrl);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 20);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
// reuse cookie jar for CSRF session continuity
$cookie = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'momo_test_cookie.txt';
curl_setopt($ch, CURLOPT_COOKIEJAR, $cookie);
curl_setopt($ch, CURLOPT_COOKIEFILE, $cookie);
$resp2 = curl_exec($ch);
$code2 = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$err2 = curl_error($ch);
curl_close($ch);

echo "   REQUEST: code=$code2\n";
if ($err2) {
    echo "   CURL Error: $err2\n";
}

$r = json_decode($resp2, true);
if ($r) {
    echo "   Response: " . json_encode($r, JSON_PRETTY_PRINT) . "\n";
} else {
    echo "   Raw Response: $resp2\n";
}

// 3) Test status check if payment was created
if (!empty($r['reference_id'])) {
    echo "\n3. Testing payment status check...\n";
    
    $statusUrl = rtrim($base, '/') . '/payments/mtn/status.php';
    $post2 = [
        'csrf_token' => generateCSRFToken(),
        'reference_id' => $r['reference_id']
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $statusUrl);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $post2);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    $resp3 = curl_exec($ch);
    $code3 = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err3 = curl_error($ch);
    curl_close($ch);
    
    echo "   STATUS: code=$code3\n";
    if ($err3) {
        echo "   CURL Error: $err3\n";
    }
    
    $r3 = json_decode($resp3, true);
    if ($r3) {
        echo "   Response: " . json_encode($r3, JSON_PRETTY_PRINT) . "\n";
    } else {
        echo "   Raw Response: $resp3\n";
    }
} else {
    echo "\n3. Skipping status test - no reference_id received\n";
}

// 4) Test database schema
echo "\n4. Testing database schema...\n";
try {
    $pdo = getDB();
    
    // Check if payments table has required columns
    $stmt = $pdo->query("DESCRIBE payments");
    $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $requiredColumns = ['reference_id', 'external_id', 'amount', 'currency', 'msisdn', 'status', 'momo_status', 'raw_response', 'error_message'];
    $missingColumns = [];
    
    foreach ($requiredColumns as $col) {
        if (!in_array($col, $columns)) {
            $missingColumns[] = $col;
        }
    }
    
    if (empty($missingColumns)) {
        echo "   Database schema: All required columns present\n";
    } else {
        echo "   Database schema: Missing columns - " . implode(', ', $missingColumns) . "\n";
    }
    
    // Test insert capability
    $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM payments");
    $stmt->execute();
    $count = $stmt->fetch()['count'];
    echo "   Current payments in database: $count\n";
    
} catch (Exception $e) {
    echo "   Database error: " . $e->getMessage() . "\n";
}

// 5) Test configuration
echo "\n5. Testing configuration...\n";
$configTests = [
    'MOMO_COLLECTION_BASE_URL' => defined('MOMO_COLLECTION_BASE_URL') ? MOMO_COLLECTION_BASE_URL : 'NOT SET',
    'MOMO_API_USER_ID' => defined('MOMO_API_USER_ID') ? (MOMO_API_USER_ID ? 'SET' : 'EMPTY') : 'NOT SET',
    'MOMO_API_KEY' => defined('MOMO_API_KEY') ? (MOMO_API_KEY ? 'SET' : 'EMPTY') : 'NOT SET',
    'MOMO_COLLECTION_SUBSCRIPTION_KEY' => defined('MOMO_COLLECTION_SUBSCRIPTION_KEY') ? (MOMO_COLLECTION_SUBSCRIPTION_KEY ? 'SET' : 'EMPTY') : 'NOT SET',
    'MOMO_CURRENCY' => defined('MOMO_CURRENCY') ? MOMO_CURRENCY : 'NOT SET',
    'MOMO_ENV' => defined('MOMO_ENV') ? MOMO_ENV : 'NOT SET',
    'PUBLIC_BASE_URL' => defined('PUBLIC_BASE_URL') ? PUBLIC_BASE_URL : 'NOT SET'
];

foreach ($configTests as $key => $value) {
    echo "   $key: $value\n";
}

echo "\n=== Test Summary ===\n";
echo "Token retrieval: " . ($hasToken === 'yes' ? 'PASS' : 'FAIL') . "\n";
echo "Payment request: " . ($code2 === 202 || (isset($r['success']) && $r['success']) ? 'PASS' : 'FAIL') . "\n";
echo "Status check: " . (isset($r3) && isset($r3['success']) ? 'PASS' : 'SKIP') . "\n";
echo "Database schema: " . (empty($missingColumns) ? 'PASS' : 'FAIL') . "\n";

if ($hasToken === 'yes' && ($code2 === 202 || (isset($r['success']) && $r['success']))) {
    echo "\n✅ Payment system is ready for production!\n";
} else {
    echo "\n❌ Payment system needs configuration fixes before production.\n";
}
?>
