<?php
/**
 * Payment Confirmation Webhook Handler
 * Processes payment confirmations from MTN MoMo and card gateways
 * Automatically unsuspends accounts upon successful payment
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/payment_reminder_manager.php';

// Logging function
function logWebhook($message, $data = null) {
    $logFile = __DIR__ . '/logs/payment_webhook.log';
    $timestamp = date('Y-m-d H:i:s');
    $entry = "[{$timestamp}] {$message}";
    if ($data) {
        $entry .= "\n" . json_encode($data, JSON_PRETTY_PRINT);
    }
    $entry .= "\n---\n";
    file_put_contents($logFile, $entry, FILE_APPEND | LOCK_EX);
}

logWebhook("Webhook received", [
    'method' => $_SERVER['REQUEST_METHOD'],
    'remote_addr' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
]);

try {
    $pdo = getDB();
    
    // Get webhook data
    $input = file_get_contents('php://input');
    $headers = getallheaders();
    
    logWebhook("Raw input received", [
        'input' => $input,
        'headers' => $headers
    ]);
    
    // Parse JSON
    $data = json_decode($input, true);
    
    if (!$data) {
        logWebhook("Invalid JSON received");
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON']);
        exit;
    }
    
    // Determine payment provider
    $provider = $data['provider'] ?? ($_GET['provider'] ?? 'unknown');
    
    logWebhook("Processing payment from provider: {$provider}");
    
    // Process based on provider
    if ($provider === 'mtn_momo' || isset($data['financialTransactionId'])) {
        // MTN Mobile Money
        $reference_id = $data['referenceId'] ?? null;
        $status = $data['status'] ?? null;
        $amount = $data['amount'] ?? null;
        $external_id = $data['financialTransactionId'] ?? null;
        
        logWebhook("MTN MoMo payment", [
            'reference' => $reference_id,
            'status' => $status,
            'amount' => $amount
        ]);
        
        if (!$reference_id) {
            throw new Exception("Missing reference ID");
        }
        
        // Find transaction by reference
        $stmt = $pdo->prepare("
            SELECT * FROM payment_transactions 
            WHERE transaction_reference = ? OR external_reference = ?
            LIMIT 1
        ");
        $stmt->execute([$reference_id, $reference_id]);
        $transaction = $stmt->fetch();
        
        if (!$transaction) {
            logWebhook("Transaction not found: {$reference_id}");
            http_response_code(404);
            echo json_encode(['error' => 'Transaction not found']);
            exit;
        }
        
        // Update transaction status
        $new_status = ($status === 'SUCCESSFUL') ? 'completed' : 'failed';
        
        $stmt = $pdo->prepare("
            UPDATE payment_transactions 
            SET status = ?, 
                external_reference = ?,
                paid_at = ?,
                raw_response = ?
            WHERE id = ?
        ");
        $stmt->execute([
            $new_status,
            $external_id,
            ($status === 'SUCCESSFUL') ? date('Y-m-d H:i:s') : null,
            $input,
            $transaction['id']
        ]);
        
        logWebhook("Transaction updated", [
            'transaction_id' => $transaction['id'],
            'new_status' => $new_status
        ]);
        
        // If successful, process payment
        if ($status === 'SUCCESSFUL') {
            $manager = new PaymentReminderManager();
            $result = $manager->processPaymentSuccess($transaction['id']);
            
            if ($result['success']) {
                logWebhook("Payment processed successfully", [
                    'transaction_id' => $transaction['id'],
                    'student_id' => $transaction['student_id']
                ]);
            } else {
                logWebhook("Payment processing failed", [
                    'error' => $result['error']
                ]);
            }
            
            // Update reminder if linked
            if ($transaction['reminder_id']) {
                $pdo->prepare("
                    UPDATE payment_reminders 
                    SET payment_completed = 1, payment_completed_at = NOW()
                    WHERE id = ?
                ")->execute([$transaction['reminder_id']]);
            }
        }
        
    } elseif ($provider === 'card' || $provider === 'paygate') {
        // Card Payment Gateway
        $reference = $data['REFERENCE'] ?? $data['reference'] ?? null;
        $transaction_status = $data['TRANSACTION_STATUS'] ?? $data['status'] ?? null;
        
        logWebhook("Card payment", [
            'reference' => $reference,
            'status' => $transaction_status
        ]);
        
        // Process card payment (similar logic as above)
        // TODO: Implement based on your specific card gateway
        
    } else {
        logWebhook("Unknown payment provider: {$provider}");
    }
    
    // Respond success
    http_response_code(200);
    echo json_encode([
        'status' => 'success',
        'message' => 'Webhook processed'
    ]);
    
} catch (Exception $e) {
    logWebhook("ERROR: " . $e->getMessage(), [
        'trace' => $e->getTraceAsString()
    ]);
    
    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}

