<?php
/**
 * Admin - Storage Usage Dashboard
 * 
 * Shows storage usage for current tenant
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/tenant_directory_manager.php';
requireAdminLogin();

// Get tenant ID from session
$tenant_id = getCurrentSchoolReference();

if (!$tenant_id) {
    die("Error: No school selected. Please select a school first.");
}

// Database connection
$conn = getDB();

// Initialize directory manager
$dirManager = new TenantDirectoryManager($conn);

// Calculate storage usage
$usage = $dirManager->calculateTenantStorageUsage($tenant_id);

// Get file statistics
$stmt = $conn->prepare("
    SELECT 
        file_category,
        COUNT(*) as file_count,
        SUM(file_size) as total_size
    FROM tenant_files
    WHERE tenant_id = ? AND is_deleted = 0
    GROUP BY file_category
");
$stmt->execute([$tenant_id]);
$categoryStats = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get largest files
$stmt = $conn->prepare("
    SELECT 
        original_filename,
        file_category,
        file_size,
        upload_date,
        id,
        file_path
    FROM tenant_files
    WHERE tenant_id = ? AND is_deleted = 0
    ORDER BY file_size DESC
    LIMIT 20
");
$stmt->execute([$tenant_id]);
$largestFiles = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get recent uploads
$stmt = $conn->prepare("
    SELECT 
        original_filename,
        file_category,
        file_size,
        upload_date
    FROM tenant_files
    WHERE tenant_id = ? AND is_deleted = 0
    ORDER BY upload_date DESC
    LIMIT 10
");
$stmt->execute([$tenant_id]);
$recentFiles = $stmt->fetchAll(PDO::FETCH_ASSOC);

// formatBytes() function is available from includes/security_headers.php

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Storage Usage - <?= htmlspecialchars($tenant_id) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        .storage-meter {
            height: 30px;
            border-radius: 15px;
            background: #e9ecef;
            overflow: hidden;
            position: relative;
        }
        .storage-fill {
            height: 100%;
            transition: width 1s ease;
        }
        .category-card {
            border-left: 4px solid;
            height: 100%;
        }
    </style>
</head>
<body class="bg-light">
    
    <!-- Navbar -->
    <nav class="navbar navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="../dashboard.php">
                <i class="bi bi-hdd"></i> Storage Usage
            </a>
            <a href="../dashboard.php" class="btn btn-outline-light btn-sm">
                <i class="bi bi-arrow-left"></i> Back
            </a>
        </div>
    </nav>

    <div class="container-fluid py-4">
        
        <!-- Storage Overview -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h4 class="mb-3">Storage Overview</h4>
                        
                        <div class="row">
                            <div class="col-md-3">
                                <h6 class="text-muted">Used</h6>
                                <h3 class="text-primary"><?= $usage['total_formatted'] ?></h3>
                            </div>
                            <div class="col-md-3">
                                <h6 class="text-muted">Quota</h6>
                                <h3><?= $usage['quota_formatted'] ?></h3>
                            </div>
                            <div class="col-md-3">
                                <h6 class="text-muted">Available</h6>
                                <h3 class="text-success"><?= $usage['available_formatted'] ?></h3>
                            </div>
                            <div class="col-md-3">
                                <h6 class="text-muted">Usage</h6>
                                <h3 class="<?= $usage['percentage'] > 80 ? 'text-danger' : 'text-success' ?>">
                                    <?= $usage['percentage'] ?>%
                                </h3>
                            </div>
                        </div>
                        
                        <div class="storage-meter mt-3">
                            <?php 
                            $color = $usage['percentage'] > 90 ? '#dc3545' : 
                                    ($usage['percentage'] > 70 ? '#ffc107' : '#28a745');
                            ?>
                            <div class="storage-fill" 
                                 style="width: <?= $usage['percentage'] ?>%; background: <?= $color ?>;">
                            </div>
                        </div>
                        
                        <?php if ($usage['percentage'] > 80): ?>
                        <div class="alert alert-warning mt-3 mb-0">
                            <i class="bi bi-exclamation-triangle"></i>
                            <strong>Storage Warning:</strong> You are using over 80% of your storage quota.
                            Consider deleting old files or contact support to increase your quota.
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Storage by Category -->
        <div class="row mb-4">
            <?php 
            $categoryColors = [
                'document' => 'primary',
                'photo' => 'success',
                'receipt' => 'info',
                'report' => 'warning',
                'backup' => 'secondary'
            ];
            
            foreach ($categoryStats as $cat): 
                $color = $categoryColors[$cat['file_category']] ?? 'secondary';
            ?>
            <div class="col-md-3 mb-3">
                <div class="card category-card border-<?= $color ?>">
                    <div class="card-body">
                        <h6 class="text-muted text-uppercase"><?= $cat['file_category'] ?>s</h6>
                        <h4><?= formatBytes($cat['total_size']) ?></h4>
                        <small class="text-muted"><?= $cat['file_count'] ?> files</small>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>

        <!-- Charts Row -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Storage by Category</h5>
                    </div>
                    <div class="card-body">
                        <canvas id="categoryChart"></canvas>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">File Count by Category</h5>
                    </div>
                    <div class="card-body">
                        <canvas id="fileCountChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Largest Files -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-file-earmark-text"></i> Largest Files</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-sm table-hover">
                                <thead>
                                    <tr>
                                        <th>Filename</th>
                                        <th>Category</th>
                                        <th>Size</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($largestFiles as $file): ?>
                                    <tr>
                                        <td>
                                            <small><?= htmlspecialchars($file['original_filename']) ?></small>
                                        </td>
                                        <td><span class="badge bg-secondary"><?= $file['file_category'] ?></span></td>
                                        <td><?= formatBytes($file['file_size']) ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-danger" 
                                                    onclick="deleteFile(<?= $file['id'] ?>)">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bi bi-clock-history"></i> Recent Uploads</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-sm table-hover">
                                <thead>
                                    <tr>
                                        <th>Filename</th>
                                        <th>Category</th>
                                        <th>Size</th>
                                        <th>Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recentFiles as $file): ?>
                                    <tr>
                                        <td><small><?= htmlspecialchars($file['original_filename']) ?></small></td>
                                        <td><span class="badge bg-secondary"><?= $file['file_category'] ?></span></td>
                                        <td><?= formatBytes($file['file_size']) ?></td>
                                        <td><small><?= date('M d, Y', strtotime($file['upload_date'])) ?></small></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

    </div>

    <script>
    // Category storage chart (Pie)
    const categoryData = <?= json_encode($categoryStats) ?>;
    const categoryCtx = document.getElementById('categoryChart').getContext('2d');
    
    new Chart(categoryCtx, {
        type: 'doughnut',
        data: {
            labels: categoryData.map(c => c.file_category),
            datasets: [{
                data: categoryData.map(c => Math.round(c.total_size / 1024 / 1024)),
                backgroundColor: [
                    'rgba(54, 162, 235, 0.8)',
                    'rgba(75, 192, 192, 0.8)',
                    'rgba(255, 206, 86, 0.8)',
                    'rgba(255, 99, 132, 0.8)',
                    'rgba(153, 102, 255, 0.8)'
                ]
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom'
                },
                title: {
                    display: false
                }
            }
        }
    });

    // File count chart (Bar)
    const fileCountCtx = document.getElementById('fileCountChart').getContext('2d');
    
    new Chart(fileCountCtx, {
        type: 'bar',
        data: {
            labels: categoryData.map(c => c.file_category),
            datasets: [{
                label: 'Number of Files',
                data: categoryData.map(c => c.file_count),
                backgroundColor: 'rgba(54, 162, 235, 0.5)',
                borderColor: 'rgba(54, 162, 235, 1)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });

    // Delete file function
    async function deleteFile(fileId) {
        if (!confirm('Are you sure you want to delete this file? This action cannot be undone.')) {
            return;
        }
        
        try {
            const response = await fetch('../../api/storage/file.php?id=' + fileId, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json'
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                alert('✓ File deleted successfully');
                location.reload();
            } else {
                alert('✗ Error: ' + result.error);
            }
        } catch (error) {
            alert('✗ Error deleting file: ' + error.message);
        }
    }
    
    // Auto-refresh every 30 seconds
    setTimeout(() => {
        location.reload();
    }, 30000);
    </script>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

