<?php
/**
 * Super Admin Dashboard
 * Manage all schools and view cross-school analytics
 */

require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();

// Check if user is super admin
if (!isSuperAdmin()) {
    header('Location: ../admin/dashboard.php');
    exit;
}

// Handle quick school selection BEFORE loading page
if (isset($_GET['select_school'])) {
    $ref = $_GET['select_school'];
    if (setCurrentSchool($ref)) {
        header('Location: ../admin/dashboard.php');
        exit;
    } else {
        $selectionError = "Failed to select school: Invalid or inactive school.";
    }
}

$pdo = getDB();

// Get all schools
$schools = getAllSchools();

// Get system-wide statistics
$stmt = $pdo->query("
    SELECT 
        COUNT(*) as total_schools
    FROM academy_references 
    WHERE is_active = 1
");
$total_schools = $stmt->fetchColumn();

$stmt = $pdo->query("
    SELECT COUNT(*) as total_students
    FROM students
");
$total_students = $stmt->fetchColumn();

$stmt = $pdo->query("
    SELECT 
        COUNT(*) as total_payments,
        SUM(amount) as total_revenue,
        SUM(platform_fee) as total_platform_fees
    FROM payments 
    WHERE status = 'completed'
");
$payment_stats = $stmt->fetch();

// Get recent activity
$stmt = $pdo->query("
    SELECT 
        s.full_name,
        s.academy_reference,
        s.created_at
    FROM students s
    ORDER BY s.created_at DESC
    LIMIT 10
");
$recent_students = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Super Admin Dashboard - Skolo-Kine</title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .stats-card {
            transition: transform 0.3s;
        }
        .stats-card:hover {
            transform: translateY(-5px);
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="dashboard.php">
                <i class="bi bi-shield-check me-2"></i>
                <strong>Super Admin</strong>
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="schools/list.php">
                    <i class="bi bi-building me-1"></i>Schools
                </a>
                <a class="nav-link" href="storage/overview.php">
                    <i class="bi bi-hdd-stack me-1"></i>Storage
                </a>
                <a class="nav-link" href="../admin/logout.php">
                    <i class="bi bi-box-arrow-right me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col-12">
                <h1 class="h3">
                    <i class="bi bi-speedometer2 me-2"></i>
                    Super Admin Dashboard
                </h1>
                <p class="text-muted">System-wide overview and management</p>
            </div>
        </div>

        <?php if (isset($selectionError)): ?>
            <div class="row mb-4">
                <div class="col-12">
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-exclamation-triangle me-2"></i>
                        <strong><?= htmlspecialchars($selectionError) ?></strong>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="row g-4 mb-4">
            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Active Schools</h6>
                                <h2 class="mb-0"><?= $total_schools ?></h2>
                            </div>
                            <div class="bg-primary bg-opacity-10 p-3 rounded">
                                <i class="bi bi-building fs-2 text-primary"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Total Students</h6>
                                <h2 class="mb-0"><?= number_format($total_students) ?></h2>
                            </div>
                            <div class="bg-success bg-opacity-10 p-3 rounded">
                                <i class="bi bi-people fs-2 text-success"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Total Revenue</h6>
                                <h2 class="mb-0">E<?= number_format($payment_stats['total_revenue'] ?? 0, 2) ?></h2>
                            </div>
                            <div class="bg-info bg-opacity-10 p-3 rounded">
                                <i class="bi bi-currency-exchange fs-2 text-info"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-3">
                <div class="card stats-card shadow-sm h-100">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="text-muted mb-1">Platform Fees</h6>
                                <h2 class="mb-0">E<?= number_format($payment_stats['total_platform_fees'] ?? 0, 2) ?></h2>
                            </div>
                            <div class="bg-warning bg-opacity-10 p-3 rounded">
                                <i class="bi bi-wallet2 fs-2 text-warning"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Schools Overview -->
        <div class="row">
            <div class="col-12 mb-4">
                <div class="card shadow-sm">
                    <div class="card-header bg-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">
                            <i class="bi bi-building me-2"></i>
                            Schools Overview
                        </h5>
                        <a href="schools/create.php" class="btn btn-primary btn-sm">
                            <i class="bi bi-plus-circle me-1"></i>
                            Add New School
                        </a>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>School Name</th>
                                        <th>Reference Code</th>
                                        <th>Students</th>
                                        <th>Revenue</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($schools as $school): 
                                        $stats = getSchoolStats($school['reference_code']);
                                    ?>
                                    <tr>
                                        <td>
                                            <strong><?= htmlspecialchars($school['academy_name']) ?></strong>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary"><?= htmlspecialchars($school['reference_code']) ?></span>
                                        </td>
                                        <td><?= number_format($stats['total_students']) ?></td>
                                        <td>E<?= number_format($stats['total_revenue'], 2) ?></td>
                                        <td>
                                            <?php if ($school['is_active']): ?>
                                                <span class="badge bg-success">Active</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">Suspended</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="?select_school=<?= urlencode($school['reference_code']) ?>" 
                                                   class="btn btn-outline-primary" 
                                                   title="Access School">
                                                    <i class="bi bi-box-arrow-in-right"></i>
                                                </a>
                                                <a href="schools/edit.php?id=<?= $school['id'] ?>" 
                                                   class="btn btn-outline-secondary" 
                                                   title="Edit">
                                                    <i class="bi bi-pencil"></i>
                                                </a>
                                                <a href="schools/view.php?id=<?= $school['id'] ?>" 
                                                   class="btn btn-outline-info" 
                                                   title="View Details">
                                                    <i class="bi bi-eye"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Activity -->
        <div class="row">
            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-clock-history me-2"></i>
                            Recent Student Registrations
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="list-group list-group-flush">
                            <?php foreach ($recent_students as $s): ?>
                            <div class="list-group-item px-0">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <strong><?= htmlspecialchars($s['full_name']) ?></strong>
                                        <br>
                                        <small class="text-muted">
                                            <span class="badge bg-info"><?= htmlspecialchars($s['academy_reference']) ?></span>
                                        </small>
                                    </div>
                                    <div>
                                        <small class="text-muted"><?= date('M d, Y', strtotime($s['created_at'])) ?></small>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-6">
                <div class="card shadow-sm">
                    <div class="card-header bg-white">
                        <h5 class="mb-0">
                            <i class="bi bi-gear me-2"></i>
                            Quick Actions
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="d-grid gap-2">
                            <a href="schools/create.php" class="btn btn-primary btn-lg">
                                <i class="bi bi-plus-circle me-2"></i>
                                Create New School
                            </a>
                            <a href="schools/list.php" class="btn btn-outline-secondary btn-lg">
                                <i class="bi bi-building me-2"></i>
                                Manage All Schools
                            </a>
                            <a href="admins/list.php" class="btn btn-outline-primary btn-lg">
                                <i class="bi bi-people me-2"></i>
                                Manage School Admins
                            </a>
                            <a href="reports/platform_fees.php" class="btn btn-outline-info btn-lg">
                                <i class="bi bi-graph-up me-2"></i>
                                View Platform Earnings
                            </a>
                            <a href="reports/system_health.php" class="btn btn-outline-success btn-lg">
                                <i class="bi bi-heart-pulse me-2"></i>
                                System Health
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

